/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.swing.*;
import javax.swing.tree.*;
import restringer.ess.*;
import restringer.ess.papyrus.*;
import restringer.gui.FilterTreeModel.Node;

/**
 * A JTree that supports filtering.
 *
 * @author Mark Fairchild
 * @version 2016/06/23
 */
final public class FilterTree extends JTree {

    /**
     * Creates a new <code>FilterTree</code>.
     */
    public FilterTree() {
        super(new FilterTreeModel());
        this.deleteHandler = null;
        this.deleteFormsHandler = null;
        this.deleteInstancesHandler = null;
        this.MI_DELETE = new JMenuItem(RES.getString("TREE DELETE MENU"), KeyEvent.VK_D);
        this.MI_FILTER = new JMenuItem(RES.getString("TREE PLUGIN FILTER"), KeyEvent.VK_F);
        this.MI_DELETE_FORMS = new JMenuItem(RES.getString("TREE DELETE FORMS"), KeyEvent.VK_C);
        this.MI_DELETE_INSTANCES = new JMenuItem(RES.getString("TREE DELETE INSTANCES"), KeyEvent.VK_S);
        this.MI_ZERO_THREAD = new JMenuItem(RES.getString("TREE ZERO THREAD"), KeyEvent.VK_Z);
        this.MI_FIND_OWNER = new JMenuItem(RES.getString("TREE GOTO OWNER"), KeyEvent.VK_F);
        this.MI_CLEANSE_FLST = new JMenuItem(RES.getString("TREE_CLEANSE_FLST"), KeyEvent.VK_C);
        this.TREE_POPUP_MENU = new JPopupMenu();
        this.PLUGIN_POPUP_MENU = new JPopupMenu();
        this.initComponents();
    }

    /**
     * Initialize the swing and AWT components.
     */
    private void initComponents() {
        this.setLargeModel(true);
        this.setRootVisible(true);
        this.setShowsRootHandles(true);

        this.TREE_POPUP_MENU.add(this.MI_DELETE);
        this.TREE_POPUP_MENU.add(this.MI_ZERO_THREAD);
        this.TREE_POPUP_MENU.add(this.MI_FIND_OWNER);
        this.TREE_POPUP_MENU.add(this.MI_CLEANSE_FLST);
        this.PLUGIN_POPUP_MENU.add(this.MI_FILTER);
        this.PLUGIN_POPUP_MENU.add(this.MI_DELETE_FORMS);
        this.PLUGIN_POPUP_MENU.add(this.MI_DELETE_INSTANCES);

        this.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), "deleteSelected");
        this.getActionMap().put("deleteSelected", new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                deleteNode();
            }
        });

        this.MI_DELETE.addActionListener(e -> {
            deleteNode();
        });

        this.MI_FILTER.addActionListener(e -> {
            Plugin plugin = (Plugin) ((Node) getSelectionPath().getLastPathComponent()).getElement();
            filter(plugin);
        });

        this.MI_DELETE_FORMS.addActionListener(e -> {
            Plugin plugin = (Plugin) ((Node) getSelectionPath().getLastPathComponent()).getElement();
            if (null != this.deleteFormsHandler) {
                this.deleteFormsHandler.accept(plugin);
            }
        });

        this.MI_DELETE_INSTANCES.addActionListener(e -> {
            Plugin plugin = (Plugin) ((Node) getSelectionPath().getLastPathComponent()).getElement();
            if (null != this.deleteInstancesHandler) {
                this.deleteInstancesHandler.accept(plugin);
            }
        });

        this.MI_ZERO_THREAD.addActionListener(e -> {
            if (null != this.zeroThreadHandler) {
                final TreePath[] PATHS = getSelectionPaths();
                if (null == PATHS || PATHS.length == 0) {
                    return;
                }

                final Map<Element, Node> ELEMENTS = getModel().parsePaths(PATHS);
                final List<ActiveScript> THREADS = ELEMENTS.keySet()
                        .stream()
                        .filter(ESS.THREAD)
                        .map(v -> (ActiveScript) v)
                        .collect(Collectors.toList());
                this.zeroThreadHandler.accept(THREADS);
            }
        });

        this.MI_FIND_OWNER.addActionListener(e -> {
            Element element = ((Node) getSelectionPath().getLastPathComponent()).getElement();
            if (null != this.findHandler) {
                if (element instanceof ActiveScript) {
                    ActiveScript script = (ActiveScript) element;
                    if (null != script.getInstance()) {
                        this.findHandler.accept(script.getInstance());
                    }
                } else if (element instanceof StackFrame) {
                    StackFrame frame = (StackFrame) element;
                    Variable owner = frame.getOwner();
                    if (null != owner && owner instanceof Variable.Ref) {
                        Variable.Ref ref = (Variable.Ref) frame.getOwner();
                        this.findHandler.accept(ref.getReferent());
                    }
                } else if (element instanceof ArrayInfo) {
                    ArrayInfo array = (ArrayInfo) element;
                    if (null != array.getHolder()) {
                        this.findHandler.accept(array.getHolder());
                    }
                    
                }
            }

        });

        this.MI_CLEANSE_FLST.addActionListener(e -> {
            ChangeForm form = (ChangeForm) ((Node) getSelectionPath().getLastPathComponent()).getElement();
            ChangeFormFLST flst = (ChangeFormFLST) form.getData();
            if (null != this.cleanFLSTHandler) {
                this.cleanFLSTHandler.accept(flst);
            }
        });

        this.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseReleased(MouseEvent evt) {
                if (evt.isPopupTrigger()) {
                    int x = evt.getPoint().x;
                    int y = evt.getPoint().y;

                    TreePath path = getClosestPathForLocation(x, y);
                    TreePath[] paths = getSelectionPaths();

                    if (!Arrays.asList(paths).contains(path)) {
                        setSelectionPath(path);
                        paths = getSelectionPaths();
                    }

                    final Map<Element, Node> ELEMENTS = getModel().parsePaths(paths);

                    if (ELEMENTS.size() == 1) {
                        final Element ELEMENT = ELEMENTS.keySet().iterator().next();

                        if (ELEMENT instanceof Plugin) {
                            PLUGIN_POPUP_MENU.show(evt.getComponent(), evt.getX(), evt.getY());

                        } else if (ESS.DELETABLE.test(ELEMENT) || ESS.THREAD.test(ELEMENT) || ESS.OWNABLE.test(ELEMENT)) {
                            MI_DELETE.setText(RES.getString("TREE DELETE ELEMENT"));
                            MI_DELETE.setVisible(ESS.DELETABLE.test(ELEMENT));
                            MI_ZERO_THREAD.setVisible(ESS.THREAD.test(ELEMENT));
                            MI_FIND_OWNER.setVisible(ESS.OWNABLE.test(ELEMENT));
                            MI_CLEANSE_FLST.setVisible(ELEMENT instanceof ChangeForm && ((ChangeForm) ELEMENT).getData() instanceof ChangeFormFLST);
                            TREE_POPUP_MENU.show(evt.getComponent(), evt.getX(), evt.getY());
                        }

                    } else if (ELEMENTS.size() > 1) {
                        
                        MI_FIND_OWNER.setVisible(false);

                        int deletable = (int) ELEMENTS.keySet().stream().filter(ESS.DELETABLE).count();
                        MI_DELETE.setEnabled(deletable > 0);
                        MI_DELETE.setText(String.format(RES.getString("TREE DELETE ELEMENTS"), deletable));

                        int threads = (int) ELEMENTS.keySet().stream().filter(ESS.THREAD).count();
                        MI_ZERO_THREAD.setVisible(threads > 0);

                        TREE_POPUP_MENU.show(evt.getComponent(), evt.getX(), evt.getY());
                    }
                }
            }
        });
    }

    /**
     * Clears the <code>ESS</code>.
     */
    public void clearESS() {
        this.setModel(new FilterTreeModel());
    }

    /**
     * Uses an <code>ESS</code> to create the tree's data model.
     *
     * @param newSave The new save to model.
     * @param filter An optional filter.
     *
     */
    public void setESS(ESS newSave, Predicate<Node> filter) {
        Objects.requireNonNull(newSave);
        final FilterTreeModel MODEL = newSave.createTreeModel();
        if (null != filter) {
            MODEL.filter(filter);
        }

        this.setModel(MODEL);
    }

    /**
     * Searches for the <code>Node</code> that represents a specified
     * <code>Element</code> and returns it.
     *
     * @param element The <code>Element</code> to find.
     * @return The corresponding <code>Node</code> or null if the
     * <code>Element</code> was not found.
     */
    public TreePath findPath(Element element) {
        TreePath path = this.getModel().findPath(element);
        return path;
    }

    /**
     * Filters the tree and its contents by <code>Plugin</code>.
     *
     * @param plugin The <code>Plugin</code> to match against.
     */
    public void filter(Plugin plugin) {
        if (null != this.pluginFilterHandler) {
            this.pluginFilterHandler.accept(plugin);
        }
    }

    /**
     * Sets the delete handler.
     *
     * @param newHandler The new delete handler.
     */
    public void setDeleteHandler(Consumer<Map<Element, Node>> newHandler) {
        this.deleteHandler = newHandler;
    }

    /**
     * Sets the filter plugin handler.
     *
     * @param newHandler The new delete handler.
     */
    public void setFilterPluginsHandler(Consumer<Plugin> newHandler) {
        this.pluginFilterHandler = newHandler;
    }

    /**
     * Sets the delete plugin forms handler.
     *
     * @param newHandler The new delete handler.
     */
    public void setDeleteFormsHandler(Consumer<Plugin> newHandler) {
        this.deleteFormsHandler = newHandler;
    }

    /**
     * Sets the delete plugin instances handler.
     *
     * @param newHandler The new delete handler.
     */
    public void setDeleteInstancesHandler(Consumer<Plugin> newHandler) {
        this.deleteInstancesHandler = newHandler;
    }

    /**
     * Sets the zero active script handler.
     *
     * @param newHandler The new handler.
     */
    public void setZeroThreadHandler(Consumer<List<ActiveScript>> newHandler) {
        this.zeroThreadHandler = newHandler;
    }

    /**
     * Sets the find element handler.
     *
     * @param newHandler The new handler.
     */
    public void setFindHandler(Consumer<Element> newHandler) {
        this.findHandler = newHandler;
    }

    /**
     * Sets the cleanse formlist handler.
     *
     * @param newHandler The new handler.
     */
    public void setCleanseFLSTHandler(Consumer<ChangeFormFLST> newHandler) {
        this.cleanFLSTHandler = newHandler;
    }

    /**
     * Deletes a node by submitting it back to the app.
     */
    private void deleteNode() {
        if (null == this.deleteHandler) {
            return;
        }

        final TreePath[] PATHS = getSelectionPaths();
        if (null == PATHS || PATHS.length == 0) {
            return;
        }

        final Map<Element, Node> ELEMENTS = getModel().parsePaths(PATHS);
        this.deleteHandler.accept(ELEMENTS);
    }

    @Override
    public FilterTreeModel getModel() {
        return (FilterTreeModel) super.getModel();
    }

    final private JMenuItem MI_DELETE;
    final private JMenuItem MI_FILTER;
    final private JMenuItem MI_DELETE_FORMS;
    final private JMenuItem MI_DELETE_INSTANCES;
    final private JMenuItem MI_ZERO_THREAD;
    final private JMenuItem MI_FIND_OWNER;
    final private JMenuItem MI_CLEANSE_FLST;
    final private JPopupMenu TREE_POPUP_MENU;
    final private JPopupMenu PLUGIN_POPUP_MENU;
    private Consumer<Map<Element, Node>> deleteHandler;
    private Consumer<List<ActiveScript>> zeroThreadHandler;
    private Consumer<Plugin> pluginFilterHandler;
    private Consumer<Plugin> deleteFormsHandler;
    private Consumer<Plugin> deleteInstancesHandler;
    private Consumer<Element> findHandler;
    private Consumer<ChangeFormFLST> cleanFLSTHandler;

    private static final ResourceBundle RES = ResourceBundle.getBundle("restringer/gui/General");

}
