/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.util.*;
import java.util.function.Predicate;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import restringer.ess.Element;
import restringer.ess.papyrus.HasID;

/**
 * A <code>TreeModel</code> that supports filtering.
 *
 * @author Mark Fairchild
 * @version 2016/07/03
 */
final public class FilterTreeModel implements TreeModel {

    /**
     *
     */
    public FilterTreeModel() {
        this.LISTENERS = new java.util.LinkedList<>();
        this.root = null;
    }

    /**
     * Creates a new internal <code>Node</code> to wrap the specified element
     *
     * @param element The element that the node will contain.
     * @param children The list of children with which to initialize the
     * <code>Node</code>.
     * @return The new node.
     */
    public Node node(Element element, Collection<? extends Element> children) {
        Objects.requireNonNull(element);
        Objects.requireNonNull(children);
        final Node NODE = new Node(element, false);

        List<Node> childNodes = new ArrayList<>(children.size());
        children.forEach(child -> {
            Node childNode = new Node(child, true);
            childNodes.add(childNode);
        });

        NODE.addAll(childNodes);

        return NODE;
    }

    /**
     * Creates a new root-style <code>Node</code>.
     *
     * @param element The element that the node will contain.
     * @param children The list of children with which to initialize the
     * <code>Node</code>.
     * @return The new node.
     */
    public Node root(Element element, Collection<Node> children) {
        Objects.requireNonNull(element);
        Objects.requireNonNull(children);
        final Node NODE = new Node(element, false);
        NODE.addAll(children);
        return NODE;
    }

    /**
     * Creates a new internal <code>Node</code> that stores an
     * <code>Element</code> list.
     *
     * @param name The name of the node.
     * @param children The list of children with which to initialize the
     * <code>Node</code>.
     * @return The new node.
     */
    public Node elementContainer(String name, Collection<? extends Element> children) {
        Objects.requireNonNull(name);
        Objects.requireNonNull(children);
        final Node NODE = new Node(name);

        List<Node> childNodes = new ArrayList<>(children.size());
        children.forEach(child -> {
            Node childNode = new Node(child, true);
            childNodes.add(childNode);
        });

        NODE.addAll(childNodes);

        return NODE;
    }

    /**
     * Creates a new internal <code>Node</code> that stores a <code>Node</code>
     * list.
     *
     * @param name The name of the node.
     * @param children The list of children with which to initialize the
     * <code>Node</code>.
     * @return The new node.
     */
    public Node nodeContainer(String name, Collection<Node> children) {
        Objects.requireNonNull(name);
        Objects.requireNonNull(children);
        final Node NODE = new Node(name);

        NODE.addAll(children);
        return NODE;
    }

    /**
     * Creates a new internal <code>Node</code> to wrap the specified element
     *
     * @param element The element that the node will contain.
     * @param child A child with which to initialize the <code>Node</code>.
     * @return The new node.
     */
    public Node node(Element element, Element child) {
        Objects.requireNonNull(element);
        final Node NODE = new Node(element, false);

        if (null != child) {
            final Node CHILD = new Node(child, true);
            NODE.addAll(Collections.singleton(CHILD));
        }

        return NODE;
    }

    /**
     * Creates a new leaf <code>Node</code> that stores an <code>Element</code>.
     *
     * @param element The element that the node will contain.
     * @return The new node.
     */
    public Node leaf(Element element) {
        Objects.requireNonNull(element);
        return new Node(element, true);
    }

    /**
     * Transforms a <code>TreePath</code> array into a map of elements and their
     * corresponding nodes.
     *
     * @param paths
     * @return
     */
    public Map<Element, Node> parsePaths(TreePath[] paths) {
        Objects.requireNonNull(paths);
        final Map<Element, Node> ELEMENTS = new LinkedHashMap<>(paths.length);

        for (TreePath path : paths) {
            if (null == path) {
                continue;
            }

            final Node NODE = (Node) path.getLastPathComponent();

            if (NODE.hasElement()) {
                ELEMENTS.put(NODE.getElement(), NODE);
            } else {
                ELEMENTS.putAll(NODE.parsePath());
            }
        }

        return ELEMENTS;
    }

    /**
     * Retrieves the node's element and all the elements of its children.
     *
     * @return
     */
    public List<Element> getElements() {
        if (null == this.root) {
            return new LinkedList<>();
        } else {
            return this.root.getElements();
        }
    }

    /**
     * Removes all filtering.
     *
     */
    public void defilter() {
        if (null == this.root) {
            return;
        }

        this.root.defilter();
        this.fireTreeNodesChanged(new TreeModelEvent(this.root, this.root.getPath()));
    }

    /**
     * Filters the model and its contents.
     *
     * @param filter The filter that determines which nodes to keep.
     */
    synchronized public void filter(Predicate<Node> filter) {
        Objects.requireNonNull(filter);

        if (null == this.root) {
            return;
        }

        this.root.filter(filter);
        this.LISTENERS.forEach(l -> l.treeStructureChanged(new TreeModelEvent(this.root, this.root.getPath())));
    }

    /**
     * After rebuilding the treemodel, paths become invalid. This corrects them.
     *
     * @param path
     * @return
     */
    public TreePath rebuildPath(TreePath path) {
        Objects.requireNonNull(path);
        if (path.getPathCount() < 1) {
            return null;
        }

        TreePath newPath = new TreePath(this.root);
        Node newNode = this.root;

        for (int i = 1; i < path.getPathCount(); i++) {
            Node originalNode = (Node) path.getPathComponent(i);
            Optional<Node> child = newNode.CHILDREN.stream().filter(n -> n.NAME.equals(originalNode.NAME)).findFirst();

            if (!child.isPresent()) {
                if (originalNode.hasElement() && originalNode.getElement() instanceof HasID) {
                    HasID original = (HasID) originalNode.getElement();
                    child = newNode.CHILDREN.stream()
                            .filter(n -> n.hasElement() && n.getElement() instanceof HasID)
                            .filter(n -> ((HasID) n.getElement()).getID() == original.getID())
                            .findAny();
                }
            }

            if (!child.isPresent()) {
                return newPath;
            }

            newNode = child.get();
            newPath = newPath.pathByAddingChild(newNode);
        }

        return newPath;
    }

    /**
     * Searches for the <code>Node</code> that represents a specified
     * <code>Element</code> and returns it.
     *
     * @param element The <code>Element</code> to find.
     * @return The corresponding <code>Node</code> or null if the
     * <code>Element</code> was not found.
     */
    public TreePath findPath(Element element) {
        if (null == this.root) {
            return null;
        }

        TreePath path = this.root.findPath(element);
        if (null != path) {
            return path;
        }

        path = this.root.findPathUnfiltered(element);
        if (null != path) {
            this.root.defilter(path, 0);
            return path;
        }

        return null;
    }

    @Override
    public Node getRoot() {
        return this.root;
    }

    public void setRoot(Node newRoot) {
        this.root = Objects.requireNonNull(newRoot);
    }

    @Override
    public Object getChild(Object parent, int index) {
        assert parent instanceof Node;
        final Node NODE = (Node) parent;
        return NODE.getChildAt(index);
    }

    @Override
    public int getChildCount(Object parent) {
        assert parent instanceof Node;
        final Node NODE = (Node) parent;
        return NODE.getChildCount();
    }

    @Override
    public int getIndexOfChild(Object parent, Object child) {
        assert parent instanceof Node;
        assert child instanceof Node;

        final Node PARENT = (Node) parent;
        final Node CHILD = (Node) child;
        return PARENT.getIndex(CHILD);
    }

    @Override
    public boolean isLeaf(Object node) {
        assert node instanceof Node;
        final Node NODE = (Node) node;
        return NODE.isLeaf();
    }

    @Override
    public void valueForPathChanged(TreePath path, Object newValue) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void addTreeModelListener(TreeModelListener l) {
        this.LISTENERS.add(l);
    }

    @Override
    public void removeTreeModelListener(TreeModelListener l) {
        this.LISTENERS.remove(l);
    }

    /**
     * @see TreeModelListener#treeNodesChanged(javax.swing.event.TreeModelEvent)
     * @param event
     */
    private void fireTreeNodesChanged(TreeModelEvent event) {
        this.LISTENERS.forEach(listener -> {
            listener.treeNodesChanged(event);
        });
    }

    /**
     * @see
     * TreeModelListener#treeNodesInserted(javax.swing.event.TreeModelEvent)
     * @param event
     */
    private void fireTreeNodesInserted(TreeModelEvent event) {
        this.LISTENERS.forEach(listener -> {
            listener.treeNodesInserted(event);
        });
    }

    /**
     * @see TreeModelListener#treeNodesRemoved(javax.swing.event.TreeModelEvent)
     * @param event
     */
    private void fireTreeNodesRemoved(TreeModelEvent event) {
        this.LISTENERS.forEach(listener -> {
            listener.treeNodesRemoved(event);
        });
    }

    private Node root;
    final List<TreeModelListener> LISTENERS;

    /**
     * A node class that wraps an <code>Element</code> or string provides
     * filtering.
     *
     */
    final public class Node implements Comparable<Node> {

        /**
         * Creates a new <code>Node</code> to wrap the specified element.
         *
         * @param element The element that the node will contain.
         * @param leaf A flag indicating whether or not the node will be a leaf.
         *
         */
        private Node(Element element, boolean leaf) {
            this.ELEMENT = Objects.requireNonNull(element);
            this.NAME = this.ELEMENT.toString();

            if (leaf) {
                this.CHILDREN = null;
                this.FILTERED = null;
            } else {
                this.CHILDREN = new ThreadsafeArray<>();
                this.FILTERED = new ThreadsafeArray<>();
            }

            this.isFiltered = false;
            this.isSelfFiltered = false;
            this.leafCount = 1;
        }

        /**
         * Creates a new container <code>Node</code>.
         *
         * @param name The name of the container.
         */
        private Node(String name) {
            this.NAME = Objects.requireNonNull(name);
            this.ELEMENT = null;
            this.CHILDREN = new ThreadsafeArray<>();
            this.FILTERED = new ThreadsafeArray<>();
            this.isFiltered = false;
            this.isSelfFiltered = false;
            this.leafCount = 0;
        }

        /**
         * Adds a <code>Collection</code> of children. Invalidates the filter,
         * if any.
         *
         * @param children The children to add.
         */
        public void addAll(Collection<Node> children) {
            if (this.isLeaf()) {
                throw new UnsupportedOperationException("Not supported.");
            }

            children.forEach(child -> child.setParent(this));
            this.CHILDREN.addAll(children);
            this.FILTERED.addAll(children);
            this.countLeaves();
            fireTreeNodesInserted(new TreeModelEvent(this, this.getPath()));
        }

        /**
         * Removes a child.
         *
         * @param child The child to remove.
         */
        public void remove(Node child) {
            if (this.isLeaf()) {
                throw new UnsupportedOperationException("Not supported.");
            }

            if (!this.FILTERED.contains(child)) {
                return;
            }

            int idx = this.getIndex(child);
            this.CHILDREN.remove(child);
            this.FILTERED.remove(child);
            child.setParent(null);
            this.countLeaves();

            int[] childIndex = new int[]{idx};
            Node[] childNode = new Node[]{child};
            fireTreeNodesRemoved(new TreeModelEvent(this, this.getPath(), childIndex, childNode));

            if (this.FILTERED.isEmpty() && this.isSelfFiltered) {
                this.isFiltered = true;
            }
        }

        /**
         * Retrieves the node's element and all the elements of its children.
         *
         * @return
         */
        public List<Element> getElements() {
            List<Element> collected = new LinkedList<>();

            if (this.hasElement() && !this.isFiltered()) {
                collected.add(this.ELEMENT);
            }

            if (!this.isLeaf()) {
                this.FILTERED.forEach(v -> {
                    if (v.hasElement() && v.isLeaf() && !v.isFiltered()) {
                        collected.add(v.getElement());
                    } else {
                        collected.addAll(v.getElements());
                    }
                });
            }

            return collected;
        }

        /**
         * @return The element.
         */
        public Element getElement() {
            return this.ELEMENT;
        }

        /**
         * Finds the path to a specified <code>Element</code>.
         *
         * @param element The <code>Element</code> for which to search.
         * @return A <code>TreePath</code> to the <code>Element</code>, or
         * <code>null</code> if it is not a leaf of this node.
         */
        public TreePath findPath(Element element) {
            if (this.ELEMENT == element) {
                return this.getPath();
            }

            if (!this.isLeaf()) {
                for (Node child : this.FILTERED) {
                    TreePath path = child.findPath(element);
                    if (null != path) {
                        return path;
                    }
                }
            }

            return null;
        }

        /**
         * Finds the path to a specified <code>Element</code>, ignoring
         * filtering.
         *
         * @param element The <code>Element</code> for which to search.
         * @return A <code>TreePath</code> to the <code>Element</code>, or
         * <code>null</code> if it is not a leaf of this node.
         */
        public TreePath findPathUnfiltered(Element element) {
            if (this.ELEMENT == element) {
                return this.getPath();
            }

            if (!this.isLeaf()) {
                for (Node child : this.CHILDREN) {
                    TreePath path = child.findPathUnfiltered(element);
                    if (null != path) {
                        return path;
                    }
                }
            }

            return null;
        }

        /**
         * Removes filtering along a treepath.
         *
         * @param path The path to defilter.
         * @param depth The position along of the path of the current node.
         */
        public void defilter(TreePath path, int depth) {
            Objects.requireNonNull(path);

            assert depth < path.getPathCount();
            Node head = (Node) path.getPathComponent(depth);
            assert head == this;

            if (depth + 1 < path.getPathCount() && !this.isLeaf()) {
                Node next = (Node) path.getPathComponent(depth + 1);

                if (this.CHILDREN.contains(next) && !this.FILTERED.contains(next)) {
                    this.FILTERED.add(next);
                    this.countLeaves();
                    int[] childIndex = new int[]{this.FILTERED.size() - 1};
                    Node[] childNode = new Node[]{next};
                    fireTreeNodesInserted(new TreeModelEvent(this, this.getPath(), childIndex, childNode));
                }
                next.defilter(path, depth + 1);
            }
        }

        /**
         * Removes all filtering.
         */
        synchronized public void defilter() {
            this.isFiltered = false;
            this.isSelfFiltered = false;

            if (!this.isLeaf()) {
                this.CHILDREN.forEach(child -> child.defilter());
                this.FILTERED.clear();
                this.FILTERED.addAll(this.CHILDREN);
                this.countLeaves();
            }
        }

        /**
         * Filters the node and its contents.
         *
         * @param filter The filter that determines which nodes to keep.
         * @return Returns true if the node should still be filtered out, false
         * if it should still be visible.
         */
        synchronized public boolean filter(Predicate<Node> filter) {
            Objects.requireNonNull(filter);

            // Determine if the node itself would be filtered out.
            // Never filter the root.
            this.isSelfFiltered = (null != this.parent) && !filter.test(this);

            // If there are no children, finish up.
            if (this.isLeaf()) {
                this.isFiltered = this.isSelfFiltered;
                return this.isFiltered;

            } // For Elements that contain other elements, don't filter 
            //children at all unless the Element itself is filtered. Doesn't
            // apply to the root.
            else if (this.hasElement() && !this.isSelfFiltered && this.parent != null) {
                this.FILTERED.clear();
                this.FILTERED.addAll(this.CHILDREN);
                this.isFiltered = this.isSelfFiltered;
                this.countLeaves();
                return this.isFiltered;

            } // For folders, determine which children to filter.
            else {
                this.FILTERED.clear();
                this.FILTERED.addAllUnless(this.CHILDREN, c -> c.filter(filter));
                this.isFiltered = this.isSelfFiltered && this.FILTERED.isEmpty();
                this.countLeaves();
                return this.isFiltered;
            }
        }

        /**
         * Sorts the children of the node.
         */
        synchronized public void sort() {
            if (!this.isLeaf()) {
                this.CHILDREN.sort((n1, n2) -> n1.toString().compareToIgnoreCase(n2.toString()));
                this.FILTERED.clear();
                this.FILTERED.addAll(this.CHILDREN);
            }
        }

        /**
         * @return A <code>List</code> of every <code>Element</code> contained
         * by the descendents of the <code>Node</code> (not including the node
         * itself).
         */
        private Map<Element, Node> parsePath() {
            final Map<Element, Node> ELEMENTS = new LinkedHashMap<>();

            if (!this.isLeaf()) {
                this.FILTERED.forEach(child -> {
                    if (child.hasElement()) {
                        ELEMENTS.put(child.ELEMENT, child);
                    }

                    ELEMENTS.putAll(child.parsePath());
                });
            }

            return ELEMENTS;
        }

        private int getLeafCount() {
            if (this.isLeaf() || this.hasElement()) {
                return 1;
            } else {
                return this.leafCount;
            }
        }

        /**
         * Keeps the leaf count uptodate.
         */
        private void countLeaves() {
            if (this.isLeaf() || this.hasElement()) {
                this.leafCount = 1;
            } else {
                this.leafCount = this.FILTERED.stream().mapToInt(n -> n.getLeafCount()).sum();
            }
        }

        /**
         * @return The parent of the <code>Node</code> or null if it is the
         * root.
         */
        public Node getParent() {
            return this.parent;
        }

        /**
         * Sets the parent of the <code>Node</code>.
         *
         * @param newParent The new parent of the <code>Node</code> or null if
         * it is the root.
         */
        public void setParent(Node newParent) {
            this.parent = newParent;

        }

        /**
         * @return The <code>TreePath</code> for the <code>Node</code>.
         */
        public TreePath getPath() {
            if (null == this.parent) {
                return new TreePath(this);
            } else {
                return this.parent.getPath().pathByAddingChild(this);
            }
        }

        /**
         * Retrieves the child at the specified index.
         *
         * @param childIndex The index of the child to retrieve.
         * @return The child at the specified index, or <code>null</code> if the
         * node is a leaf or the index is invalid.
         */
        public Node getChildAt(int childIndex) {
            if (this.isLeaf()) {
                throw new IllegalStateException("Leaves don't have children!!");
            } else if (childIndex >= this.FILTERED.size()) {
                return null;
                //return this.FILTERED.get(this.FILTERED.size() - 1);
                //throw new IllegalArgumentException("childIndex >= size!!");
            }

            return this.FILTERED.get(childIndex);

        }

        /**
         * @return The number of children or 0 if the node is a leaf.
         */
        public int getChildCount() {
            if (this.isLeaf()) {
                return 0;
            } else {
                return this.FILTERED.size();
            }
        }

        /**
         * Retrieves the index of a specified child node.
         *
         * @param node The node for which to determine the index.
         * @return The index or -1 if the node is not a child.
         */
        public int getIndex(Node node) {
            if (this.isLeaf()) {
                return -1;
            } else {
                return this.FILTERED.indexOf(node);
            }
        }

        /**
         * @return A flag indicating if the <code>Node</code> has an element.
         */
        public boolean hasElement() {
            return null != this.ELEMENT;
        }

        /**
         * @return A flag indicating if the <code>Node</code> is a leaf.
         */
        public boolean isLeaf() {
            return null == this.CHILDREN;
        }

        /**
         * @return A flag indicating if the <code>Node</code> is filtered or
         * not.
         */
        public boolean isFiltered() {
            return this.isFiltered;
        }

        /**
         * @return The name of the <code>Node</code>.
         */
        public String getName() {
            return this.NAME;
        }

        @Override
        public String toString() {
            if (this.hasElement()) {
                return this.NAME;
            } else if (null != this.NAME) {
                return this.NAME + " (" + this.leafCount + ")";
            } else {
                return "(root)";
            }
        }

        /**
         * Compares nodes using their name fields.
         *
         * @see java.lang.Comparable#compareTo(java.lang.Object)
         * @param o
         * @return
         */
        @Override
        public int compareTo(Node o) {
            int cmp = this.getName().compareToIgnoreCase(o.getName());
            if (cmp != 0) {
                return cmp;
            }

            return Integer.compare(this.hashCode(), o.hashCode());
        }

        final private Element ELEMENT;
        final private String NAME;
        private Node parent = null;
        final private ThreadsafeArray<Node> CHILDREN;
        final private ThreadsafeArray<Node> FILTERED;
        private boolean isSelfFiltered;
        private boolean isFiltered;
        private int leafCount;
    }
}
