/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.util.Collection;
import java.awt.event.*;
import java.awt.*;
import java.io.*;
import java.util.Collections;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableRowSorter;
import restringer.Mod;
import restringer.Settings;
import restringer.Profile;
import restringer.Game;
import restringer.pex.RemappingStats;

/**
 * Presents a list of mod to the user and allows them to be selected.
 *
 * @author Mark Fairchild
 * @version 2016/07/07
 */
public class ModChooser extends JFrame {

    /**
     * Creates a new <code>ModChooser</code>.
     *
     */
    public ModChooser() {
        this(null);
    }

    /**
     * Creates a new <code>ModChooser</code>.
     *
     * @param sw The <code>SaveWindow</code> that will receive the analysis if
     * modchooser is being called to analyze mods for a savefile.
     */
    private ModChooser(SaveWindow sw) {
        Settings settingsFile;
        try {
            settingsFile = Settings.readSettings();
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(null, "The settings file \"settings.json\" could not be read.", "Settings", JOptionPane.ERROR_MESSAGE);
            settingsFile = new Settings();
        }

        this.SETTINGS = settingsFile;
        this.selectedProfile = this.SETTINGS.getSelectedProfile();
        this.SAVEWINDOW = sw;

        this.MODEL = new ModTableModel(this.SETTINGS, this.selectedProfile);
        this.TABLE = new JTable(this.MODEL);
        this.PROCESS = new JButton("Make Patch");
        this.OUTPUT_PATH = new JButton();

        this.LOAD_MOD2 = new JMenuItem("Load a mod", KeyEvent.VK_L);
        this.LOAD_MODORGANIZER = new JMenuItem("Load mods from Mod Organizer", KeyEvent.VK_M);
        this.LOAD_SKYRIM = new JMenuItem("Load Skyrim DATA directory", KeyEvent.VK_D);
        this.SELECT_ALL = new JMenuItem("Select all", KeyEvent.VK_A);
        this.SELECT_NONE = new JMenuItem("Select none", KeyEvent.VK_N);

        this.LOAD_MOD1 = new JMenuItem("Load a mod", KeyEvent.VK_L);
        this.REMOVE_MOD = new JMenuItem("Remove mod", KeyEvent.VK_R);
        this.SET_PATCH_DIR = new JMenuItem("Set patch directory", KeyEvent.VK_P);
        this.SET_SKYRIM_DIR = new JMenuItem("Set Skyrim directory", KeyEvent.VK_S);
        this.SET_MO_DIR = new JMenuItem("Set ModOrganizer directory", KeyEvent.VK_D);
        this.EXIT = new JMenuItem("Exit", KeyEvent.VK_E);

        this.CREATE_PROFILE = new JMenuItem("Create Profile", KeyEvent.VK_C);
        this.DELETE_PROFILE = new JMenuItem("Delete Profile", KeyEvent.VK_D);

        this.POPUP_MENU = new JPopupMenu("Actions");
        this.MOD_MENU = new JMenu("File");
        this.PROFILE_MENU = new JMenu("Profiles");
        this.TXTFILTER = new JTextField(20);
        this.SORTER = new TableRowSorter<>(this.MODEL);
        this.MENUBAR = new JMenuBar();
        this.LBLARROW = new JLabel("\u2192");
        this.LBLFILTER = new JLabel("Filter:");
        this.LBLPROFILE = new JLabel("Profile:");

        final Profile[] PROFILES = new Profile[this.SETTINGS.profiles.size()];
        this.SETTINGS.profiles.toArray(PROFILES);
        this.PROFILESMODEL = new DefaultComboBoxModel<>(PROFILES);
        this.CBPROFILES = new JComboBox<>(this.PROFILESMODEL);

        this.CBCOMPRESS = new JCheckBox("Compress patch?", false);

        this.LOGWINDOW = new LogWindow();
        this.PROFILEINFO = new ProfileInfoPanel();

        this.HOLDER = new JPanel(new FlowLayout(FlowLayout.LEADING));
        this.BUTTONPANEL = new JPanel();
        this.FILTERPANEL = new JPanel(new FlowLayout(FlowLayout.LEADING));
        this.MODPANEL = new JPanel(new BorderLayout());
        this.LOGPANEL = new JPanel();
        this.INFOPANEL = new JPanel();
        this.PROGRESSPANEL = new JPanel(new FlowLayout(FlowLayout.LEADING));
        this.BOTTOMPANEL = new JPanel(new BorderLayout());
        this.SCROLLER = new JScrollPane(this.TABLE, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        this.INFOSCROLLER = new JScrollPane(this.INFOPANEL, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        this.RIGHTSPLITTER = new JSplitPane(JSplitPane.VERTICAL_SPLIT, this.INFOSCROLLER, this.LOGPANEL);
        this.MAINSPLITTER = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, this.MODPANEL, this.RIGHTSPLITTER);
        this.RENDERER = new ModTableCellRenderer();
        this.LBLMEMORY = new MemoryLabel();

        this.initComponents();
    }

    /**
     * Initialize the swing and AWT components.
     */
    private void initComponents() {
        this.setTitle("ReStringer");
        try {
            final InputStream INPUT = ModChooser.class.getResourceAsStream("String.png");
            final Image ICON = ImageIO.read(INPUT);
            this.setIconImage(ICON);
        } catch (IOException ex) {
        }

        LOG.getParent().addHandler(this.LOGWINDOW.getHandler());

        this.TABLE.setPreferredScrollableViewportSize(new Dimension(500, 700));
        this.TABLE.setFillsViewportHeight(true);
        this.TABLE.setTransferHandler(new ModTableTransferHandler(this.MODEL));
        this.TABLE.setDropMode(DropMode.ON_OR_INSERT_ROWS);
        this.TABLE.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        this.TABLE.setDragEnabled(true);

        this.TABLE.getColumn(this.TABLE.getColumnName(0)).setMaxWidth(25);
        this.TABLE.getColumn(this.TABLE.getColumnName(0)).setMinWidth(25);
        this.TABLE.getColumn(this.TABLE.getColumnName(2)).setMaxWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(2)).setMinWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(3)).setMaxWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(3)).setMinWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(4)).setMaxWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(4)).setMinWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(5)).setMaxWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(5)).setMinWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(6)).setMaxWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(6)).setMinWidth(50);
        this.TABLE.getColumn(this.TABLE.getColumnName(7)).setMaxWidth(60);
        this.TABLE.getColumn(this.TABLE.getColumnName(7)).setMinWidth(60);

        this.PROCESS.setFont(this.PROCESS.getFont().deriveFont(Font.BOLD));

        this.CBPROFILES.setPrototypeDisplayValue(new Profile("RestringerProfile Goes Here"));
        this.CBPROFILES.setSelectedItem(this.selectedProfile);

        this.updateSettings();

        this.LOAD_MOD2.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L, KeyEvent.CTRL_MASK));
        this.LOAD_MODORGANIZER.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_M, KeyEvent.CTRL_MASK));
        this.SELECT_ALL.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, KeyEvent.CTRL_MASK));
        this.SELECT_NONE.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, KeyEvent.CTRL_MASK));
        this.REMOVE_MOD.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, KeyEvent.CTRL_MASK));
        this.SET_PATCH_DIR.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, KeyEvent.CTRL_MASK));
        this.SET_SKYRIM_DIR.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, KeyEvent.CTRL_MASK));
        this.SET_MO_DIR.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, KeyEvent.CTRL_MASK));
        this.EXIT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E, KeyEvent.CTRL_MASK));

        this.POPUP_MENU.add(this.LOAD_MOD1);
        this.POPUP_MENU.add(this.REMOVE_MOD);
        this.POPUP_MENU.add(this.SELECT_ALL);
        this.POPUP_MENU.add(this.SELECT_NONE);

        this.MOD_MENU.add(this.LOAD_MOD2);
        this.MOD_MENU.add(this.LOAD_MODORGANIZER);
        this.MOD_MENU.add(this.LOAD_SKYRIM);
        this.MOD_MENU.addSeparator();
        this.MOD_MENU.add(this.SET_PATCH_DIR);
        this.MOD_MENU.add(this.SET_SKYRIM_DIR);
        this.MOD_MENU.add(this.SET_MO_DIR);
        this.MOD_MENU.addSeparator();
        this.MOD_MENU.addSeparator();
        this.MOD_MENU.add(this.EXIT);
        this.MOD_MENU.setMnemonic('D');
        this.REMOVE_MOD.setEnabled(false);

        this.PROFILE_MENU.add(this.CREATE_PROFILE);
        this.PROFILE_MENU.add(this.DELETE_PROFILE);

        this.MENUBAR.add(this.MOD_MENU);
        this.MENUBAR.add(this.PROFILE_MENU);

        this.MAINSPLITTER.setResizeWeight(0.5);
        this.RIGHTSPLITTER.setResizeWeight(0.2);

        this.FILTERPANEL.add(this.LBLPROFILE);
        this.FILTERPANEL.add(this.CBPROFILES);

        this.FILTERPANEL.add(this.LBLFILTER);
        this.FILTERPANEL.add(this.TXTFILTER);

        this.BUTTONPANEL.setLayout(new BoxLayout(BUTTONPANEL, BoxLayout.LINE_AXIS));
        this.BUTTONPANEL.add(this.PROCESS);
        this.BUTTONPANEL.add(this.LBLARROW);
        this.BUTTONPANEL.add(this.OUTPUT_PATH);
        this.BUTTONPANEL.add(new JLabel("     "));
        this.BUTTONPANEL.add(this.CBCOMPRESS);
        this.HOLDER.add(BUTTONPANEL);

        this.LBLMEMORY.setFont(new Font(Font.MONOSPACED, 0, 12));
        this.PROGRESSPANEL.add(this.LBLMEMORY);
        this.BOTTOMPANEL.add(BorderLayout.NORTH, this.PROGRESSPANEL);
        this.BOTTOMPANEL.add(BorderLayout.SOUTH, this.HOLDER);

        this.MODPANEL.add(this.FILTERPANEL, BorderLayout.NORTH);
        this.MODPANEL.add(this.SCROLLER, BorderLayout.CENTER);
        this.MODPANEL.add(this.BOTTOMPANEL, BorderLayout.SOUTH);

        this.INFOPANEL.setBorder(BorderFactory.createTitledBorder("Info"));
        this.INFOPANEL.setLayout(new GridLayout(1, 1));
        this.INFOPANEL.add(this.PROFILEINFO);
        this.PROFILEINFO.setProfile(this.SETTINGS.getSelectedProfile());

        this.LOGPANEL.setBorder(BorderFactory.createTitledBorder("Log"));
        this.LOGPANEL.setLayout(new GridLayout(1, 1));
        this.LOGPANEL.add(this.LOGWINDOW);

        this.TABLE.setDefaultRenderer(String.class, this.RENDERER);
        this.TABLE.setDefaultRenderer(Integer.class, this.RENDERER);

        this.setLayout(new BorderLayout());
        this.add(this.MAINSPLITTER, BorderLayout.CENTER);

        this.setJMenuBar(this.MENUBAR);
        this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        this.pack();
        this.setLocationRelativeTo(null);
        this.loadPreferences();

        this.getGlassPane().addKeyListener(this.KEYBOARDHANDLER);
        this.TABLE.getSelectionModel().addListSelectionListener(this.TABLEHANDLER);

        // Set up the popup menu.
        this.LOAD_MOD1.addActionListener(this.ACTIONHANDLER);
        this.LOAD_MOD2.addActionListener(this.ACTIONHANDLER);
        this.LOAD_MODORGANIZER.addActionListener(this.ACTIONHANDLER);
        this.LOAD_SKYRIM.addActionListener(this.ACTIONHANDLER);
        this.SELECT_ALL.addActionListener(this.ACTIONHANDLER);
        this.SELECT_NONE.addActionListener(this.ACTIONHANDLER);
        this.REMOVE_MOD.addActionListener(ACTIONHANDLER);
        this.SET_PATCH_DIR.addActionListener(this.ACTIONHANDLER);
        this.SET_MO_DIR.addActionListener(ACTIONHANDLER);
        this.EXIT.addActionListener(this.ACTIONHANDLER);
        this.PROCESS.addActionListener(ACTIONHANDLER);
        this.OUTPUT_PATH.addActionListener(ACTIONHANDLER);
        this.CREATE_PROFILE.addActionListener(ACTIONHANDLER);
        this.DELETE_PROFILE.addActionListener(ACTIONHANDLER);
        this.SET_SKYRIM_DIR.addActionListener(ACTIONHANDLER);
        this.CBCOMPRESS.addActionListener(this.ACTIONHANDLER);

        this.SCROLLER.addMouseListener(this.MOUSEHANDLER);
        this.TABLE.addMouseListener(this.MOUSEHANDLER);
        this.TABLE.setRowSorter(this.SORTER);
        this.MODEL.addTableModelListener(this.TABLEHANDLER);

        this.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                ModChooser.this.exit();
            }
        });

        this.CBPROFILES.addActionListener(e -> {
            if (e.getSource() == CBPROFILES) {
                this.selectedProfile = (Profile) CBPROFILES.getSelectedItem();
                int index = SETTINGS.profiles.indexOf(this.selectedProfile);
                SETTINGS.selectedProfile = index;
                MODEL.setProfile(this.selectedProfile);
                PROFILEINFO.setProfile(SETTINGS.getSelectedProfile());
            }
        });

        this.TXTFILTER.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void insertUpdate(DocumentEvent e) {
                filter();
            }

            @Override
            public void removeUpdate(DocumentEvent e) {
                filter();
            }

            @Override
            public void changedUpdate(DocumentEvent e) {
                filter();
            }
        });

        this.TABLE.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), "deleteSelected");
        this.TABLE.getActionMap().put("deleteSelected", new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                ModChooser.this.removeMod();
            }
        });

        this.TABLE.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_HOME, 0), "selectFirstRow");
        this.TABLE.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_END, 0), "selectLastRow");
        this.TABLE.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_HOME, KeyEvent.SHIFT_DOWN_MASK), "selectToFirstRow");
        this.TABLE.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_END, KeyEvent.SHIFT_DOWN_MASK), "selectToLastRow");
        this.TABLE.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, 0), "toggleChecksOnSelection");

        this.TABLE.getActionMap().put("selectToFirstRow", new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                TABLE.getSelectionModel().setLeadSelectionIndex(0);
            }
        });

        this.TABLE.getActionMap().put("selectToLastRow", new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                int last = TABLE.getRowCount() - 1;
                TABLE.getSelectionModel().setLeadSelectionIndex(last);
            }
        });

        this.TABLE.getActionMap().put("toggleChecksOnSelection", new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                int[] rows = TABLE.getSelectedRows();
                for (int i = 0; i < rows.length; i++) {
                    rows[i] = TABLE.convertRowIndexToModel(rows[i]);
                }

                MODEL.toggleRows(rows);
            }
        });

        this.LBLMEMORY.initialize();
    }

    /**
     * Save minor settings like window position, size, and state.
     */
    private void savePreferences() {
        PREFS.putInt("mods.extendedState", this.getExtendedState());
        PREFS.putInt("mods.windowWidth", this.getSize().width);
        PREFS.putInt("mods.windowHeight", this.getSize().height);
        PREFS.putInt("mods.windowX", this.getLocation().x);
        PREFS.putInt("mods.windowY", this.getLocation().y);
        PREFS.putInt("mods.mainDivider", this.MAINSPLITTER.getDividerLocation());
        PREFS.putBoolean("mods.compress", this.CBCOMPRESS.isSelected());
    }

    /**
     * Loads minor settings like window position, size, and state.
     */
    private void loadPreferences() {
        int state = PREFS.getInt("mods.extendedState", JFrame.MAXIMIZED_BOTH);
        int width = PREFS.getInt("mods.windowWidth", this.getSize().width);
        int height = PREFS.getInt("mods.windowHeight", this.getSize().height);
        int x = PREFS.getInt("mods.windowX", this.getLocation().x);
        int y = PREFS.getInt("mods.windowY", this.getLocation().y);
        int mainDivider = PREFS.getInt("mods.mainDivider", this.MAINSPLITTER.getDividerLocation());
        boolean compress = PREFS.getBoolean("mods.compress", false);

        this.CBCOMPRESS.setSelected(compress);
        this.setLocation(x, y);
        this.setSize(width, height);
        this.setExtendedState(state);
        this.MAINSPLITTER.setDividerLocation(mainDivider);
    }

    /**
     * Exits the app.
     */
    private void exit() {
        SwingUtilities.invokeLater(() -> {
            ModChooser.this.savePreferences();
            this.LBLMEMORY.terminate();
            this.setVisible(false);
            this.dispose();
        });
    }

    /**
     *
     * @param mods
     */
    final public void addMods(Collection<Mod> mods) {
        this.MODEL.addAllMods(mods);
    }

    /**
     * Tries to add the Skyrim base game.
     */
    private void addSkyrim() {
        try {
            this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

            File skyrimDir = Configurator.getGameDirectory(Game.SKYRIM_LE);
            if (!Configurator.validateGameDirectory(Game.SKYRIM_LE, skyrimDir)) {
                skyrimDir = Configurator.selectGameDirectory(this, Game.SKYRIM_LE);
            }
            if (!Configurator.validateGameDirectory(Game.SKYRIM_LE, skyrimDir)) {
                return;
            }

            final Mod SKYRIM = new Mod(Game.SKYRIM_LE, skyrimDir);
            this.MODEL.addMod(0, SKYRIM, false);

        } finally {
            this.setCursor(Cursor.getDefaultCursor());
        }
    }

    /**
     * Tries to add the mods from ModOrganizer to the list.
     */
    private void addModOrganizerMods() {
        try {
            if (this.MODEL.getRowCount() > 0) {
                final String MSG = "This will overwrite the current list of mods. Is that okay?";
                int result = JOptionPane.showConfirmDialog(this, MSG, "Overwrite?", JOptionPane.YES_NO_OPTION);
                if (result == JOptionPane.NO_OPTION) {
                    return;
                }
            }

            this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

            File moDir = Configurator.getModOrganizerDirectory();
            if (!Configurator.validateModOrganizerDirectory(moDir)) {
                moDir = Configurator.selectModOrganizerDirectory(this);

                if (!Configurator.validateModOrganizerDirectory(moDir)) {
                    return;
                }
            }

            final java.util.List<Mod> MODS = Configurator.analyzeModOrganizer(Game.SKYRIM_LE, moDir);
            Collections.reverse(MODS);

            //sortedMods.addAll(unsortedMods);
            this.MODEL.removeAll();
            this.MODEL.addAllMods(MODS);

        } finally {
            this.setCursor(Cursor.getDefaultCursor());
        }
    }

    /**
     * Tries to set the output directory.
     */
    private void setOutputDirectory() {
        File prevOutput = this.selectedProfile.getOutputDirectory();
        File newOutput = Configurator.selectOutputDirectory(prevOutput, this);

        if (Configurator.validateOutputPath(newOutput)) {
            this.selectedProfile.setOutputDirectory(newOutput);
            this.updateSettings();
            this.saveSettings();
        }
    }

    /**
     * Updates the output directory.
     */
    private void updateSettings() {
        final File OUTPUTDIR = this.selectedProfile.getOutputDirectory();

        if (OUTPUTDIR == null) {
            this.OUTPUT_PATH.setText("PATH FOR PATCH NEEDS TO BE SET");
            this.PROCESS.setEnabled(false);
        } else if (!Configurator.validateOutputPath(OUTPUTDIR)) {
            this.OUTPUT_PATH.setText("INVALID OUTPUT PATH");
            this.PROCESS.setEnabled(false);
        } else {
            this.OUTPUT_PATH.setText(OUTPUTDIR.getPath());
            int numSelected = this.MODEL.getNumChecked();

            if (numSelected > 0) {
                this.PROCESS.setEnabled(true);
            } else {
                this.PROCESS.setEnabled(false);
            }
        }

        this.PROFILEINFO.update();
    }

    /**
     * Writes the settings and profiles to a file.
     */
    public void saveSettings() {
        try {
            Settings.writeSettings(ModChooser.this.SETTINGS);
        } catch (IOException ex) {
            final String MSG = String.format("Unable to write to \"settings.json\". Settings will not be saved.");
            LOG.warning(MSG);
            JOptionPane.showMessageDialog(ModChooser.this.getParent(), MSG, "Settings", JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Creates a new profile.
     */
    private void createProfile() {
        final String MSG = "Name for the new profile?";
        String newName = JOptionPane.showInputDialog(this, MSG, "Create Profile", JOptionPane.QUESTION_MESSAGE);

        if (!Profile.validateName(newName)) {
            JOptionPane.showMessageDialog(this, "Invalid profile name.", "Error", JOptionPane.ERROR_MESSAGE);
            return;
        }

        for (Profile profile : this.SETTINGS.profiles) {
            if (profile.getName().equalsIgnoreCase(newName)) {
                JOptionPane.showMessageDialog(this, "There is already a profile with that name.", "Error", JOptionPane.ERROR_MESSAGE);
                return;
            }
        }

        Profile newProfile = new Profile(newName);
        this.SETTINGS.profiles.add(newProfile);
        this.PROFILESMODEL.addElement(newProfile);
        this.PROFILESMODEL.setSelectedItem(newProfile);
        this.saveSettings();
    }

    /**
     * Deletes the current profile.
     */
    private void deleteProfile() {
        final String MSG = "Are you sure you want to delete the current profile?";
        int result = JOptionPane.showConfirmDialog(this, MSG, "Delete Profile", JOptionPane.YES_NO_OPTION);

        if (result != JOptionPane.YES_OPTION) {
            return;
        }

        int index = this.SETTINGS.profiles.indexOf(this.selectedProfile);

        this.SETTINGS.profiles.remove(this.selectedProfile);
        this.PROFILESMODEL.removeElement(this.selectedProfile);

        if (this.SETTINGS.profiles.isEmpty()) {
            Profile newProfile = this.SETTINGS.getSelectedProfile();
            this.PROFILESMODEL.addElement(newProfile);
            this.PROFILESMODEL.setSelectedItem(newProfile);
        } else if (index > 0) {
            index--;
            this.SETTINGS.selectedProfile = index;
            Profile newProfile = this.SETTINGS.getSelectedProfile();
            this.PROFILESMODEL.setSelectedItem(newProfile);
        } else {
            this.SETTINGS.selectedProfile = index;
            Profile newProfile = this.SETTINGS.getSelectedProfile();
            this.PROFILESMODEL.setSelectedItem(newProfile);
        }

        this.saveSettings();
    }

    /**
     * Tries to set the ModOrganizer directory.
     */
    private void setModOrganizerDirectory() {
        Configurator.selectModOrganizerDirectory(this);
    }

    /**
     * Tries to set the Skyrim directory.
     */
    private void setSkyrimDirectory() {
        Configurator.selectGameDirectory(this, Game.SKYRIM_LE);
    }

    /**
     * Responds to the table data being changed.
     */
    private void tableChanged() {
        int numSelected = ModChooser.this.MODEL.getNumChecked();
        String txt = String.format("MAKE PATCH! (%d mods)", numSelected);
        ModChooser.this.PROCESS.setText(txt);
        this.updateSettings();
    }

    /**
     * Adds a mod (or maybe a few mods).
     */
    private void addMod() {
        File selection = Configurator.selectMod(this);
        if (null == selection) {
            return;
        }

        final Mod MOD = new Mod(Game.SKYRIM_LE, selection);
        this.addMods(Collections.singleton(MOD));
    }

    /**
     * Removes a mod that is under the cursor.
     */
    private void removeMod() {
        int[] selected = this.TABLE.getSelectedRows();
        final java.util.List<Mod> MODS = new java.util.LinkedList<>();

        for (int i = 0; i < selected.length; i++) {
            int rowIndex = selected[i];
            Mod mod = (Mod) this.MODEL.getMod(rowIndex);
            MODS.add(mod);
        }

        this.MODEL.removeMods(MODS);
    }

    /**
     * Filters the mod list.
     */
    private void filter() {
        try {
            String txt = this.TXTFILTER.getText();
            if (txt.isEmpty()) {
                RowFilter<ModTableModel, Integer> nameFilter = new RowFilter<ModTableModel, Integer>() {
                    @Override
                    public boolean include(RowFilter.Entry<? extends ModTableModel, ? extends Integer> entry) {
                        return true;
                    }
                };

                this.SORTER.setRowFilter(nameFilter);
            } else {

                final String REGEX = txt;
                final Pattern PATTERN = Pattern.compile(REGEX, Pattern.CASE_INSENSITIVE);

                RowFilter<ModTableModel, Integer> nameFilter = new RowFilter<ModTableModel, Integer>() {
                    @Override
                    public boolean include(RowFilter.Entry<? extends ModTableModel, ? extends Integer> entry) {
                        ModTableModel model = entry.getModel();
                        Mod mod = model.getMod(entry.getIdentifier());
                        String modName = mod.getName();
                        final Matcher MATCHER = PATTERN.matcher(modName);
                        boolean match = MATCHER.find();
                        return match;
                    }
                };

                this.SORTER.setRowFilter(nameFilter);
            }
        } catch (PatternSyntaxException ex) {
        }
    }

    /**
     * Starts the actual processing.
     */
    private void process() {
        if (!Configurator.validateOutputPath(this.selectedProfile.getOutputDirectory())) {
            this.setOutputDirectory();

            if (!Configurator.validateOutputPath(this.selectedProfile.getOutputDirectory())) {
                return;
            }
        }

        ProgressIndicator.startWaitCursor(getRootPane());
        final ProgressModel PROCESSMODEL = new ProgressModel();
        final JProgressBar BAR = new JProgressBar(PROCESSMODEL);
        BAR.setPreferredSize(new Dimension(200, BAR.getPreferredSize().height));
        this.PROGRESSPANEL.add(BAR);

        SwingWorker<RemappingStats, Object> worker = new SwingWorker<RemappingStats, Object>() {
            @Override
            protected RemappingStats doInBackground() throws Exception {

                try {
                    boolean compress = CBCOMPRESS.isSelected();
                    RemappingStats stats = selectedProfile.execute2(PROCESSMODEL, compress);
                    JOptionPane.showMessageDialog(ModChooser.this, stats.toString(), "FINISHED", JOptionPane.INFORMATION_MESSAGE);
                    return stats;

                } catch (Exception | Error ex) {
                    final String MSG = String.format("Error while restringing scripts:\n%s", ex.getMessage());
                    LOG.severe(MSG);
                    LOG.severe(ex.toString());
                    JOptionPane.showMessageDialog(ModChooser.this, MSG, "Write Error", JOptionPane.ERROR_MESSAGE);
                    System.out.println(ex.getMessage());
                    ex.printStackTrace(System.out);
                    return null;

                } finally {
                    ProgressIndicator.stopWaitCursor(getRootPane());
                    PROGRESSPANEL.remove(BAR);
                }
            }
        };

        worker.execute();
    }

    /**
     * Handles action events.
     */
    final private class ActionHandler implements ActionListener {

        @Override
        public void actionPerformed(ActionEvent e) {
            if (e.getSource() == ModChooser.this.LOAD_MODORGANIZER) {
                ModChooser.this.addModOrganizerMods();
            } else if (e.getSource() == ModChooser.this.LOAD_MOD1) {
                ModChooser.this.addMod();
            } else if (e.getSource() == ModChooser.this.LOAD_MOD2) {
                ModChooser.this.addMod();
            } else if (e.getSource() == ModChooser.this.LOAD_SKYRIM) {
                ModChooser.this.addSkyrim();
            } else if (e.getSource() == ModChooser.this.REMOVE_MOD) {
                ModChooser.this.removeMod();
            } else if (e.getSource() == ModChooser.this.SELECT_ALL) {
                ModChooser.this.MODEL.checkAll();
            } else if (e.getSource() == ModChooser.this.SELECT_NONE) {
                ModChooser.this.MODEL.checkNone();
            } else if (e.getSource() == ModChooser.this.SET_PATCH_DIR) {
                ModChooser.this.setOutputDirectory();
            } else if (e.getSource() == ModChooser.this.SET_MO_DIR) {
                ModChooser.this.setModOrganizerDirectory();
            } else if (e.getSource() == ModChooser.this.PROCESS) {
                ModChooser.this.process();
            } else if (e.getSource() == ModChooser.this.EXIT) {
                ModChooser.this.exit();
            } else if (e.getSource() == ModChooser.this.OUTPUT_PATH) {
                ModChooser.this.setOutputDirectory();
            } else if (e.getSource() == ModChooser.this.TXTFILTER) {
                ModChooser.this.filter();
            } else if (e.getSource() == ModChooser.this.CREATE_PROFILE) {
                ModChooser.this.createProfile();
            } else if (e.getSource() == ModChooser.this.DELETE_PROFILE) {
                ModChooser.this.deleteProfile();
            } else if (e.getSource() == ModChooser.this.SET_SKYRIM_DIR) {
                ModChooser.this.setSkyrimDirectory();
            } else if (e.getSource() == ModChooser.this.CBCOMPRESS) {
                ModChooser.this.savePreferences();
            } else {
                JOptionPane.showMessageDialog(ModChooser.this, "Do something else.");
            }
        }
    }

    /**
     * Handles keyboard events.
     */
    final private class KeyboardHandler implements KeyListener {

        @Override
        public void keyTyped(KeyEvent e) {
        }

        @Override
        public void keyPressed(KeyEvent e) {
        }

        @Override
        public void keyReleased(KeyEvent e) {
        }

    }

    /**
     * Handles mouse events.
     */
    final private class MouseHandler extends MouseAdapter {

        @Override
        public void mouseReleased(MouseEvent e) {
            if (e.isPopupTrigger()) {
                int[] selectedRows = TABLE.getSelectedRows();
                if (selectedRows == null || selectedRows.length == 0) {
                    int row = TABLE.rowAtPoint(e.getPoint());
                    TABLE.getSelectionModel().setSelectionInterval(row, row);
                }
                ModChooser.this.POPUP_MENU.show(e.getComponent(), e.getX(), e.getY());
            }
        }
    }

    /**
     * Handles table change events.
     */
    final private class TableModelHandler implements TableModelListener, ListSelectionListener {

        @Override
        public void tableChanged(TableModelEvent e) {
            if (e.getColumn() == 0 || e.getColumn() == TableModelEvent.ALL_COLUMNS) {
                ModChooser.this.tableChanged();
            }

            if (e.getColumn() == 7 || e.getColumn() == TableModelEvent.ALL_COLUMNS) {
                ModChooser.this.PROFILEINFO.update();
            }
        }

        @Override
        public void valueChanged(ListSelectionEvent e) {
            int[] selection = ModChooser.this.TABLE.getSelectedRows();
            if (selection.length == 0) {
                REMOVE_MOD.setEnabled(false);
            } else {
                REMOVE_MOD.setEnabled(true);
            }
        }

    }

    /**
     * Used to render cells.
     */
    final private class ModTableCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            int index = TABLE.getRowSorter().convertRowIndexToModel(row);
            Mod mod = MODEL.getMod(index);

            Component c;

            if (column > 1 && null == value) {
                c = super.getTableCellRendererComponent(table, "(reading)", isSelected, hasFocus, row, column);
            } else {
                c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            }

            if (mod.isEmpty()) {
                c.setFont(c.getFont().deriveFont(Font.ITALIC));

                if (isSelected) {
                    c.setForeground(Color.LIGHT_GRAY);
                } else {
                    c.setForeground(new Color(128, 0, 0));
                }
            } else {
                c.setFont(c.getFont().deriveFont(0));

                if (isSelected) {
                    c.setForeground(Color.WHITE);
                } else {
                    c.setForeground(Color.BLACK);
                }
            }

            return c;
        }
    }

    private Profile selectedProfile;
    final private Settings SETTINGS;
    final private ModTableModel MODEL;
    final private TableRowSorter<ModTableModel> SORTER;
    final private MemoryLabel LBLMEMORY;
    final private JTable TABLE;
    final private JButton PROCESS;
    final private JButton OUTPUT_PATH;
    final private JMenuItem LOAD_MODORGANIZER;
    final private JMenuItem LOAD_SKYRIM;
    final private JMenuItem LOAD_MOD1;
    final private JMenuItem LOAD_MOD2;
    final private JMenuItem SELECT_ALL;
    final private JMenuItem SELECT_NONE;
    final private JMenuItem REMOVE_MOD;
    final private JMenuItem SET_PATCH_DIR;
    final private JMenuItem SET_MO_DIR;
    final private JMenuItem SET_SKYRIM_DIR;
    final private JMenuItem EXIT;
    final private JMenuItem CREATE_PROFILE;
    final private JMenuItem DELETE_PROFILE;
    final private JMenu MOD_MENU;
    final private JMenu PROFILE_MENU;
    final private JPopupMenu POPUP_MENU;
    final private JTextField TXTFILTER;
    final private JLabel LBLARROW;
    final private JLabel LBLFILTER;
    final private JLabel LBLPROFILE;
    final private JCheckBox CBCOMPRESS;
    final private JComboBox<Profile> CBPROFILES;
    final private DefaultComboBoxModel<Profile> PROFILESMODEL;
    final private MouseHandler MOUSEHANDLER = new MouseHandler();
    final private ActionHandler ACTIONHANDLER = new ActionHandler();
    final private KeyboardHandler KEYBOARDHANDLER = new KeyboardHandler();
    final private TableModelHandler TABLEHANDLER = new TableModelHandler();
    final private LogWindow LOGWINDOW;
    final private ProfileInfoPanel PROFILEINFO;
    final private ModTableCellRenderer RENDERER;
    final private JMenuBar MENUBAR;
    final private JPanel BOTTOMPANEL;
    final private JPanel BUTTONPANEL;
    final private JPanel FILTERPANEL;
    final private JPanel HOLDER;
    final private JScrollPane SCROLLER;
    final private JScrollPane INFOSCROLLER;
    final private JPanel MODPANEL;
    final private JPanel LOGPANEL;
    final private JPanel INFOPANEL;
    final private JPanel PROGRESSPANEL;
    final private JSplitPane MAINSPLITTER;
    final private JSplitPane RIGHTSPLITTER;
    final private SaveWindow SAVEWINDOW;

    static final private Logger LOG = Logger.getLogger(ModChooser.class.getCanonicalName());
    static final private java.util.prefs.Preferences PREFS = java.util.prefs.Preferences.userNodeForPackage(ModChooser.class);

}
