/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import restringer.Game;
import it.unimi.dsi.fastutil.ints.IntSet;
import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.PrintWriter;
import java.io.IOException;
import java.net.JarURLConnection;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import java.util.function.Predicate;
import java.util.regex.*;
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;
import java.util.stream.Collectors;
import javafx.application.Platform;
import javafx.embed.swing.JFXPanel;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.plaf.basic.BasicComboBoxRenderer;
import javax.swing.text.StyleContext;
import javax.swing.tree.TreePath;
import restringer.IString;
import restringer.Mod;
import restringer.Analysis;
import restringer.ess.*;
import restringer.ess.papyrus.*;
import restringer.gui.FilterTreeModel.Node;

/**
 * A window that displays savegame data and allows limited editing.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
public class SaveWindow extends JFrame implements ActionListener {

    /**
     *
     */
    public SaveWindow() {
        this(null);
    }

    /**
     * @param saveFile
     */
    public SaveWindow(File saveFile) {
        final JFXPanel jfxPanel = new JFXPanel();

        this.TIMER = new restringer.Timer("SaveWindow timer");
        this.TIMER.start();

        this.save = null;
        this.analysis = null;
        this.filter = null;

        this.TREE = new FilterTree();
        this.TREESCROLLER = new JScrollPane(this.TREE);
        this.TOPPANEL = new JPanel();
        this.MODPANEL = new JPanel(new FlowLayout(FlowLayout.LEADING));
        this.MODCOMBO = new JComboBox<>();
        this.MODLABEL = new JLabel("Mod Filter:");
        this.PLUGINCOMBO = new JComboBox<>();
        this.FILTERFIELD = new JTextField(14);
        this.FILTERPANEL = new JPanel(new FlowLayout(FlowLayout.LEADING));
        this.MAINPANEL = new JPanel(new BorderLayout());
        this.PROGRESSPANEL = new JPanel(new FlowLayout(FlowLayout.LEADING));
        this.PROGRESSBAR = new JProgressBar();
        this.TABLE = new VariableTable();
        this.DATASCROLLER = new JScrollPane(this.TABLE);
        this.INFOPANE = new JTextPane();
        this.INFOSCROLLER = new JScrollPane(this.INFOPANE);
        this.RIGHTSPLITTER = new JSplitPane(JSplitPane.VERTICAL_SPLIT, this.INFOSCROLLER, this.DATASCROLLER);
        this.MAINSPLITTER = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, this.MAINPANEL, this.RIGHTSPLITTER);
        this.MENUBAR = new JMenuBar();
        this.FILEMENU = new JMenu("File");
        this.CLEANMENU = new JMenu("Clean");
        this.OPTIONSMENU = new JMenu("Options");
        this.DATAMENU = new JMenu("Data");
        this.HELPMENU = new JMenu("Help");

        this.MI_EXIT = new JMenuItem("Exit", KeyEvent.VK_E);
        this.MI_LOAD = new JMenuItem("Open", KeyEvent.VK_O);
        this.MI_LOADESPS = new JMenuItem("Parse ESP/ESMs.", KeyEvent.VK_P);
        this.MI_SAVE = new JMenuItem("Save", KeyEvent.VK_S);
        this.MI_SAVEAS = new JMenuItem("Save As", KeyEvent.VK_A);
        this.MI_EXPORTPLUGINS = new JMenuItem("Export plugin list", KeyEvent.VK_X);

        this.MI_SHOWUNATTACHED = new JCheckBoxMenuItem("Show unattached instances", false);
        this.MI_SHOWUNDEFINED = new JCheckBoxMenuItem("Show undefined elements", false);
        this.MI_SHOWNULLREFS = new JCheckBoxMenuItem("Show Formlists containg nullrefs", false);
        this.MI_SHOWNONEXISTENTCREATED = new JCheckBoxMenuItem("Show non-existent-form instances", false);

        this.MI_REMOVEUNATTACHED = new JMenuItem("Remove unattached instances", KeyEvent.VK_1);
        this.MI_REMOVEUNDEFINED = new JMenuItem("Remove undefined elements", KeyEvent.VK_2);
        this.MI_RESETHAVOK = new JMenuItem("Reset Havok", KeyEvent.VK_3);
        this.MI_CLEANSEFORMLISTS = new JMenuItem("Purify FormLists", KeyEvent.VK_4);
        this.MI_REMOVENONEXISTENT = new JMenuItem("Remove non-existent form instances", KeyEvent.VK_5);
        this.MI_BATCHCLEAN = new JMenuItem("Batch Clean", KeyEvent.VK_6);
        this.MI_KILL = new JMenuItem("Kill Listed");

        this.MI_USEMO = new JCheckBoxMenuItem("Integrate with Mod Organizer", false);
        this.MI_USENMM = new JCheckBoxMenuItem("Integrate with Nexus Mod Manager", false);
        this.MI_USENMM.setEnabled(false);
        this.MI_SHOWMODS = new JCheckBoxMenuItem("Show Mod Filter box", false);

        this.MI_LOOKUPID = new JMenuItem("Lookup ID by name");
        this.MI_LOOKUPBASE = new JMenuItem("Lookup base object/npc");

        this.MI_SHOWLOG = new JMenuItem("Show Log", KeyEvent.VK_S);
        this.MI_ABOUT = new JMenuItem("About", KeyEvent.VK_A);

        this.CLEAR_FILTER = new JButton("Clear Filters");
        this.LOGWINDOW = new LogWindow();
        this.LBLMEMORY = new MemoryLabel();
        this.LBLSAVEINFO = new JLabel();
        this.FILTERTIMER = new java.util.Timer();

        this.initComponents(saveFile);

        TIMER.stop();
        LOG.info(String.format("Version: %s", this.getVersion()));
        LOG.info(String.format("SaveWindow constructed; took %s.", this.TIMER.getFormattedTime()));
    }

    /**
     * Initialize the swing and AWT components.
     *
     * @param saveFile
     */
    private void initComponents(File saveFile) {
        this.TREE.addTreeSelectionListener(e -> updateContextInformation());

        this.DATASCROLLER.setBorder(BorderFactory.createTitledBorder(this.DATASCROLLER.getBorder(), "Data"));

        this.INFOPANE.setFont(this.INFOPANE.getFont().deriveFont(12.0f));
        this.INFOPANE.setEditable(false);
        this.INFOPANE.setContentType("text/html");
        this.INFOPANE.addHyperlinkListener(e -> {
            if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
                final String URL = e.getDescription();
                LOG.info(String.format("HYPERLINK: %s", URL));
                SaveWindow.this.findElement(URL);
            }
        });

        this.INFOSCROLLER.setBorder(BorderFactory.createTitledBorder(this.INFOSCROLLER.getBorder(), "Information"));

        this.RIGHTSPLITTER.setResizeWeight(0.5);
        this.MAINSPLITTER.setResizeWeight(0.5);

        this.MAINPANEL.setMinimumSize(new Dimension(350, 400));

        this.PLUGINCOMBO.setRenderer(new PluginListCellRenderer());
        this.PLUGINCOMBO.setPrototypeDisplayValue(Plugin.PROTOTYPE);
        this.PLUGINCOMBO.setToolTipText("Select a plugin for filtering.");

        this.PROGRESSPANEL.add(this.LBLMEMORY);
        this.PROGRESSPANEL.add(this.LBLSAVEINFO);
        this.PROGRESSPANEL.add(this.PROGRESSBAR);
        this.PROGRESSBAR.setVisible(false);
        this.PROGRESSBAR.setEnabled(false);

        this.TREESCROLLER.getViewport().putClientProperty("EnableWindowBlit", Boolean.TRUE);
        this.FILTERFIELD.setToolTipText("Enter a regular expression for filtering.");
        this.FILTERPANEL.add(this.FILTERFIELD);
        this.FILTERPANEL.add(this.PLUGINCOMBO);
        this.FILTERPANEL.add(this.CLEAR_FILTER);

        this.MODPANEL.add(this.MODLABEL);
        this.MODPANEL.add(this.MODCOMBO);
        this.MODPANEL.setVisible(false);
        this.MODCOMBO.setRenderer(new ModListCellRenderer());
        this.MODCOMBO.setToolTipText("Select a mod for filtering.");

        this.TOPPANEL.setLayout(new BoxLayout(this.TOPPANEL, BoxLayout.Y_AXIS));
        this.TOPPANEL.add(this.MODPANEL);
        this.TOPPANEL.add(this.FILTERPANEL);

        this.MAINPANEL.add(this.TREESCROLLER, BorderLayout.CENTER);
        this.MAINPANEL.add(this.TOPPANEL, BorderLayout.PAGE_START);
        this.MAINPANEL.add(this.PROGRESSPANEL, BorderLayout.PAGE_END);

        this.FILEMENU.add(this.MI_LOAD);
        this.FILEMENU.add(this.MI_SAVE);
        this.FILEMENU.add(this.MI_SAVEAS);
        this.FILEMENU.addSeparator();
        this.FILEMENU.add(this.MI_LOADESPS);
        this.FILEMENU.addSeparator();
        this.FILEMENU.add(this.MI_EXPORTPLUGINS);
        this.FILEMENU.addSeparator();
        this.FILEMENU.add(this.MI_EXIT);
        this.FILEMENU.setMnemonic('f');

        this.CLEANMENU.add(this.MI_SHOWUNATTACHED);
        this.CLEANMENU.add(this.MI_SHOWUNDEFINED);
        this.CLEANMENU.add(this.MI_SHOWNULLREFS);
        this.CLEANMENU.add(this.MI_SHOWNONEXISTENTCREATED);
        this.CLEANMENU.addSeparator();
        this.CLEANMENU.add(this.MI_REMOVEUNATTACHED);
        this.CLEANMENU.add(this.MI_REMOVEUNDEFINED);
        this.CLEANMENU.add(this.MI_RESETHAVOK);
        this.CLEANMENU.add(this.MI_CLEANSEFORMLISTS);
        this.CLEANMENU.add(this.MI_REMOVENONEXISTENT);
        this.CLEANMENU.addSeparator();
        this.CLEANMENU.add(this.MI_BATCHCLEAN);
        this.CLEANMENU.add(this.MI_KILL);
        this.CLEANMENU.setMnemonic('c');

        this.OPTIONSMENU.add(this.MI_USEMO);
        this.OPTIONSMENU.add(this.MI_USENMM);
        this.OPTIONSMENU.add(this.MI_SHOWMODS);
        this.OPTIONSMENU.setMnemonic('o');

        this.DATAMENU.add(this.MI_LOOKUPID);
        this.DATAMENU.add(this.MI_LOOKUPBASE);
        this.DATAMENU.setMnemonic('d');
        this.DATAMENU.setEnabled(false);
        this.MI_LOOKUPID.setEnabled(false);
        this.MI_LOOKUPBASE.setEnabled(false);

        this.HELPMENU.add(this.MI_SHOWLOG);
        this.HELPMENU.add(this.MI_ABOUT);
        this.HELPMENU.setMnemonic('h');

        this.MENUBAR.add(this.FILEMENU);
        this.MENUBAR.add(this.CLEANMENU);
        this.MENUBAR.add(this.OPTIONSMENU);
        this.MENUBAR.add(this.DATAMENU);
        this.MENUBAR.add(this.HELPMENU);

        this.MI_EXIT.addActionListener(this);
        this.MI_LOAD.addActionListener(this);
        this.MI_LOADESPS.addActionListener(this);
        this.MI_SAVE.addActionListener(this);
        this.MI_SAVEAS.addActionListener(this);
        this.MI_EXPORTPLUGINS.addActionListener(this);
        this.MI_SHOWUNATTACHED.addActionListener(this);
        this.MI_SHOWUNDEFINED.addActionListener(this);
        this.MI_SHOWNULLREFS.addActionListener(this);
        this.MI_SHOWNONEXISTENTCREATED.addActionListener(this);
        this.MI_REMOVEUNATTACHED.addActionListener(this);
        this.MI_REMOVEUNDEFINED.addActionListener(this);
        this.MI_RESETHAVOK.addActionListener(this);
        this.MI_CLEANSEFORMLISTS.addActionListener(this);
        this.MI_REMOVENONEXISTENT.addActionListener(this);
        this.MI_BATCHCLEAN.addActionListener(this);
        this.MI_KILL.addActionListener(this);
        this.MI_USEMO.addActionListener(this);
        this.MI_USENMM.addActionListener(this);
        this.MI_SHOWMODS.addActionListener(this);
        this.MI_LOOKUPID.addActionListener(this);
        this.MI_LOOKUPBASE.addActionListener(this);
        this.MI_SHOWLOG.addActionListener(this);
        this.MI_ABOUT.addActionListener(this);

        this.MI_EXIT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E, KeyEvent.CTRL_DOWN_MASK));
        this.MI_LOAD.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, KeyEvent.CTRL_DOWN_MASK));
        this.MI_LOADESPS.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, KeyEvent.CTRL_DOWN_MASK));
        this.MI_SAVE.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, KeyEvent.CTRL_DOWN_MASK));
        this.MI_SAVEAS.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, KeyEvent.CTRL_DOWN_MASK));
        this.MI_REMOVEUNATTACHED.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_1, KeyEvent.CTRL_DOWN_MASK));
        this.MI_REMOVEUNDEFINED.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_2, KeyEvent.CTRL_DOWN_MASK));
        this.MI_RESETHAVOK.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_3, KeyEvent.CTRL_DOWN_MASK));
        this.MI_CLEANSEFORMLISTS.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_4, KeyEvent.CTRL_DOWN_MASK));
        this.MI_REMOVENONEXISTENT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_5, KeyEvent.CTRL_DOWN_MASK));
        this.MI_BATCHCLEAN.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_6, KeyEvent.CTRL_DOWN_MASK));

        this.CLEAR_FILTER.addActionListener(this);

        LOG.getParent().addHandler(this.LOGWINDOW.getHandler());

        final String TITLE = String.format("ReSaver %s: (no save loaded)", this.getVersion());
        this.setTitle(TITLE);
        this.setJMenuBar(this.MENUBAR);
        this.setContentPane(this.MAINSPLITTER);
        this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        this.setPreferredSize(new java.awt.Dimension(800, 600));
        this.pack();
        this.setLocationRelativeTo(null);
        SaveWindow.this.loadPreferences();

        this.MI_SHOWNONEXISTENTCREATED.addItemListener(e -> {
            if (e.getStateChange() == ItemEvent.SELECTED) {
                final String WARN = "Non-existent forms are used intentionally by some mods. Use caution when deleting them.";
                final String WARN_TITLE = "Warning";
                JOptionPane.showMessageDialog(this, WARN, WARN_TITLE, JOptionPane.WARNING_MESSAGE);
            }
        });

        this.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent evt) {
                SaveWindow.this.exitWithPrompt();
            }

            @Override
            public void windowOpened(WindowEvent evt) {
                if (null != saveFile) {
                    SaveWindow.this.open(saveFile);
                } else if (null == SaveWindow.this.save) {
                    SaveWindow.this.open(false);
                }
                SaveWindow.this.loadPreferences();
            }
        });

        try {
            final java.io.InputStream INPUT = ModChooser.class.getResourceAsStream("Disk.png");
            final Image ICON = javax.imageio.ImageIO.read(INPUT);
            super.setIconImage(ICON);
        } catch (IOException ex) {
        }

        this.MODCOMBO.addItemListener(e -> updateFilters());
        this.PLUGINCOMBO.addItemListener(e -> updateFilters());
        this.TABLE.setQueryHandler(var -> findElement(var));
        this.TREE.setDeleteHandler(paths -> deletePaths(paths));
        this.TREE.setDeleteFormsHandler(plugin -> deleteForms(plugin));
        this.TREE.setDeleteInstancesHandler(plugin -> deleteInstances(plugin));
        this.TREE.setFilterPluginsHandler(plugin -> PLUGINCOMBO.setSelectedItem(plugin));
        this.TREE.setZeroThreadHandler(threads -> zeroThreads(threads));
        this.TREE.setFindHandler(element -> this.findElement(element));
        this.TREE.setCleanseFLSTHandler(flst -> this.cleanseFormList(flst));
        final int DELAY = 700;
        final restringer.Timer DELAYTRACKER = new restringer.Timer("Delayer");

        this.FILTERFIELD.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void insertUpdate(DocumentEvent evt) {
                DELAYTRACKER.restart();
                final java.util.TimerTask FILTERTASK = new java.util.TimerTask() {
                    @Override
                    public void run() {
                        long elaspsed = DELAYTRACKER.getElapsed() / 900000;
                        if (elaspsed >= DELAY) {
                            updateFilters();
                        }
                    }
                };
                FILTERTIMER.schedule(FILTERTASK, DELAY);
            }

            @Override
            public void removeUpdate(DocumentEvent evt) {
                DELAYTRACKER.restart();
                final java.util.TimerTask FILTERTASK = new java.util.TimerTask() {
                    @Override
                    public void run() {
                        long elaspsed = DELAYTRACKER.getElapsed() / 900000;
                        if (elaspsed >= DELAY) {
                            updateFilters();
                        }
                    }
                };
                FILTERTIMER.schedule(FILTERTASK, DELAY);
            }

            @Override
            public void changedUpdate(DocumentEvent evt) {
                DELAYTRACKER.restart();
                final java.util.TimerTask FILTERTASK = new java.util.TimerTask() {
                    @Override
                    public void run() {
                        long elaspsed = DELAYTRACKER.getElapsed() / 900000;
                        if (elaspsed >= DELAY) {
                            updateFilters();
                        }
                    }
                };
                FILTERTIMER.schedule(FILTERTASK, DELAY);
            }
        });

        this.LBLMEMORY.initialize();
    }

    /**
     * Retrieves the version string from the Jar file, if possible.
     *
     * @return
     */
    private String getVersion() {
        try {
            java.net.URL res = SaveWindow.class.getResource(SaveWindow.class.getSimpleName() + ".class");
            java.net.JarURLConnection conn = (JarURLConnection) res.openConnection();
            java.util.jar.Manifest manifest = conn.getManifest();
            java.util.jar.Attributes attr = manifest.getMainAttributes();
            return attr.getValue("Implementation-Version");
        } catch (IOException | ClassCastException ex) {
            return "(dev)";
        }
    }

    /**
     * Produces a version string.
     *
     * @return
     */
    private String getInfoText() {
        try {
            java.net.URL res = SaveWindow.class.getResource(SaveWindow.class.getSimpleName() + ".class");
            java.net.JarURLConnection conn = (JarURLConnection) res.openConnection();
            java.util.jar.Manifest manifest = conn.getManifest();
            java.util.jar.Attributes attr = manifest.getMainAttributes();

            final StringBuilder BUF = new StringBuilder();
            BUF.append("ReSaver was developed for YOU personally. I hope you enjoy it.\n");
            BUF.append("\nBuilt on: ");
            BUF.append(attr.getValue("Built-Date"));
            BUF.append("\nVersion: ");
            BUF.append(attr.getValue("Implementation-Version"));
            BUF.append("\nCopyright Mark Fairchild 2016.");
            BUF.append("\nDistributed under the Apache 2.0 license.");
            return BUF.toString();
        } catch (IOException | ClassCastException ex) {
            final StringBuilder BUF = new StringBuilder();
            BUF.append("ReSaver was developed for YOU personally. I hope you enjoy it.\n");
            BUF.append("\n(development version)");
            BUF.append("\nCopyright Mark Fairchild 2016.");
            BUF.append("\nDistributed under the Apache 2.0 license.");
            return BUF.toString();
        }
    }

    /**
     * Clears the modified flag.
     */
    void clearModified() {
        this.modified = false;
    }

    /**
     * Sets the <code>Analysis</code>.
     *
     * @param newAnalysis The mod data, or null if there is no mod data
     * available.
     *
     */
    void setAnalysis(Analysis newAnalysis) {
        if (newAnalysis != this.analysis) {
            this.analysis = newAnalysis;
            this.updateContextInformation();
            this.save.addNames(analysis);
        }

        if (null != this.analysis) {
            this.DATAMENU.setEnabled(true);
            this.MI_LOOKUPID.setEnabled(true);
            this.MI_LOOKUPBASE.setEnabled(true);
        } else {
            this.DATAMENU.setEnabled(false);
            this.MI_LOOKUPID.setEnabled(false);
            this.MI_LOOKUPBASE.setEnabled(false);
        }

        if (null == this.analysis || !this.MI_SHOWMODS.isSelected()) {
            this.MODCOMBO.setModel(new DefaultComboBoxModel<>());
            this.MODPANEL.setVisible(false);

        } else {
            final Mod[] MODS = new Mod[this.analysis.MODS.size()];
            this.analysis.MODS.toArray(MODS);
            Arrays.sort(MODS, (a, b) -> a.getName().compareToIgnoreCase(b.getName()));

            DefaultComboBoxModel<Mod> modModel = new DefaultComboBoxModel<>(MODS);
            modModel.insertElementAt(null, 0);
            this.MODCOMBO.setModel(modModel);
            this.MODCOMBO.setSelectedIndex(0);
            this.MODPANEL.setVisible(true);
        }

        this.rebuildTree(true);
    }

    /**
     * Regenerates the treeview if the underlying model has changed.
     *
     * @param restorePath A flag indicating whether or not to try to restore the
     * current path.
     */
    private void rebuildTree(boolean restorePath) {
        if (null == this.save) {
            return;
        }

        LOG.info("================");
        LOG.info("Re-initializing treeview.");
        TIMER.restart();

        if (restorePath) {
            TreePath[] paths = SaveWindow.this.TREE.getSelectionPaths();
            SaveWindow.this.TREE.setESS(SaveWindow.this.save, SaveWindow.this.filter);

            if (null != paths) {
                for (int i = 0; i < paths.length; i++) {
                    paths[i] = SaveWindow.this.TREE.getModel().rebuildPath(paths[i]);
                }
            }

            LOG.info(String.format("Restoring path: ", Arrays.toString(paths)));
            SaveWindow.this.TREE.setSelectionPaths(paths);

        } else {
            SaveWindow.this.TREE.setESS(SaveWindow.this.save, SaveWindow.this.filter);
        }

        TIMER.stop();
        LOG.info(String.format("Treeview initialized, took %s.", TIMER.getFormattedTime()));

    }

    /**
     * Clears the <code>ESS</code>.
     */
    void clearESS() {
        this.MI_SAVE.setEnabled(false);
        this.MI_EXPORTPLUGINS.setEnabled(false);
        this.MI_REMOVEUNATTACHED.setEnabled(false);
        this.MI_REMOVEUNDEFINED.setEnabled(false);
        this.MI_RESETHAVOK.setEnabled(false);
        this.MI_CLEANSEFORMLISTS.setEnabled(false);
        this.MI_REMOVENONEXISTENT.setEnabled(false);
        this.PLUGINCOMBO.setModel(new DefaultComboBoxModel<>());

        this.save = null;
        this.clearModified();
        this.clearContextInformation();

        final String TITLE = String.format("ReSaver %s: (no save loaded)", this.getVersion());
        this.setTitle(TITLE);
    }

    /**
     * Sets the <code>ESS</code> containing the papyrus section to display.
     *
     * @param saveFile The file that contains the <code>ESS</code>.
     * @param newSave The new <code>ESS</code>.
     *
     */
    void setESS(File saveFile, ESS newSave) {
        Objects.requireNonNull(saveFile);
        Objects.requireNonNull(newSave);

        if (newSave.getPapyrus().getStringTable().isSTBCorrection()) {
            this.MI_SAVE.setEnabled(false);
            this.MI_SAVEAS.setEnabled(false);
        } else {
            this.MI_SAVE.setEnabled(true);
            this.MI_SAVEAS.setEnabled(true);
        }

        this.MI_EXPORTPLUGINS.setEnabled(true);
        this.MI_REMOVEUNATTACHED.setEnabled(true);
        this.MI_REMOVEUNDEFINED.setEnabled(true);
        this.MI_RESETHAVOK.setEnabled(true);
        this.MI_CLEANSEFORMLISTS.setEnabled(true);
        this.MI_REMOVENONEXISTENT.setEnabled(true);

        this.save = newSave;
        this.clearContextInformation();

        LOG.info("================");
        LOG.info("Initializing treeview.");
        TIMER.restart();

        final java.util.List<Plugin> PLUGINS = newSave.getPluginInfo().getPlugins();
        final Plugin[] PLUGINS_ARR = PLUGINS.toArray(new Plugin[0]);
        Arrays.sort(PLUGINS_ARR);
        DefaultComboBoxModel<Plugin> pluginModel = new DefaultComboBoxModel<>(PLUGINS_ARR);
        pluginModel.insertElementAt(null, 0);

        if (null != this.PLUGINCOMBO.getSelectedItem() && this.PLUGINCOMBO.getSelectedItem() instanceof Plugin) {
            final Plugin PREV = (Plugin) this.PLUGINCOMBO.getSelectedItem();
            this.PLUGINCOMBO.setModel(pluginModel);
            this.PLUGINCOMBO.setSelectedItem(PREV);

        } else {
            this.PLUGINCOMBO.setModel(pluginModel);
            this.PLUGINCOMBO.setSelectedIndex(0);
        }

        this.TREE.setESS(this.save, this.filter);
        this.TREE.updateUI();
        javax.swing.tree.TreePath path = this.TREE.getModel().getRoot().getPath();
        this.TREE.setSelectionPath(path);

        final float SIZE = (float) saveFile.length() / 1048576.0f;
        final Long DIGEST = newSave.getDigest();

        String fullName = saveFile.getName();
        final int MAXLEN = 80;
        final String NAME = (fullName.length() > MAXLEN
                ? (fullName.substring(0, MAXLEN) + "...")
                : fullName);

        final String TITLE = String.format("ReSaver %s: %s (%1.2f mb, dig=%08x)", this.getVersion(), NAME, SIZE, DIGEST);
        this.setTitle(TITLE);

        TIMER.stop();
        LOG.info(String.format("Treeview initialized, took %s.", TIMER.getFormattedTime()));
    }

    /**
     * Clears all filtering.
     */
    private void clearFilter() {
        LOG.info("Clearing filter.");
        this.MI_SHOWNONEXISTENTCREATED.setSelected(false);
        this.MI_SHOWNULLREFS.setSelected(false);
        this.MI_SHOWUNDEFINED.setSelected(false);
        this.MI_SHOWUNATTACHED.setSelected(false);
        this.FILTERFIELD.setText("");
        this.MODCOMBO.setSelectedItem(null);
        this.PLUGINCOMBO.setSelectedItem(null);
        this.filter = null;
        this.TREE.getModel().defilter();
    }

    /**
     * Updates the filter.
     *
     * @param model The model to which the filters should be applied.
     */
    private boolean createFilter(FilterTreeModel model) {
        LOG.info("Updating filter.");
        final Mod MOD = this.MODCOMBO.getItemAt(this.MODCOMBO.getSelectedIndex());
        final Plugin PLUGIN = (Plugin) this.PLUGINCOMBO.getSelectedItem();
        final String TXT = this.FILTERFIELD.getText();
        final PluginInfo PLUGINS = (null != this.save ? this.save.getPluginInfo() : null);

        Predicate<Node> mainfilter = FilterMaker.createFilter(MOD, PLUGIN, TXT, PLUGINS, this.analysis,
                this.MI_SHOWUNDEFINED.isSelected(),
                this.MI_SHOWUNATTACHED.isSelected(),
                this.MI_SHOWNULLREFS.isSelected(),
                this.MI_SHOWNONEXISTENTCREATED.isSelected());

        if (null == mainfilter) {
            this.filter = null;
            model.defilter();
            return true;
        } else {
            this.filter = mainfilter;
            model.filter(this.filter);
            return true;
        }
    }

    /**
     * Updates the filter.
     *
     */
    private void updateFilters() {
        ProgressIndicator.startWaitCursor(this.getRootPane());
        SwingUtilities.invokeLater(() -> {
            TIMER.restart();

            TreePath path = this.TREE.getSelectionPath();
            boolean result = this.createFilter(this.TREE.getModel());

            if (!result) {
                return;
            }

            this.TREE.updateUI();

            if (null != path) {
                LOG.info(String.format("Updating filter: restoring path = %s", path.toString()));

                Object o = path.getLastPathComponent();

                if (null == o) {
                    this.TREE.clearSelection();
                    this.clearContextInformation();
                } else {
                    Node node = (Node) o;
                    if (!node.isFiltered()) {
                        this.TREE.setSelectionPath(path);
                        this.TREE.scrollPathToVisible(path);
                    } else {
                        this.TREE.clearSelection();
                        this.clearContextInformation();
                    }
                }
            }

            TIMER.stop();
            LOG.info(String.format("Filter updated, took %s.", TIMER.getFormattedTime()));
            ProgressIndicator.stopWaitCursor(this.getRootPane());
        });
    }

    /**
     *
     * @param model
     */
    void startProgressBar(BoundedRangeModel model) {
        this.PROGRESSBAR.setModel(model);
        this.PROGRESSBAR.setVisible(true);
        this.PROGRESSBAR.setEnabled(true);
    }

    /**
     *
     */
    void clearProgressBar() {
        this.PROGRESSBAR.setVisible(false);
        this.PROGRESSBAR.setEnabled(false);
    }

    /**
     * Exits the application immediately.
     */
    void exit() {
        javax.swing.SwingUtilities.invokeLater(() -> {
            SaveWindow.this.savePreferences();
            this.FILTERTIMER.cancel();
            this.FILTERTIMER.purge();
            this.LBLMEMORY.terminate();
            this.setVisible(false);
            this.dispose();
            Platform.exit();
        });
    }

    /**
     * Exits the application after checking if the user wishes to save.
     */
    void exitWithPrompt() {
        if (null != this.save && this.modified) {
            int result = JOptionPane.showConfirmDialog(this, "Do you want to save the current file first?", "Save First?", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);

            switch (result) {
                case JOptionPane.CANCEL_OPTION:
                    return;

                case JOptionPane.YES_OPTION:
                    this.save(false, false, true);
                    break;

                case JOptionPane.NO_OPTION:
                    this.exit();
                    break;
            }
        }

        this.exit();
    }

    /**
     * Saves the currently loaded save, if any.
     *
     * @param promptForFile A flag indicating the the user should be asked what
     * filename to use.
     * @param promptToOpen A flag indicating that the user should be given a
     * chance to open a new file.
     * @param quitAfter A flag indicating that the application should close
     * after saving.
     */
    private void save(boolean promptForFile, boolean promptToOpen, boolean quitAfter) {
        if (null == this.save) {
            return;
        }

        try {
            final FutureTask<File> PROMPT = new FutureTask<>(() -> {
                File newSaveFile = promptForFile
                        ? Configurator.selectNewSaveFile(this, this.save.getHeader().GAME)
                        : Configurator.confirmSaveFile(this, this.save.getHeader().GAME);

                if (Configurator.validateNewSavegame(newSaveFile)) {
                    return newSaveFile;
                }
                return null;
            });

            final ModalProgressDialog MODAL = new ModalProgressDialog(this, "File Selection", PROMPT);
            MODAL.setVisible(true);
            final File SAVEFILE = PROMPT.get();

            if (!Configurator.validateNewSavegame(SAVEFILE)) {
                return;
            }

            final Saver SAVER = new Saver(this, SAVEFILE, this.save, promptToOpen, quitAfter);
            SAVER.execute();

        } catch (InterruptedException | ExecutionException ex) {
        }
    }

    /**
     * Opens a save file.
     *
     * @param saveFile The save file to read.
     */
    void open(File saveFile) {
        if (!Configurator.validateSavegame(saveFile)) {
            return;
        }

        final Opener OPENER = new Opener(this, saveFile);
        OPENER.execute();
    }

    /**
     * Opens a save file.
     *
     * @param promptToSave A flag indicating that the user should be given a
     * chance to save their file.
     */
    void open(boolean promptToSave) {
        if (null != this.save && this.modified && promptToSave) {
            int result = JOptionPane.showConfirmDialog(this, "Do you want to save the current file first?", "Save First?", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);

            switch (result) {
                case JOptionPane.YES_OPTION:
                    this.save(false, true, false);
                    return;

                case JOptionPane.CANCEL_OPTION:
                    return;

                case JOptionPane.NO_OPTION:
                    break;
            }
        }

        try {
            final FutureTask<File> PROMPT = new FutureTask<>(() -> {
                File saveFile = Configurator.selectSaveFile(SaveWindow.this);
                return saveFile;
            });

            final ModalProgressDialog MODAL = new ModalProgressDialog(this, "File Selection", PROMPT);
            MODAL.setVisible(true);
            final File SAVEFILE = PROMPT.get();

            if (!Configurator.validateSavegame(SAVEFILE)) {
                return;
            }

            final Opener OPENER = new Opener(this, SAVEFILE);
            OPENER.execute();

        } catch (InterruptedException | ExecutionException ex) {
        }
    }

    /**
     * Scans ESPs for contextual information.
     */
    private void scanESPs() {
        if (null == this.save) {
            return;
        }
        
        final Game GAME = this.save.getHeader().GAME;
        
        try {
            final FutureTask<File> PROMPT_GAMEDIR = new FutureTask<>(() -> {
                File gameDir = Configurator.getGameDirectory(GAME);
                if (!Configurator.validateGameDirectory(GAME, gameDir)) {
                    gameDir = Configurator.selectGameDirectory(this, GAME);
                }
                if (!Configurator.validateGameDirectory(GAME, gameDir)) {
                    return null;
                }
                return gameDir;
            });

            final ModalProgressDialog MODAL1 = new ModalProgressDialog(this, "File Selection", PROMPT_GAMEDIR);
            MODAL1.setVisible(true);
            final File GAME_DIR = PROMPT_GAMEDIR.get();

            final File MO_DIR, NMM_DIR;

            if (!this.MI_USEMO.isSelected()) {
                MO_DIR = null;
            } else {
                final FutureTask<File> PROMPT_MO = new FutureTask<>(() -> {
                    File moDir = Configurator.getModOrganizerDirectory();
                    if (Configurator.validateModOrganizerDirectory(moDir)) {
                        return moDir;
                    }

                    moDir = Configurator.selectModOrganizerDirectory(this);
                    if (Configurator.validateModOrganizerDirectory(moDir)) {
                        return moDir;
                    }

                    return null;
                });

                final ModalProgressDialog MODAL2 = new ModalProgressDialog(this, "File Selection", PROMPT_MO);
                MODAL2.setVisible(true);
                MO_DIR = PROMPT_MO.get();
            }

            if (!this.MI_USENMM.isSelected()) {
                NMM_DIR = null;
            } else {
                final FutureTask<File> PROMPT_NMM = new FutureTask<>(() -> {
                    /*File moDir = Configurator.getModOrganizerDirectory();
                    if (Configurator.validateModOrganizerDirectory(moDir)) {
                        return moDir;
                    }

                    moDir = Configurator.selectModOrganizerDirectory(this);
                    if (Configurator.validateModOrganizerDirectory(moDir)) {
                        return moDir;
                    }
                     */
                    return null;
                });

                final ModalProgressDialog MODAL3 = new ModalProgressDialog(this, "File Selection", PROMPT_NMM);
                MODAL3.setVisible(true);
                NMM_DIR = PROMPT_NMM.get();
            }

            final Scanner SCANNER = new Scanner(this, this.save, GAME_DIR, MO_DIR, NMM_DIR);
            SCANNER.execute();

        } catch (InterruptedException | ExecutionException ex) {
        }
    }

    /**
     * Exports a list of plugins.
     */
    private void exportPlugins() {
        try {
            final FutureTask<File> PROMPT = new FutureTask<>(() -> {
                File file = Configurator.selectPluginsExport(this, this.save.getOriginalFile());
                if (Configurator.validatePluginsExport(file)) {
                    return file;
                }
                return null;
            });

            final ModalProgressDialog MODAL = new ModalProgressDialog(this, "File Selection", PROMPT);
            MODAL.setVisible(true);
            final File EXPORT = PROMPT.get();
            if (null == EXPORT) {
                return;
            }

            try (PrintWriter out = new PrintWriter(EXPORT)) {
                this.save.getPluginInfo().getPlugins().forEach(p -> out.println(p.NAME));
                final String MSG = String.format("Plugins list exported.");
                JOptionPane.showMessageDialog(SaveWindow.this, MSG, "Success", JOptionPane.INFORMATION_MESSAGE);

            } catch (IOException ex) {
                final String MSG = String.format("Error while writing file \"%s\".\n%s", EXPORT.getName(), ex.getMessage());
                LOG.severe(MSG);
                LOG.severe(ex.toString());
                System.err.println(ex.getMessage());
                ex.printStackTrace(System.err);
                JOptionPane.showMessageDialog(SaveWindow.this, MSG, "Write Error", JOptionPane.ERROR_MESSAGE);
            }

        } catch (InterruptedException | ExecutionException ex) {
        }
    }

    /**
     * Prompts the user for a name, and finds the corresponding ID.
     */
    private void lookupID() {
        final String MSG = "Enter the name of the object or NPC:";
        final String TITLE = "Enter Name";

        String input = JOptionPane.showInputDialog(this, MSG, TITLE, JOptionPane.QUESTION_MESSAGE);
        if (null == input || input.trim().isEmpty()) {
            return;
        }

        IntSet matches = new it.unimi.dsi.fastutil.ints.IntRBTreeSet(this.analysis.IDS.getID(input, this.analysis.STRINGS));
        if (matches.isEmpty()) {
            JOptionPane.showMessageDialog(this, "No matches were found.", "No matches", JOptionPane.ERROR_MESSAGE);
            return;
        }

        final StringBuilder BUF = new StringBuilder();
        BUF.append("The following matches were found:\n\n");

        matches.forEach(id -> {
            BUF.append(String.format("%08x", id));

            int pluginIndex = id >>> 24;
            final java.util.List<Plugin> PLUGINS = this.save.getPluginInfo().getPlugins();

            if (0 <= pluginIndex && pluginIndex < PLUGINS.size()) {
                final Plugin PLUGIN = PLUGINS.get(pluginIndex);
                BUF.append(" (").append(PLUGIN).append(")");
            }
            BUF.append('\n');
        });

        JOptionPane.showMessageDialog(this, BUF.toString(), "Matches", JOptionPane.INFORMATION_MESSAGE);
        System.out.println(matches);
    }

    /**
     * Prompts the user for the name or ID of a reference, and finds the id/name
     * of the base object.
     */
    private void lookupBase() {

    }

    /**
     * Removes unattached script instances (instances with no valid Ref).
     *
     */
    private void cleanUnattached() {
        try {
            if (null == this.save) {
                return;
            }

            LOG.info("Cleaning unattached instances.");

            Papyrus papyrus = this.save.getPapyrus();
            int result = papyrus.cleanUnattachedInstances();

            String msg = String.format("Removed %d orphaned script instances.", result);
            LOG.info(msg);
            JOptionPane.showMessageDialog(this, msg, "Cleaned", JOptionPane.INFORMATION_MESSAGE);

            if (result > 0) {
                this.rebuildTree(true);
                this.modified = true;
            }

        } catch (Exception ex) {
            final String MSG = "Error cleaning unattached scripts.";
            final String TITLE = "Cleaning Error";
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(SaveWindow.this, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Remove undefined script instances (instances with no Script).
     */
    private void cleanUndefined() {
        try {
            if (null == this.save) {
                return;
            }

            LOG.info("Cleaning undefined elements.");

            Papyrus papyrus = this.save.getPapyrus();
            int[] result = papyrus.cleanUndefinedElements();

            final StringBuilder BUF = new StringBuilder();
            if (result[0] > 0) {
                BUF.append("Removed ").append(result[0]).append(" undefined elements.");
            }
            if (result[1] > 0) {
                BUF.append("Terminated ").append(result[1]).append(" undefined threads.");
            }
            final String MSG = BUF.toString();
            LOG.info(MSG);
            JOptionPane.showMessageDialog(this, MSG, "Cleaned", JOptionPane.INFORMATION_MESSAGE);

            if (result[0] > 0) {
                this.rebuildTree(true);
                this.modified = true;
            }

        } catch (Exception ex) {
            final String MSG = "Error cleaning undefined elements.";
            final String TITLE = "Cleaning Error";
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(SaveWindow.this, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     *
     */
    private void resetHavok() {
        if (null != this.save) {
            this.save.resetHavok();
            JOptionPane.showMessageDialog(this, "Not implemented yet.");
            this.modified = true;
        }
    }

    /**
     *
     */
    private void cleanseFormLists() {
        try {
            if (null == this.save) {
                return;
            }

            LOG.info("Cleansing formlists.");
            int[] results = this.save.cleanseFormLists();

            if (results[0] == 0) {
                final String MSG = "No nullrefs were found in any formlists.";
                final String TITLE = "No nullrefs found.";
                LOG.info(MSG);
                JOptionPane.showMessageDialog(this, MSG, TITLE, JOptionPane.INFORMATION_MESSAGE);

            } else {
                this.modified = true;
                final String MSG = String.format("%d nullrefs were cleansed from %d formlists.", results[0], results[1]);
                final String TITLE = "Nullrefs cleansed.";
                LOG.info(MSG);
                JOptionPane.showMessageDialog(this, MSG, TITLE, JOptionPane.INFORMATION_MESSAGE);
            }

            this.rebuildTree(true);

        } catch (Exception ex) {
            final String MSG = "Error cleansing formlists.";
            final String TITLE = "Cleansing Error";
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(SaveWindow.this, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     *
     * @param flst
     */
    private void cleanseFormList(ChangeFormFLST flst) {
        try {
            if (null == this.save) {
                return;
            }

            LOG.info(String.format("Cleansing formlist %s.", flst));
            int result = flst.cleanse();

            if (result == 0) {
                final String MSG = "No nullrefs were found.";
                final String TITLE = "No nullrefs found.";
                LOG.info(MSG);
                JOptionPane.showMessageDialog(this, MSG, TITLE, JOptionPane.INFORMATION_MESSAGE);
            } else {
                this.modified = true;
                final String MSG = String.format("%d nullrefs were cleansed.", result);
                final String TITLE = "Nullrefs cleansed.";
                LOG.info(MSG);
                JOptionPane.showMessageDialog(this, MSG, TITLE, JOptionPane.INFORMATION_MESSAGE);
            }

            this.rebuildTree(true);

        } catch (Exception ex) {
            final String MSG = "Error cleansing formlists.";
            final String TITLE = "Cleansing Error";
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(SaveWindow.this, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Removes script instances attached to nonexistent created forms.
     */
    private void cleanNonexistent() {
        // Check with the user first. This operation can mess up mods.
        final String WARN = "This cleaning operation can cause some mods to stop working. Are you sure you want to do this?";
        final String WARN_TITLE = "Warning";
        int confirm = JOptionPane.showConfirmDialog(this, WARN, WARN_TITLE, JOptionPane.YES_NO_OPTION);
        if (confirm != JOptionPane.YES_OPTION) {
            return;
        }

        try {
            if (null == this.save) {
                return;
            }

            LOG.info(String.format("Removing nonexistent created forms."));
            int result = this.save.removeNonexistentCreated();

            if (result == 0) {
                final String MSG = "No scripts attached to non-existent created forms were found.";
                final String TITLE = "No non-existent created";
                LOG.info(MSG);
                JOptionPane.showMessageDialog(this, MSG, TITLE, JOptionPane.INFORMATION_MESSAGE);
            } else {
                this.modified = true;
                final String MSG = String.format("%d instances were removed.", result);
                final String TITLE = "Instances removed.";
                LOG.info(MSG);
                JOptionPane.showMessageDialog(this, MSG, TITLE, JOptionPane.INFORMATION_MESSAGE);
            }

            this.rebuildTree(true);

        } catch (Exception ex) {
            final String MSG = "Error cleansing non-existent created.";
            final String TITLE = "Cleansing Error";
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(SaveWindow.this, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
        }

    }

    /**
     * Save minor settings like window position, size, and state.
     */
    private void savePreferences() {
        final java.util.prefs.Preferences PREFS = java.util.prefs.Preferences.userNodeForPackage(SaveWindow.class);
        PREFS.putInt("save.extendedState", this.getExtendedState());

        if (this.getExtendedState() == JFrame.NORMAL) {
            PREFS.putInt("save.windowWidth", this.getSize().width);
            PREFS.putInt("save.windowHeight", this.getSize().height);
            PREFS.putInt("save.windowX", this.getLocation().x);
            PREFS.putInt("save.windowY", this.getLocation().y);
            PREFS.putInt("save.mainDivider", this.MAINSPLITTER.getDividerLocation());
            PREFS.putInt("save.rightDivider", this.RIGHTSPLITTER.getDividerLocation());
            System.out.printf("Pos = %s\n", this.getLocation());
            System.out.printf("Size = %s\n", this.getSize());
            System.out.printf("Dividers = %d,%d\n", this.MAINSPLITTER.getDividerLocation(), this.RIGHTSPLITTER.getDividerLocation());
        } else {
            PREFS.putInt("save.mainDividerMax", this.MAINSPLITTER.getDividerLocation());
            PREFS.putInt("save.rightDividerMax", this.RIGHTSPLITTER.getDividerLocation());
            try {
                PREFS.flush();
            } catch (BackingStoreException ex) {
                ex.printStackTrace(System.err);
            }
        }

        PREFS.put("save.regex", this.FILTERFIELD.getText());
        PREFS.putBoolean("save.useMO", this.MI_USEMO.isSelected());
        PREFS.putBoolean("save.useNMM", this.MI_USENMM.isSelected());
        PREFS.putBoolean("save.showMods", this.MI_SHOWMODS.isSelected());
    }

    /**
     * Loads minor settings like window position, size, and state.
     */
    private void loadPreferences() {
        final java.util.prefs.Preferences PREFS = java.util.prefs.Preferences.userNodeForPackage(SaveWindow.class);

        int state = PREFS.getInt("save.extendedState", JFrame.MAXIMIZED_BOTH);
        this.setExtendedState(state);

        if (state == JFrame.NORMAL) {
            java.awt.Point pos = this.getLocation();
            java.awt.Dimension size = this.getSize();
            int x = PREFS.getInt("save.windowX", pos.x);
            int y = PREFS.getInt("save.windowY", pos.y);
            int width = PREFS.getInt("save.windowWidth", size.width);
            int height = PREFS.getInt("save.windowHeight", size.height);
            this.setLocation(x, y);
            this.setSize(width, height);
            int mainDivider = this.MAINSPLITTER.getDividerLocation();
            mainDivider = PREFS.getInt("save.mainDivider", mainDivider);
            int rightDivider = this.RIGHTSPLITTER.getDividerLocation();
            rightDivider = PREFS.getInt("save.rightDivider", rightDivider);
            this.MAINSPLITTER.setDividerLocation(mainDivider);
            this.RIGHTSPLITTER.setDividerLocation(rightDivider);
        } else {
            int mainDivider = this.MAINSPLITTER.getDividerLocation();
            mainDivider = PREFS.getInt("save.mainDividerMax", mainDivider);
            int rightDivider = this.RIGHTSPLITTER.getDividerLocation();
            rightDivider = PREFS.getInt("save.rightDividerMax", rightDivider);
            this.MAINSPLITTER.setDividerLocation(mainDivider);
            this.RIGHTSPLITTER.setDividerLocation(rightDivider);
        }

        this.MI_USEMO.setSelected(PREFS.getBoolean("save.useMO", false));
        this.MI_USENMM.setSelected(PREFS.getBoolean("save.useNMM", false));
        this.MI_SHOWMODS.setSelected(PREFS.getBoolean("save.showMods", false));
        this.FILTERFIELD.setText(PREFS.get("save.regex", ""));
        this.updateFilters();
    }

    /**
     * @param batch The list of cleaning operations. If null, prompt for the
     * list.
     */
    private void batchClean(String batch) {
        if (null == batch) {
            final JTextArea TEXT = new JTextArea();
            TEXT.setColumns(50);
            TEXT.setRows(10);
            TEXT.setLineWrap(false);
            TEXT.setWrapStyleWord(false);

            final JScrollPane SCROLLER = new JScrollPane(TEXT);
            SCROLLER.setBorder(BorderFactory.createTitledBorder("Enter Scripts"));
            final String TITLE = "Batch Clean";
            int result = JOptionPane.showConfirmDialog(this, SCROLLER, TITLE, JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);

            if (result == JOptionPane.CANCEL_OPTION) {
                return;
            }

            batch = TEXT.getText();
        }

        if (null == batch || batch.isEmpty()) {
            return;
        }

        final String PATTERN = "^([^\\.@\\s]+)(?:\\.pex)?(?:\\s*@@\\s*(.*))?";
        final Pattern REGEX = Pattern.compile(PATTERN, Pattern.CASE_INSENSITIVE);

        final String[] LINES = batch.split("\n");
        final java.util.Set<IString> CLEAN_NAMES = new java.util.TreeSet<>();
        final ScriptMap SCRIPTS = this.save.getPapyrus().getScripts();

        for (String line : LINES) {
            final Matcher MATCHER = REGEX.matcher(line);
            if (!MATCHER.find()) {
                assert false;
            }

            java.util.List<String> groups = new java.util.LinkedList<>();
            for (int i = 0; i <= MATCHER.groupCount(); i++) {
                groups.add(MATCHER.group(i));
            }
            System.out.printf("Groups = %d: %s\n", MATCHER.groupCount(), groups);

            final IString SCRIPT = IString.get(MATCHER.group(1).trim());

            if (SCRIPTS.containsKey(SCRIPT)) {
                if (null == MATCHER.group(2)) {
                    CLEAN_NAMES.add(SCRIPT);
                    LOG.info(String.format("Script present, adding to cleaning list: %s", SCRIPT));

                } else {
                    LOG.info(String.format("Script present, prompting for deletion: %s", SCRIPT));
                    final String PROMPT = MATCHER.group(2).trim();
                    final String MSG = String.format("Delete %s?\n%s", SCRIPT, PROMPT);
                    final String TITLE = "Confirm";
                    int result = JOptionPane.showConfirmDialog(this, MSG, TITLE, JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
                    if (result == JOptionPane.OK_OPTION) {
                        CLEAN_NAMES.add(SCRIPT);
                    } else if (result == JOptionPane.CANCEL_OPTION) {
                        return;
                    }
                }
            }
        }

        final StringBuilder BUF = new StringBuilder();
        BUF.append("The following scripts will be cleaned: \n\n");
        CLEAN_NAMES.forEach(v -> BUF.append(v).append('\n'));

        final JTextArea TEXT = new JTextArea(BUF.toString());
        TEXT.setColumns(40);
        TEXT.setEditable(false);
        final JScrollPane SCROLLER = new JScrollPane(TEXT);
        final String TITLE = "Batch Clean";

        int result = JOptionPane.showConfirmDialog(this, SCROLLER, TITLE, JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
        if (result == JOptionPane.NO_OPTION) {
            return;
        }

        final java.util.Set<Script> CLEAN = SCRIPTS.values()
                .stream()
                .filter(script -> CLEAN_NAMES.contains(script.getName()))
                .collect(Collectors.toSet());

        final Papyrus PAPYRUS = this.save.getPapyrus();
        long scripts = 0;
        long instances = 0;
        long references = 0;
        long threads = 0;
        this.modified = true;

        for (Script script : CLEAN) {
            assert SCRIPTS.containsValue(script);
            SCRIPTS.values().remove(script);
            scripts++;

            instances += (int) PAPYRUS.getInstances()
                    .values()
                    .stream()
                    .filter(v -> v.getScript() == script)
                    .count();

            references += (int) PAPYRUS.getReferences()
                    .values()
                    .stream()
                    .filter(v -> v.getScript() == script)
                    .count();

            threads += (int) PAPYRUS.getActiveScripts()
                    .values()
                    .stream()
                    .filter(v -> v.hasScript(script))
                    .count();

            PAPYRUS.getInstances()
                    .values()
                    .removeIf(v -> v.getScript() == script);

            PAPYRUS.getReferences()
                    .values()
                    .removeIf(v -> v.getScript() == script);

            PAPYRUS.getActiveScripts()
                    .values()
                    .stream()
                    .filter(v -> v.hasScript(script))
                    .forEach(v -> v.zero());
        }

        this.rebuildTree(true);

        final String MSG = String.format("Cleaned %d scripts, %d instances, and %d references. %d threads were terminated.", scripts, instances, references, threads);
        JOptionPane.showMessageDialog(this, MSG, TITLE, JOptionPane.INFORMATION_MESSAGE);
    }

    /**
     *
     */
    private void kill() {
        try {
            final java.util.List<Element> ELEMENTS = this.TREE.getModel().getElements();
            this.modified = true;
            this.save.removeElements(new java.util.HashSet<>(ELEMENTS));
            //elements.forEach(v -> this.save.removeElement(v));
            this.rebuildTree(false);
        } catch (Exception ex) {
            final String MSG = "Error cleansing formlists.";
            final String TITLE = "Cleansing Error";
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(SaveWindow.this, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Handles action events from menus and buttons.
     *
     * @param evt
     */
    @Override
    public void actionPerformed(ActionEvent evt) {
        if (evt.getSource() == this.MI_EXIT) {
            this.exitWithPrompt();
        } else if (evt.getSource() == this.MI_SAVE) {
            this.save(false, false, false);
        } else if (evt.getSource() == this.MI_SAVEAS) {
            this.save(true, false, false);
        } else if (evt.getSource() == this.MI_LOAD) {
            this.open(true);
        } else if (evt.getSource() == this.MI_LOADESPS) {
            this.scanESPs();
        } else if (evt.getSource() == this.MI_SHOWUNATTACHED) {
            this.updateFilters();
        } else if (evt.getSource() == this.MI_SHOWUNDEFINED) {
            this.updateFilters();
        } else if (evt.getSource() == this.MI_SHOWNULLREFS) {
            this.updateFilters();
        } else if (evt.getSource() == this.MI_SHOWNONEXISTENTCREATED) {
            this.updateFilters();
        } else if (evt.getSource() == this.MI_REMOVEUNATTACHED) {
            this.cleanUnattached();
        } else if (evt.getSource() == this.MI_REMOVEUNDEFINED) {
            this.cleanUndefined();
        } else if (evt.getSource() == this.MI_RESETHAVOK) {
            this.resetHavok();
        } else if (evt.getSource() == this.MI_CLEANSEFORMLISTS) {
            this.cleanseFormLists();
        } else if (evt.getSource() == this.MI_REMOVENONEXISTENT) {
            this.cleanNonexistent();
        } else if (evt.getSource() == this.MI_BATCHCLEAN) {
            this.batchClean(null);
        } else if (evt.getSource() == this.MI_KILL) {
            this.kill();
        } else if (evt.getSource() == this.MI_EXPORTPLUGINS) {
            this.exportPlugins();
        } else if (evt.getSource() == this.MI_SHOWMODS) {
            this.setAnalysis(this.analysis);
        } else if (evt.getSource() == this.MI_SHOWLOG) {
            this.showLog();
        } else if (evt.getSource() == this.MI_LOOKUPID) {
            this.lookupID();
        } else if (evt.getSource() == this.MI_LOOKUPBASE) {
            this.lookupBase();
        } else if (evt.getSource() == this.MI_ABOUT) {
            JOptionPane.showMessageDialog(this, this.getInfoText(), "About", JOptionPane.INFORMATION_MESSAGE);
        } else if (evt.getSource() == this.CLEAR_FILTER) {
            this.clearFilter();
        }
    }

    /**
     *
     */
    private void showLog() {
        JDialog dialog = new JDialog(this, "Log");
        dialog.setContentPane(this.LOGWINDOW);
        dialog.setModalityType(JDialog.ModalityType.MODELESS);
        dialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        dialog.setPreferredSize(new Dimension(600, 400));
        dialog.setLocationRelativeTo(null);
        dialog.pack();
        dialog.setVisible(true);
    }

    /**
     * Selects the element.
     *
     * @param element The <code>Element</code> to find.
     */
    private void findElement(String url) {
        Objects.requireNonNull(url);
        if (null == this.save) {
            return;
        }

        final Matcher MATCHER = URLREGEX.matcher(url);
        if (!MATCHER.matches()) {
            return;
        }

        final String TYPE = MATCHER.group(1).toLowerCase();
        final String LABEL = MATCHER.group(2);
        final Papyrus PAPYRUS = this.save.getPapyrus();
        final long VAL;
        final EID ID, proto;

        switch (TYPE) {
            case "script":
                this.findElement(PAPYRUS.getScripts().get(IString.get(LABEL)));
                break;
            case "structdef":
                this.findElement(PAPYRUS.getStructDefs().get(IString.get(LABEL)));
                break;
            case "instance":
                VAL = Long.parseUnsignedLong(LABEL, 16);
                proto = this.save.getPapyrus().getInstances().values().stream().findAny().get().getID();
                ID = proto.derive(VAL);
                this.findElement(PAPYRUS.getInstances().get(ID));
                break;
            case "struct":
                VAL = Long.parseUnsignedLong(LABEL, 16);
                proto = this.save.getPapyrus().getStructs().values().stream().findAny().get().getID();
                ID = proto.derive(VAL);
                this.findElement(PAPYRUS.getStructs().get(ID));
                break;
            case "reference":
                VAL = Long.parseUnsignedLong(LABEL, 16);
                proto = this.save.getPapyrus().getReferences().values().stream().findAny().get().getID();
                ID = proto.derive(VAL);
                this.findElement(PAPYRUS.getReferences().get(ID));
                break;
            case "plugin":
                java.util.List<Plugin> PLUGINS = this.save.getPluginInfo().getPlugins();
                PLUGINS.stream()
                        .filter(v -> v.NAME.equalsIgnoreCase(LABEL))
                        .findAny()
                        .ifPresent(v -> this.findElement(v));
                break;
            case "array":
                VAL = Long.parseUnsignedLong(LABEL, 16);
                proto = this.save.getPapyrus().getArrays().values().stream().findAny().get().getID();
                ID = proto.derive(VAL);
                this.findElement(PAPYRUS.getArrays().get(ID));
                break;
            case "refid":
                VAL = Long.parseUnsignedLong(LABEL, 16);
                ID = EID.make8Byte(VAL);
                final restringer.ess.RefID REFID = new restringer.ess.RefID(ID);
                this.findElement(this.save.getChangeForms().get(REFID));
                break;
            default:
        }
    }

    /**
     * Selects the element.
     *
     * @param element The <code>Element</code> to find.
     */
    private void findElement(Element element) {
        if (null == element) {
            return;
        }

        TreePath path = this.TREE.findPath(element);

        if (null == path) {
            JOptionPane.showMessageDialog(this, "The element was not found.", "Not Found", JOptionPane.ERROR_MESSAGE);
            return;
        }

        this.TREE.updateUI();
        this.TREE.scrollPathToVisible(path);
        this.TREE.setSelectionPath(path);
    }

    /**
     * Selects the element stored in a reference variable or array variable.
     *
     * @param var The <code>Variable</code> whose contents should be found.
     */
    private void findElement(Variable var) {
        Objects.requireNonNull(var);

        if (var instanceof Variable.Array) {
            Variable.Array arrayVar = (Variable.Array) var;
            final EID ID = arrayVar.getArrayID();
            if (ID.isZero()) {
                return;
            }

            ArrayInfo array = this.save.getPapyrus().getArrays().get(ID);
            assert array == arrayVar.getArray();

            if (null == array) {
                JOptionPane.showMessageDialog(this, "The array with that ID was not found.", "Not Found", JOptionPane.ERROR_MESSAGE);
                return;
            }

            this.findElement(array);

        } else if (var.hasRef()) {
            final EID ID = var.getRef();
            if (ID.isZero()) {
                return;
            }

            if (null == var.getReferent()) {
                JOptionPane.showMessageDialog(this, "The element with that ID was not found.", "Not Found", JOptionPane.ERROR_MESSAGE);
                return;
            }

            this.findElement(var.getReferent());
        }

    }

    /**
     * Deletes a plugin's script instances.
     *
     * @param plugin The <code>Plugin</code> whose instances will be deleted.
     */
    private void deleteInstances(Plugin plugin) {
        Objects.requireNonNull(plugin);

        final int ROW = this.TREE.getSelectionModel().getLeadSelectionRow();

        final String QUESTION = String.format("Are you sure you want to delete all of the script instances associated with this plugin?\n%s", plugin.NAME);
        int result = JOptionPane.showConfirmDialog(this, QUESTION, "Delete Instances", JOptionPane.YES_NO_OPTION);
        if (result == JOptionPane.NO_OPTION) {
            return;
        }

        int numInstances = plugin.getInstances().size();
        int numRemoved = this.save.removePluginInstances(plugin);
        if (numRemoved < numInstances) {
            throw new IllegalStateException(String.format("An error occurred while deleting the plugin's script instances:\n%s", plugin));
        }

        if (numRemoved > 0) {
            this.modified = true;
            this.rebuildTree(true);
        }

        final String MSG = String.format("%d Instances deleted.", numInstances);
        LOG.info(MSG);
        JOptionPane.showMessageDialog(this, MSG, "Instances Deleted", JOptionPane.INFORMATION_MESSAGE);

    }

    /**
     * Deletes a plugin's forms.
     *
     * @param plugin The <code>Plugin</code> whose forms will be deleted.
     */
    private void deleteForms(Plugin plugin) {
        Objects.requireNonNull(plugin);

        final int ROW = this.TREE.getSelectionModel().getLeadSelectionRow();

        final String QUESTION = String.format("Are you sure you want to delete all of the Forms associated with this plugin?\n%s", plugin.NAME);
        int result = JOptionPane.showConfirmDialog(this, QUESTION, "Delete Forms", JOptionPane.YES_NO_OPTION);
        if (result == JOptionPane.NO_OPTION) {
            return;
        }

        int numForms = plugin.getForms().size();
        int numRemoved = this.save.removePluginForms(plugin);

        if (numRemoved < numForms) {
            throw new IllegalStateException(String.format("An error occurred while deleting the plugin's forms:\n%s", plugin));
        }

        if (numRemoved > 0) {
            this.modified = true;
            this.rebuildTree(true);
        }

        final String MSG = String.format("%d ChangeForms deleted.", numForms);
        LOG.info(MSG);
        JOptionPane.showMessageDialog(this, MSG, "ChangeForms Deleted", JOptionPane.INFORMATION_MESSAGE);
    }

    /**
     * Zero a thread, terminating it.
     *
     * @param threads An <code>Element</code> <code>List</code> that will be
     * terminated.
     */
    private void zeroThreads(java.util.List<ActiveScript> threads) {
        Objects.requireNonNull(threads);

        switch (threads.size()) {
            case 0:
                return;

            case 1: {
                final String QUESTION = "Are you sure you want to terminate this thread?";
                final String TITLE = "Zero Thread";
                int result = JOptionPane.showConfirmDialog(this, QUESTION, TITLE, JOptionPane.YES_NO_OPTION);
                if (result == JOptionPane.NO_OPTION) {
                    return;
                }
                break;
            }

            default: {
                final String QUESTION = String.format("Are you sure you want to terminate these %d threads?", threads.size());
                final String TITLE = "Zero Threads";
                int result = JOptionPane.showConfirmDialog(this, QUESTION, TITLE, JOptionPane.YES_NO_OPTION);
                if (result == JOptionPane.NO_OPTION) {
                    return;
                }
                break;
            }
        }

        this.modified = true;
        threads.forEach(t -> t.zero());
        this.rebuildTree(true);

        final String MSG = "Thread terminated and zeroed.";
        LOG.info(MSG);
        JOptionPane.showMessageDialog(this, MSG, "Thread Zeroed", JOptionPane.INFORMATION_MESSAGE);
    }

    /**
     *
     * Deletes selected elements of the tree.
     *
     * @param elements The selections to delete.
     */
    private void deletePaths(Map<Element, Node> elements) {
        if (null == this.save || null == elements || 0 == elements.size()) {
            return;
        }

        if (elements.size() == 1) {
            final Element ELEMENT = elements.keySet().iterator().next();

            if (ESS.THREAD.test(ELEMENT)) {
                final String WARN = String.format("Element \"%s\" is a Papyrus thread. Deleting it could make your savefile impossible to load. Are you sure you want to proceed?", ELEMENT.toString());
                final String TITLE = "Warning";
                int result = JOptionPane.showConfirmDialog(this, WARN, TITLE, JOptionPane.OK_CANCEL_OPTION);
                if (result == JOptionPane.CANCEL_OPTION) {
                    return;
                }

            } else if (ESS.DELETABLE.test(ELEMENT)) {
                final String QUESTION = String.format("Are you sure you want to delete this element?\n%s", ELEMENT);
                int result = JOptionPane.showConfirmDialog(this, QUESTION, "Delete Element", JOptionPane.OK_CANCEL_OPTION);
                if (result == JOptionPane.CANCEL_OPTION) {
                    return;
                }

            } else if (ELEMENT instanceof SuspendedStack) {
                final String WARN = String.format("Element \"%s\" is a Suspended Stack. Deleting it could make your savefile impossible to load. Are you sure you want to proceed?", ELEMENT.toString());
                final String TITLE = "Warning";
                int result = JOptionPane.showConfirmDialog(this, WARN, TITLE, JOptionPane.OK_CANCEL_OPTION);
                if (result == JOptionPane.CANCEL_OPTION) {
                    return;
                }
            }

            int removed = this.save.removeElements(java.util.Collections.singleton(ELEMENT));
            if (removed < 1) {
                throw new IllegalStateException(String.format("Couldn't delete element:\n%s", ELEMENT));
            }

            final String MSG = String.format("Element Deleted:\n%s", ELEMENT);
            LOG.info(MSG);
            JOptionPane.showMessageDialog(this, MSG, "Element Deleted", JOptionPane.INFORMATION_MESSAGE);

        } else {

            final java.util.Set<Element> DELETABLE = elements.values().stream()
                    .filter(v -> v.hasElement())
                    .map(v -> v.getElement())
                    .filter(ESS.DELETABLE)
                    .filter(v -> !(v instanceof ActiveScript))
                    .filter(v -> !(v instanceof SuspendedStack))
                    .collect(Collectors.toSet());

            final java.util.Set<SuspendedStack> STACKS = elements.values().stream()
                    .filter(v -> v.hasElement())
                    .map(v -> v.getElement())
                    .filter(v -> v instanceof SuspendedStack)
                    .map(v -> (SuspendedStack) v)
                    .collect(Collectors.toSet());

            final java.util.Set<ActiveScript> THREADS = elements.values().stream()
                    .filter(v -> v.hasElement())
                    .map(v -> v.getElement())
                    .filter(v -> v instanceof ActiveScript)
                    .map(v -> (ActiveScript) v)
                    .collect(Collectors.toSet());

            boolean deleteStacks = false;
            if (!STACKS.isEmpty()) {
                final String WARN = "Deleting Suspended Stacks could make your savefile impossible to load.\nAre you sure you want to delete the Suspended Stacks?\nIf you select \"No\" then they will be skipped instead of deleted.";
                final String TITLE = "Warning";
                int result = JOptionPane.showConfirmDialog(this, WARN, TITLE, JOptionPane.YES_NO_CANCEL_OPTION);
                if (result == JOptionPane.CANCEL_OPTION) {
                    return;
                }
                deleteStacks = (result == JOptionPane.YES_OPTION);
            }

            boolean deleteThreads = false;
            if (!THREADS.isEmpty()) {
                final String WARN = "Deleting Active Scripts could make your savefile impossible to load.\nAre you sure you want to delete the Active Scripts?\nIf you select \"No\" then they will be terminated instead of deleted.";
                final String TITLE = "Warning";
                int result = JOptionPane.showConfirmDialog(this, WARN, TITLE, JOptionPane.YES_NO_CANCEL_OPTION);
                if (result == JOptionPane.CANCEL_OPTION) {
                    return;
                }
                deleteThreads = (result == JOptionPane.YES_OPTION);
            }

            int count = DELETABLE.size();
            count += (deleteStacks ? STACKS.size() : 0);
            count += (deleteThreads ? THREADS.size() : 0);

            final String QUESTION = String.format("Are you sure you want to delete these %d elements and their dependents?", count);
            int result = JOptionPane.showConfirmDialog(this, QUESTION, "Delete Elements", JOptionPane.YES_NO_OPTION);
            if (result == JOptionPane.NO_OPTION) {
                return;
            }

            int deleted = 0;
            int terminated = 0;

            deleted += this.save.removeElements(DELETABLE);

            if (deleteThreads) {
                deleted += this.save.removeElements(THREADS);
            } else {
                terminated += THREADS.size();
                THREADS.forEach(v -> v.zero());
            }

            if (deleteStacks) {
                deleted += this.save.removeElements(STACKS);
            }

            final StringBuilder BUF = new StringBuilder();
            BUF.append(deleted).append(" elements deleted.");

            if (terminated > 0) {
                BUF.append("\n").append(terminated).append(" threads terminated.");
            }

            final String MSG = BUF.toString();
            LOG.info(MSG);
            JOptionPane.showMessageDialog(this, MSG, "Elements Deleted", JOptionPane.INFORMATION_MESSAGE);
        }

        // This makes it easier to rebuild the current path.
        final int ROW = this.TREE.getSelectionModel().getMaxSelectionRow();
        this.TREE.setSelectionRow(ROW + 1);
        this.modified = true;
        this.rebuildTree(true);
    }

    /**
     * Updates infopane data.
     *
     */
    private void updateContextInformation() {
        final TreePath PATH = this.TREE.getSelectionPath();
        if (null == PATH) {
            this.clearContextInformation();
            return;
        }

        final Object OBJ = PATH.getLastPathComponent();
        if (!(OBJ instanceof Node)) {
            this.clearContextInformation();
            return;
        }

        final Node NODE = (Node) OBJ;

        if (!(NODE.getElement() instanceof Element)) {
            this.clearContextInformation();
            return;
        }

        final Element ELEMENT = (Element) NODE.getElement();
        this.showContextInformation(ELEMENT);
    }

    /**
     * Clears infopane data.
     *
     */
    private void clearContextInformation() {
        this.INFOPANE.setText("");
        this.TABLE.clearTable();
    }

    /**
     *
     * @param element
     */
    private void showContextInformation(Element element) {
        this.TABLE.clearTable();

        if (element instanceof ESS) {
            assert element == this.save;
            final String INFO = this.save.getInfo(analysis);
            this.INFOPANE.setText(INFO);
            int width = this.INFOPANE.getWidth() * 95 / 100;

            javax.swing.ImageIcon icon = this.save.getHeader().getImage(width);
            javax.swing.JLabel label = new javax.swing.JLabel(icon);
            javax.swing.text.Document doc = this.INFOPANE.getDocument();
            javax.swing.text.StyleContext ctx = new javax.swing.text.StyleContext();
            javax.swing.text.Style labelStyle = ctx.getStyle(StyleContext.DEFAULT_STYLE);
            javax.swing.text.StyleConstants.setComponent(labelStyle, label);

            try {
                doc.insertString(doc.getLength(), "Ignored", labelStyle);
            } catch (javax.swing.text.BadLocationException ex) {
                ex.printStackTrace(System.err);
            }

        } else {
            if (element instanceof Plugin) {
                Plugin plugin = (Plugin) element;
                this.INFOPANE.setText(plugin.getInfo(analysis, save));
            } else if (element instanceof TString) {
                TString str = (TString) element;
                this.INFOPANE.setText(str.getInfo(this.analysis, this.save));
            } else if (element instanceof Plugin) {
                Plugin plugin = (Plugin) element;
                this.INFOPANE.setText(plugin.getInfo(this.analysis, this.save));
            } else if (element instanceof Script) {
                Script script = (Script) element;
                this.INFOPANE.setText(script.getInfo(this.analysis, this.save));
                this.TABLE.displayScript(script, this.save);
            } else if (element instanceof ScriptInstance) {
                ScriptInstance instance = (ScriptInstance) element;
                this.INFOPANE.setText(instance.getInfo(this.analysis, this.save));
                this.TABLE.displayScriptInstance(instance, this.save);
            } else if (element instanceof StructDef) {
                StructDef def = (StructDef) element;
                this.INFOPANE.setText(def.getInfo(this.analysis, this.save));
                this.TABLE.displayStructDef(def, this.save);
            } else if (element instanceof Struct) {
                Struct struct = (Struct) element;
                this.INFOPANE.setText(struct.getInfo(this.analysis, this.save));
                this.TABLE.displayStruct(struct, this.save);
            } else if (element instanceof ArrayInfo) {
                ArrayInfo array = (ArrayInfo) element;
                this.INFOPANE.setText(array.getInfo(this.analysis, this.save));
                this.TABLE.displayArray(array, this.save);
            } else if (element instanceof StackFrame) {
                StackFrame frame = (StackFrame) element;
                this.INFOPANE.setText(frame.getInfo(this.analysis, this.save));
                this.TABLE.displayStackFrame(frame, this.save);
            } else if (element instanceof Reference) {
                Reference ref = (Reference) element;
                this.INFOPANE.setText(ref.getInfo(this.analysis, this.save));
                this.TABLE.displayReference(ref, this.save);
            } else if (element instanceof ActiveScript) {
                ActiveScript script = (ActiveScript) element;
                this.INFOPANE.setText(script.getInfo(this.analysis, this.save));
                this.TABLE.clearTable();
            } else if (element instanceof SuspendedStack) {
                SuspendedStack stack = (SuspendedStack) element;
                this.INFOPANE.setText(stack.getInfo(this.analysis, this.save));
                this.TABLE.clearTable();
            } else if (element instanceof FunctionMessage) {
                FunctionMessage message = (FunctionMessage) element;
                String msg = message.getInfo(this.analysis, this.save);
                this.INFOPANE.setText(message.getInfo(this.analysis, this.save));
                this.TABLE.clearTable();
            } else if (element instanceof FunctionMessageData) {
                FunctionMessageData message = (FunctionMessageData) element;
                String msg = message.getInfo(this.analysis, this.save);
                this.INFOPANE.setText(message.getInfo(this.analysis, this.save));
                this.TABLE.displayFunctionMessageData(message, this.save);
            } else if (element instanceof ChangeForm) {
                ChangeForm form = (ChangeForm) element;
                this.INFOPANE.setText(form.getInfo(this.analysis, this.save));
                this.TABLE.clearTable();
            } else if (element instanceof QueuedUnbind) {
                QueuedUnbind qu = (QueuedUnbind) element;
                this.INFOPANE.setText(qu.getInfo(this.analysis, this.save));
                this.TABLE.clearTable();
            } else {
                this.INFOPANE.setText("");
                this.TABLE.clearTable();
            }

            this.TABLE.getModel().addTableModelListener(e -> {
                SaveWindow.this.modified = true;
            });
        }

        this.INFOPANE.setCaretPosition(0);
    }

    /**
     * Used to render cells.
     */
    final private class ModListCellRenderer implements ListCellRenderer<Mod> {

        @Override
        public Component getListCellRendererComponent(JList list, Mod value, int index, boolean isSelected, boolean cellHasFocus) {
            if (null == value) {
                return RENDERER.getListCellRendererComponent(list, null, index, isSelected, cellHasFocus);
            }
            return RENDERER.getListCellRendererComponent(list, value.getName(), index, isSelected, cellHasFocus);
        }

        final private BasicComboBoxRenderer RENDERER = new BasicComboBoxRenderer();
    }

    /**
     * Used to render cells.
     */
    final private class PluginListCellRenderer implements ListCellRenderer<Plugin> {

        @Override
        public Component getListCellRendererComponent(JList list, Plugin value, int index, boolean isSelected, boolean cellHasFocus) {
            if (null == value) {
                return RENDERER.getListCellRendererComponent(list, null, index, isSelected, cellHasFocus);
            }
            return RENDERER.getListCellRendererComponent(list, value.NAME, index, isSelected, cellHasFocus);
        }

        final private BasicComboBoxRenderer RENDERER = new BasicComboBoxRenderer();
    }

    private ESS save;
    private Analysis analysis;
    private boolean modified;
    private Predicate<Node> filter;

    final private java.util.Timer FILTERTIMER;
    final private MemoryLabel LBLMEMORY;
    final private JLabel LBLSAVEINFO;
    final private FilterTree TREE;
    final private VariableTable TABLE;
    final private JTextPane INFOPANE;
    final private JButton CLEAR_FILTER;
    final private JScrollPane TREESCROLLER;
    final private JScrollPane DATASCROLLER;
    final private JScrollPane INFOSCROLLER;
    final private JSplitPane MAINSPLITTER;
    final private JSplitPane RIGHTSPLITTER;
    final private JPanel MAINPANEL;
    final private JPanel MODPANEL;
    final private JComboBox<Mod> MODCOMBO;
    final private JComboBox<Plugin> PLUGINCOMBO;
    final private JLabel MODLABEL;
    final private JPanel FILTERPANEL;
    final private JTextField FILTERFIELD;
    final private JPanel TOPPANEL;
    final private JPanel PROGRESSPANEL;
    final private JProgressBar PROGRESSBAR;
    final private JMenuBar MENUBAR;
    final private JMenu FILEMENU;
    final private JMenu DATAMENU;
    final private JMenu CLEANMENU;
    final private JMenu OPTIONSMENU;
    final private JMenu HELPMENU;
    final private JMenuItem MI_LOAD;
    final private JMenuItem MI_SAVE;
    final private JMenuItem MI_SAVEAS;
    final private JMenuItem MI_EXIT;
    final private JMenuItem MI_LOADESPS;
    final private JMenuItem MI_LOOKUPID;
    final private JMenuItem MI_LOOKUPBASE;
    final private JMenuItem MI_REMOVEUNATTACHED;
    final private JMenuItem MI_REMOVEUNDEFINED;
    final private JMenuItem MI_RESETHAVOK;
    final private JMenuItem MI_CLEANSEFORMLISTS;
    final private JMenuItem MI_REMOVENONEXISTENT;
    final private JMenuItem MI_BATCHCLEAN;
    final private JMenuItem MI_KILL;
    final private JCheckBoxMenuItem MI_USEMO;
    final private JCheckBoxMenuItem MI_USENMM;
    final private JCheckBoxMenuItem MI_SHOWMODS;
    final private JMenuItem MI_SHOWLOG;
    final private JMenuItem MI_ABOUT;
    final private JMenuItem MI_EXPORTPLUGINS;
    final private JCheckBoxMenuItem MI_SHOWUNATTACHED;
    final private JCheckBoxMenuItem MI_SHOWUNDEFINED;
    final private JCheckBoxMenuItem MI_SHOWNULLREFS;
    final private JCheckBoxMenuItem MI_SHOWNONEXISTENTCREATED;
    final private LogWindow LOGWINDOW;
    final private restringer.Timer TIMER;
    static final private Logger LOG = Logger.getLogger(SaveWindow.class.getCanonicalName());

    static final private Pattern URLREGEX = Pattern.compile("^(.+)://(.+)$", Pattern.CASE_INSENSITIVE);
}
