/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.swing.JOptionPane;
import javax.swing.SwingWorker;
import restringer.Game;
import restringer.Mod;
import restringer.Profile;
import restringer.esp.ESP;
import restringer.esp.ESPIDMap;
import restringer.esp.StringsFile;
import restringer.ess.ESS;
import restringer.ess.Plugin;

/**
 *
 * @author Mark Fairchild
 */
public class Scanner extends SwingWorker<ESPIDMap, Double> {

    /**
     *
     * @param window
     * @param save
     * @param gameDir
     * @param moDir
     * @param nmmDir
     */
    public Scanner(SaveWindow window, ESS save, File gameDir, File moDir, File nmmDir) {
        this.WINDOW = Objects.requireNonNull(window, "The window field must not be null.");
        this.SAVE = Objects.requireNonNull(save, "The save field must not be null.");
        this.GAME_DIR = Objects.requireNonNull(gameDir, "The game directory field must not be null.");
        this.MODORGANIZER_DIR = moDir;
        this.NMM_DIR = nmmDir;
    }

    /**
     *
     * @return @throws Exception
     */
    @Override
    protected ESPIDMap doInBackground() throws Exception {
        final restringer.Timer TIMER = restringer.Timer.startNew("Load Plugins");
        final ProgressModel MODEL = new ProgressModel();
        final Game GAME = this.SAVE.getHeader().GAME;
        
        ProgressIndicator.startWaitCursor(this.WINDOW.getRootPane());
        this.WINDOW.addWindowListener(this.LISTENER);
        this.WINDOW.startProgressBar(MODEL);

        try {
            final java.util.List<Plugin> PLUGINS = this.SAVE.getPluginInfo().getPlugins();
            final java.util.List<Mod> MODS = new ArrayList<>(256);
            final java.util.List<String> ESP_NAMES = PLUGINS.stream().map(p -> p.NAME).collect(Collectors.toList());
            final java.util.List<File> ERR = java.util.Collections.synchronizedList(new java.util.LinkedList<>());
            final Map<Plugin, Mod> PLUGIN_MOD_MAP = new java.util.LinkedHashMap<>(256);
            final Map<Plugin, File> PLUGIN_ESP_MAP = new java.util.LinkedHashMap<>(256);
            final Map<Plugin, Profile.Analysis> ANALYSES = new java.util.concurrent.ConcurrentHashMap<>(256);

            LOG.info("Scanning plugins.");

            MODS.add(new Mod(GAME, this.GAME_DIR));

            if (null != this.MODORGANIZER_DIR) {
                LOG.info("Checking Mod Organizer.");
                final java.util.List<Mod> MOMODS = Configurator.analyzeModOrganizer(GAME, this.MODORGANIZER_DIR);
                MODS.addAll(MOMODS);
            }

            MODS.forEach(mod -> {
                mod.getESPFiles().forEach(file -> {
                    PLUGINS.stream()
                            .filter(p -> p.NAME.equalsIgnoreCase(file.getName()))
                            .findAny()
                            .ifPresent(p -> {
                                PLUGIN_MOD_MAP.putIfAbsent(p, mod);
                                PLUGIN_ESP_MAP.putIfAbsent(p, file);
                            });
                });
            });

            long totalSize = 0;
            long[] BYTES_READ = new long[]{0};
            totalSize += PLUGIN_ESP_MAP.values().stream().mapToLong(f -> f.length()).sum();
            MODEL.setMaximum((int) (totalSize / 1024));

            final ESPIDMap IDS = new ESPIDMap("global");
            final restringer.esp.StringTable STRINGS = new restringer.esp.StringTable();

            PLUGINS.parallelStream().forEach(plugin -> {
                if (!PLUGIN_ESP_MAP.containsKey(plugin)) {
                    LOG.warning(String.format("Plugin %s is missing.", plugin));

                } else {
                    final File ESPFILE = PLUGIN_ESP_MAP.get(plugin);
                    final Mod MOD = PLUGIN_MOD_MAP.get(plugin);

                    // Read stringtables.
                    try {
                        java.util.List<StringsFile> strings = MOD.readStrings(ESPFILE, "english", true);
                        strings.parallelStream().forEach(v -> STRINGS.populateFromFile(v, plugin.INDEX));

                    } catch (Mod.StringsReadError ex) {
                        LOG.warning(String.format("Error reading stringtables for plugin %s", plugin));
                        LOG.warning(ex.getMessage());
                    }

                    // Read the scripts.
                    final restringer.Profile.Analysis ANALYSIS = restringer.Profile.analyzeMod(MOD);
                    ANALYSES.put(plugin, ANALYSIS);

                    // Skims the ESP.
                    try {
                        ESPIDMap espMap = ESP.skimESP(GAME, ESPFILE, plugin.INDEX, ESP_NAMES);
                        IDS.addAll(espMap);

                        LOG.info(String.format("Scanned plugin: %s", plugin));

                    } catch (IOException ex) {
                        ERR.add(ESPFILE);
                        LOG.warning(String.format("Error reading plugin: %s.", plugin));
                        LOG.warning(ex.getMessage());
                        System.err.println(ex.getMessage());
                        System.err.println(ex.getMessage());
                        ex.printStackTrace(System.err);

                    } finally {
                        synchronized (BYTES_READ) {
                            BYTES_READ[0] += ESPFILE.length();
                            MODEL.setValue((int) (BYTES_READ[0] / 1024));
                        }
                    }
                }
            });

            // Merge the analyses.
            final Profile.Analysis PROFILEANALYSIS = PLUGINS
                    .stream()
                    .filter(p -> ANALYSES.containsKey(p))
                    .map(p -> ANALYSES.get(p))
                    .reduce(new Profile.Analysis(), (a1, a2) -> a1.merge(a2));

            final restringer.Analysis ANALYSIS = new restringer.Analysis(PROFILEANALYSIS, IDS, STRINGS);            
            if (null != this.SAVE) {
                this.WINDOW.setAnalysis(ANALYSIS);
            }

            TIMER.stop();
            LOG.info(String.format("Plugin scanning completed, took %s.", TIMER.getFormattedTime()));

            JOptionPane.showMessageDialog(this.WINDOW, "Done scanning plugins.", "Done", JOptionPane.INFORMATION_MESSAGE);

            if (!ERR.isEmpty()) {
                final StringBuilder BUF = new StringBuilder();
                BUF.append("The following plugins could not be read:\n");
                ERR.forEach(p -> BUF.append("\t").append(p.getName()).append("\n"));
                JOptionPane.showMessageDialog(this.WINDOW, BUF.toString(), "Errors", JOptionPane.ERROR_MESSAGE);
            }

            return IDS;

        } catch (Exception | Error ex) {
            final String MSG = String.format("Error reading plugins. %s", ex.getMessage());
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(this.WINDOW, MSG, "Read Error", JOptionPane.ERROR_MESSAGE);
            return null;

        } finally {
            this.WINDOW.removeWindowListener(this.LISTENER);
            ProgressIndicator.stopWaitCursor(this.WINDOW.getRootPane());
            this.WINDOW.clearProgressBar();
        }
    }

    final private SaveWindow WINDOW;
    final private ESS SAVE;
    final private File GAME_DIR;
    final private File MODORGANIZER_DIR;    
    final private File NMM_DIR;
    static final private Logger LOG = Logger.getLogger(Scanner.class.getCanonicalName());

    final private WindowAdapter LISTENER = new WindowAdapter() {
        @Override
        public void windowClosing(WindowEvent e) {
            if (!isDone()) {
                cancel(true);
            }
        }
    };
}
