/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.io.File;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;
import javafx.application.Platform;
import javafx.stage.FileChooser;
import javax.swing.JDialog;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

/**
 *
 * @author Mark Fairchild
 */
public class SynchronousFXChooser {
    
    public SynchronousFXChooser(Supplier<FileChooser> fileChooserFactory) {
        this.fileChooserFactory = fileChooserFactory;
    }
    
    public <T> T showDialog(Function<FileChooser, T> method) {
        return showDialog(method, 1L, TimeUnit.SECONDS);
    }
    
    public <T> T showDialog(Function<FileChooser, T> method, long timeout, TimeUnit unit) {
        Callable<T> task = () -> {
            FileChooser chooser = fileChooserFactory.get();
            return method.apply(chooser);
        };
        
        SynchronousJFXCaller<T> caller = new SynchronousJFXCaller<>(task);
        try {
            return caller.call(timeout, unit);
        } catch (RuntimeException | Error ex) {
            throw ex;
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            return null;
        } catch (Exception ex) {
            throw new AssertionError("Got unexpected checked exception from SynchronousJFXCaller.call()", ex);
        }
    }
    
    public File showOpenDialog() {
        return showDialog(chooser -> chooser.showOpenDialog(null));
    }
    
    public File showSaveDialog() {
        return showDialog(chooser -> chooser.showSaveDialog(null));
    }
    
    public List<File> showOpenMultipleDialog() {
        return showDialog(chooser -> chooser.showOpenMultipleDialog(null));
    }
    
    final private Supplier<FileChooser> fileChooserFactory;

    static private class SynchronousJFXCaller<T> {
        
        public SynchronousJFXCaller(Callable<T> callable) {
            this.callable = callable;
        }
        
        public T call(long startTimeout, TimeUnit startTimeoutUnit) throws Exception {
            final CountDownLatch taskStarted = new CountDownLatch(1);
            final AtomicBoolean taskCancelled = new AtomicBoolean(false);
            final JDialog modalBlocker = new JDialog();
            modalBlocker.setModal(true);
            modalBlocker.setUndecorated(true);
            modalBlocker.setOpacity(0.0f);
            modalBlocker.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
            final CountDownLatch modalityLatch = new CountDownLatch(1);
            final FutureTask<T> task = new FutureTask<>(() -> {
                synchronized(taskStarted) {
                    if (taskCancelled.get()) {
                        return null;
                    } else {
                        taskStarted.countDown();
                    }
                }
                try {
                    return callable.call();
                } finally {
                    modalityLatch.await();
                    SwingUtilities.invokeLater(() -> {
                        modalBlocker.setVisible(false);
                    });
                }
            });
            Platform.runLater(task);
            if (!taskStarted.await(startTimeout, startTimeoutUnit)) {
                synchronized(taskStarted) {
                    if (!taskStarted.await(0, TimeUnit.MILLISECONDS)) {
                        taskCancelled.set(true);
                        throw new IllegalStateException("JavaFX was shut down or is unresponsive.");
                    }
                }
            }
            SwingUtilities.invokeLater(() -> {
                modalityLatch.countDown();
            });
            modalBlocker.setVisible(true);
            modalBlocker.dispose();
            try {
                return task.get();
            } catch(ExecutionException ex) {
                Throwable ec = ex.getCause();
                if (ec instanceof Exception) {
                    throw (Exception) ec;
                } else if (ec instanceof Error) {
                    throw (Error) ec;                    
                } else {
                    throw new AssertionError("Unexpected exception type", ec);
                }
            }
        }
        
        final private Callable<T> callable;
    }
}
