/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.util.List;
import java.util.ResourceBundle;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import restringer.ess.papyrus.*;

/**
 * A table model for list of variables.
 *
 * @author Mark Fairchild
 * @version 2016/06/23
 */
abstract public class VariableTableModel implements javax.swing.table.TableModel {

    /**
     * Creates a new <code>VariableTableModel</code>.
     */
    public VariableTableModel() {
        this.LISTENERS = new java.util.LinkedList<>();
    }

    /**
     * @return The total number of variables in the model.
     */
    abstract protected int getNumVariables();

    /**
     * Getter for the <code>Variable</code> at the specified index.
     *
     * @param index The index of the <code>Variable</code> to return.
     * @return The <code>Variable</code>.
     */
    abstract protected Variable getVariable(int index);

    /**
     * Getter for the <code>MemberDesc</code> at the specified index.
     *
     * @param index The index of the <code>MemberDesc</code> to return.
     * @return The <code>MemberDesc</code>.
     */
    abstract protected MemberDesc getMemberData(int index);

    /**
     * Setter for the <code>Variable</code> at the specified index.
     *
     * @param index The index of the <code>Variable</code> to replace.
     * @param newVar The new <code>Variable</code>.
     */
    abstract protected void setVariable(int index, Variable newVar);

    /**
     * @return Indicates whether to include a column for the names of variables.
     */
    abstract protected boolean supportsMemberData();

    @Override
    public int getRowCount() {
        return this.getNumVariables();
    }

    @Override
    public int getColumnCount() {
        return (this.supportsMemberData() ? 4 : 3);
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        assert 0 <= rowIndex && rowIndex < this.getRowCount();

        Variable var = this.getVariable(rowIndex);

        if (!this.supportsMemberData() && columnIndex == 2) {
            columnIndex++;
        }
        
        switch (columnIndex) {
            case 0:
                return rowIndex;
            case 1:
                return var.toTypeString();
            case 2:
                MemberDesc mb = this.getMemberData(rowIndex);
                return (null != mb ? mb.getName() : "");
            case 3:
                return var;
            default:
                throw new IllegalStateException();
        }
    }

    @Override
    public String getColumnName(int columnIndex) {
        if (!this.supportsMemberData() && columnIndex == 2) {
            columnIndex++;
        }

        return COLUMNNAMES[columnIndex];
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        if (!this.supportsMemberData() && columnIndex == 2) {
            columnIndex++;
        }

        return COLUMNTYPES[columnIndex];
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        if (this.supportsMemberData() && columnIndex != 3) {
            return false;
        } else if (!this.supportsMemberData() && columnIndex != 2) {
            return false;
        }

        assert 0 <= rowIndex && rowIndex < this.getRowCount();
        Variable var = this.getVariable(rowIndex);

        switch (var.getType()) {
            case STRING:
            case INTEGER:
            case FLOAT:
            case BOOLEAN:
            case REF:
                return true;
            default:
                return false;
        }
    }

    @Override
    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        if (!this.isCellEditable(rowIndex, columnIndex)) {
            throw new UnsupportedOperationException("Not supported."); //To change body of generated methods, choose Tools | Templates.
        } else if (!(aValue instanceof Variable)) {
            throw new UnsupportedOperationException("Not supported."); //To change body of generated methods, choose Tools | Templates.
        }

        this.setVariable(rowIndex, (Variable) aValue);
        this.fireTableCellUpdate(rowIndex, columnIndex);
    }

    public void fireTableCellUpdate(int row, int column) {
        TableModelEvent event = new TableModelEvent(this, row, row, column, TableModelEvent.UPDATE);
        this.LISTENERS.forEach(l -> l.tableChanged(event));
    }

    @Override
    public void addTableModelListener(TableModelListener l) {
        this.LISTENERS.add(l);
    }

    @Override
    public void removeTableModelListener(TableModelListener l) {
        this.LISTENERS.remove(l);
    }

    final private List<TableModelListener> LISTENERS;
    static final private ResourceBundle RES = ResourceBundle.getBundle("restringer/gui/General");
    static final private String[] COLUMNNAMES = getColumnNames();
    static final private Class<?>[] COLUMNTYPES = getColumnTypes();

    static private String[] getColumnNames() {
        return new String[]{
            RES.getString("VARIABLETABLEMODEL #"),
            RES.getString("VARIABLETABLEMODEL TYPE"),
            RES.getString("VARIABLETABLEMODEL NAME"),
            RES.getString("VARIABLETABLEMODEL VALUE")};
    }

    static private Class<?>[] getColumnTypes() {
        return new Class<?>[]{Integer.class, String.class, String.class, Variable.class};
    }
}
