/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.pex;

import java.io.DataInput;
import java.io.IOException;
import java.util.Objects;

/**
 * Describes the different opcodes that can appear in PEX functions, and stores
 * the number of arguments that they accept. If the number of arguments is
 * negative, it indicates that a variable number of arguments can follow the
 * main arguments.
 *
 * @author Mark Fairchild
 * @version 2016/07/04
 */
public enum Opcode {
    NOP(0),           // 0    00
    IADD(3),          // 1    01
    FADD(3),          // 2    02
    ISUB(3),          // 3    03
    FSUB(3),          // 4    04
    IMUL(3),          // 5    05
    FMUL(3),          // 6    06
    IDIV(3),          // 7    07
    FDIV(3),          // 8    08
    IMOD(3),          // 9    09
    NOT(2),           // 10   0a
    INEG(2),          // 11   0b
    FNEG(2),          // 12   0c
    ASSIGN(2),        // 13   0d
    CAST(2),          // 14   0e
    CMP_EQ(3),        // 15   0f
    CMP_LT(3),        // 16   10
    CMP_LE(3),        // 17   11
    CMP_GT(3),        // 18   12
    CMP_GE(3),        // 19   13
    JMP(1),           // 20   14
    JMPT(2),          // 21   15
    JMPF(2),          // 22   16
    CALLMETHOD(-3),   // 23   17
    CALLPARENT(-2),   // 24   18
    CALLSTATIC(-3),   // 25   19
    RETURN(1),        // 26   1a
    STRCAT(3),        // 27   1b
    PROPGET(3),       // 28   1c
    PROPSET(3),       // 29   1d
    ARR_CREATE(2),    // 30   1e
    ARR_LENGTH(2),    // 31   1f
    ARR_GET(3),       // 32   20
    ARR_SET(3),       // 33   21
    ARR_FIND(4),      // 34   22
    ARR_RFIND(4),     // 35   23
    UNKNOWN_OP1(3),    // 36   24
    UNKNOWN_OP2(1),    // 37   25
    UNKNOWN_OP3(3),    // 38   26
    UNKNOWN_OP4(3),    // 39   27
    UNKNOWN_OP5(3),    // 40   28
    UNKNOWN_OP6(3),    // 41   29
    UNKNOWN_OP7(3),    // 42   2a
    UNKNOWN_OP8(3),    // 43   2b
    UNKNOWN_OP9(3),    // 44   2c
    UNKNOWN_OP10(3);    // 45   2d

    public boolean isConditional() {
        return (this == JMPT) || (this == JMPF);
    }

    public boolean isBranching() {
        return (this == JMPT) || (this == JMPF) || (this == JMP);
    }

    public boolean isArithmetic() {
        return this == IADD || this == FADD || 
                this == ISUB || this == FSUB || 
                this == IMUL || this == FMUL || 
                this == IDIV || this == FDIV || 
                this == IMOD;
    }

    private Opcode(int args) {
        this.ARGS = args;
    }

    /**
     * Read a <code>DataType</code> from an input stream.
     *
     * @param input The input stream.
     * @return The <code>DataType</code>.
     */
    static Opcode read(DataInput input) throws IOException {
        Objects.requireNonNull(input);

        int index = input.readUnsignedByte();
        if (index < 0 || index >= VALUES.length) {
            throw new IOException("Invalid Opcode.");
        }

        return VALUES[index];
    }

    public final int ARGS;
    static final private Opcode[] VALUES = Opcode.values();

}
