/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.pex;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import restringer.Game;
import restringer.IString;
import restringer.LittleEndianInputStream;
import restringer.pex.StringTable.TString;

/**
 * Describes a Skyrim PEX script and will read and write it from streams.
 *
 * @author Mark Fairchild
 * @version 2016/07/04
 */
final public class PexFile {

    /**
     * Reads a script file and creates a PexFile object to represent it.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param data An array of bytes containing the script data.
     * @return The PexFile object.
     *
     * @throws IOException
     *
     */
    static public PexFile readScript(byte[] data) throws IOException {
        final int MAGIC;
        try (final DataInputStream BIS = new DataInputStream(new ByteArrayInputStream(data, 0, 4))) {
            MAGIC = BIS.readInt();
        }

        // Prepare input stream. The DataInput interface just happen to be 
        // perfect for this kind of thing.
        switch (MAGIC) {
            case 0xdec057fa:
                try (final LittleEndianInputStream INPUT = LittleEndianInputStream.debug(data)) {
                    return new PexFile(Game.FALLOUT4, INPUT);
                }
            case 0xfa57c0de:
                try (final DataInputStream INPUT = new DataInputStream(new ByteArrayInputStream(data))) {
                    return new PexFile(Game.SKYRIM_LE, INPUT);
                }
            default:
                throw new IOException("Invalid magic number.");
        }
    }

    /**
     * Reads a script file and creates a PexFile object to represent it.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param scriptFile The script file to read, which must exist and be
     * readable.
     * @return The PexFile object.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public PexFile readScript(File scriptFile) throws FileNotFoundException, IOException {
        final int MAGIC;
        try (final DataInputStream BIS = new DataInputStream(new BufferedInputStream(new FileInputStream(scriptFile)))) {
            MAGIC = BIS.readInt();
        }

        // Prepare input stream. The DataInput interface just happen to be 
        // perfect for this kind of thing.
        switch (MAGIC) {
            case 0xdec057fa:
                try (final LittleEndianInputStream INPUT = LittleEndianInputStream.opend(scriptFile)) {
                    return new PexFile(Game.FALLOUT4, INPUT);
                }
            case 0xfa57c0de:
                try (final DataInputStream INPUT = new DataInputStream(new BufferedInputStream(new FileInputStream(scriptFile)))) {
                    return new PexFile(Game.SKYRIM_LE, INPUT);
                }
            default:
                throw new IOException("Invalid magic number.");
        }
    }

    /**
     * Writes a PexFile object to a script file.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param script The PexFile object to write.
     * @param scriptFile The script file to write. If it exists, it must be a
     * file and it must be writable.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public void writeScript(PexFile script, File scriptFile) throws FileNotFoundException, IOException {
        assert !scriptFile.exists() || scriptFile.isFile();
        assert !scriptFile.exists() || scriptFile.canWrite();

        // Prepare output streams. The DataOutput interface just happen to be 
        // perfect for this kind of thing.
        try (DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(scriptFile)))) {
            script.write(dos);
        }
    }

    /**
     * Creates a Pex by reading from a DataInput.
     *
     * @param game The game for which the script was compiled.
     * @param input A datainput for a Skyrim PEX file.
     * @throws IOException Exceptions aren't handled.
     */
    private PexFile(Game game, DataInput input) throws IOException {
        try {
            this.HEADER = new Header(input);

            this.STRINGS = new StringTable(input);
            this.DEBUG = new DebugInfo(input, this.STRINGS);

            int flagCount = input.readUnsignedShort();
            this.USERFLAGDEFS = new ArrayList<>(flagCount);
            while (0 < flagCount) {
                this.USERFLAGDEFS.add(new UserFlag(input, this.STRINGS));
                flagCount--;
            }

            int scriptCount = input.readUnsignedShort();
            if (scriptCount < 1) {
                throw new IllegalStateException("Pex files must contain at least one script.");
            }

            this.SCRIPTS = new ArrayList<>(scriptCount);
            while (0 < scriptCount) {
                Pex pex = new Pex(input, game, this.USERFLAGDEFS, this.STRINGS);
                this.SCRIPTS.add(pex);
                scriptCount--;
            }
        } catch (IOException ex) {
            throw ex;
        }
    }

    /**
     * Write the object to a <code>DataOutput</code>.
     *
     * @param output The <code>DataOutput</code> to write.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     */
    public void write(DataOutput output) throws IOException {
        this.HEADER.write(output);

        this.STRINGS.write(output);
        this.DEBUG.write(output);

        output.writeShort((short) this.USERFLAGDEFS.size());
        for (UserFlag flag : this.USERFLAGDEFS) {
            flag.write(output);
        }

        output.writeShort(this.SCRIPTS.size());
        for (Pex pex : this.SCRIPTS) {
            pex.write(output);
        }

        /*
        output.writeShort((short) this.OBJECTS.size());
        for (Pex obj : this.OBJECTS) {
            obj.write(output);
        }*/
    }

    /**
     * Rebuilds the string table. This is necessary if ANY strings in ANY of the
     * PexFile's members has changed at all. Otherwise, writing the PexFile will
     * produce an invalid file.
     *
     */
    public void rebuildStringTable() {
        final Set<TString> INUSE = new java.util.LinkedHashSet<>();
        this.DEBUG.collectStrings(INUSE);
        this.USERFLAGDEFS.forEach(flag -> flag.collectStrings(INUSE));
        this.SCRIPTS.forEach(obj -> obj.collectStrings(INUSE));
        this.STRINGS.rebuildStringTable(INUSE);
    }

    /**
     * Tries to disassemble the script.
     *
     * @param level Partial disassembly flag.
     * @param code The code strings.
     */
    public void disassemble(List<String> code, AssemblyLevel level) {
        this.SCRIPTS.forEach(v -> v.disassemble(code, level));
    }

    /**
     * Pretty-prints the PexFile.
     *
     * @return A string representation of the PexFile.
     */
    @Override
    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append(this.HEADER.toString());
        buf.append(this.DEBUG.toString());

        buf.append("USER FLAGS\n");
        buf.append(this.USERFLAGDEFS.toString());

        this.SCRIPTS.forEach(obj -> buf.append("\n\nOBJECT\n").append(obj).append("\n"));

        return buf.toString();

    }

    /**
     * Read an IString from a PEX file.
     *
     * @param input The <code>DataInput</code> to read.
     * @return The IString that was read.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     */
    private IString readIString(DataInput input) throws IOException {
        int index = input.readUnsignedShort();

        if (index < 0 || index >= STRINGS.size()) {
            throw new IOException();
        }

        return this.STRINGS.get(index);
    }

    /**
     * Write an IString to a PEX file.
     *
     * @param str The IString to write.
     * @param output The <code>DataOutput</code> to write.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     */
    private void writeIString(IString str, DataOutput output) throws IOException {
        short index = (short) STRINGS.indexOf(str);
        output.writeShort(index);
    }

    /**
     * @return The compilation date of the <code>PexFile</code>.
     */
    public long getDate() {
        return this.HEADER.compilationTime;
    }

    /**
     * @return The filename of the <code>PexFile</code>, determined from the
     * header.
     */
    public IString getFilename() {
        final String SOURCE = this.HEADER.soureFilename;
        final String REGEX = "(psc)$";
        final String REPLACEMENT = "pex";
        final Pattern PATTERN = Pattern.compile(REGEX, Pattern.CASE_INSENSITIVE);
        final Matcher MATCHER = PATTERN.matcher(SOURCE);
        final String COMPILED = MATCHER.replaceAll(REPLACEMENT);
        return IString.get(COMPILED);
    }

    /**
     * @return A <code>ScriptStats</code> object for the <code>Pex</code>.
     */
    public ScriptStats analyze() {
        final ScriptStats STATS = new ScriptStats();
        this.SCRIPTS.forEach(obj -> obj.analyze(STATS));
        return STATS;
    }

    final public Header HEADER;
    final public StringTable STRINGS;
    final public DebugInfo DEBUG;
    final public List<UserFlag> USERFLAGDEFS;
    final public List<Pex> SCRIPTS;

    /**
     * Describes the header of a PexFile file. Useless beyond that.
     *
     */
    final public class Header {

        /**
         * Creates a Header by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @throws IOException Exceptions aren't handled.
         */
        private Header(DataInput input) throws IOException {
            this.magic = input.readInt();
            this.version = input.readInt();
            this.compilationTime = input.readLong();
            this.soureFilename = input.readUTF();
            this.userName = input.readUTF();
            this.machineName = input.readUTF();
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            output.writeInt(this.magic);
            output.writeInt(this.version);
            output.writeLong(this.compilationTime);
            output.writeUTF(this.soureFilename);
            output.writeUTF(this.userName);
            output.writeUTF(this.machineName);
        }

        /**
         * Pretty-prints the Header.
         *
         * @return A string representation of the Header.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append(String.format("%s compiled at %d by %s on %s.\n", this.soureFilename, this.compilationTime, this.userName, this.machineName));
            buf.append(String.format("%h v%d\n\n", this.magic, this.version));
            return buf.toString();
        }

        private int magic = 0;
        private int version = 0;
        private long compilationTime = 0;
        private String soureFilename = "";
        private String userName = "";
        private String machineName = "";

    }

    /**
     * Describe the debugging info section of a PEX file.
     *
     */
    final public class DebugInfo {

        /**
         * Creates a DebugInfo by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param strings The <code>StringTable</code> for the
         * <code>PexFile</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private DebugInfo(DataInput input, StringTable strings) throws IOException {
            this.hasDebugInfo = input.readByte();

            if (this.hasDebugInfo == 0) {
                this.DEBUGFUNCTIONS = new ArrayList<>(0);

            } else {
                this.modificationTime = input.readLong();
                int functionCount = input.readUnsignedShort();
                this.DEBUGFUNCTIONS = new ArrayList<>(functionCount);
                for (int i = 0; i < functionCount; i++) {
                    this.DEBUGFUNCTIONS.add(new DebugFunction(input, strings));
                }
            }
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            output.write(this.hasDebugInfo);

            if (this.hasDebugInfo != 0) {
                output.writeLong(this.modificationTime);
                output.writeShort(this.DEBUGFUNCTIONS.size());

                for (DebugFunction function : this.DEBUGFUNCTIONS) {
                    function.write(output);
                }
            }
        }

        /**
         * Removes all debug info.
         */
        public void clear() {
            this.hasDebugInfo = 0;
            this.DEBUGFUNCTIONS.clear();

        }

        /**
         * Collects all of the strings used by the DebugInfo and adds them to a
         * set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            this.DEBUGFUNCTIONS.forEach(func -> func.collectStrings(strings));
        }

        /**
         * Pretty-prints the DebugInfo.
         *
         * @return A string representation of the DebugInfo.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append("DEBUGINFO\n");
            this.DEBUGFUNCTIONS.forEach(function -> {
                buf.append("\t");
                buf.append(function.toString());
                buf.append("\n");
            });
            buf.append("\n");
            return buf.toString();
        }

        private byte hasDebugInfo;
        private long modificationTime;
        final private List<DebugFunction> DEBUGFUNCTIONS;

    }

    /**
     * Describes the debugging information for a function.
     *
     */
    final class DebugFunction {

        /**
         * Creates a DebugFunction by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param strings The <code>StringTable</code> for the
         * <code>PexFile</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private DebugFunction(DataInput input, StringTable strings) throws IOException {
            this.OBJECTNAME = strings.read(input);
            this.STATENAME = strings.read(input);
            this.FUNCNAME = strings.read(input);
            this.FUNCTYPE = input.readByte();

            int instructionCount = input.readUnsignedShort();
            this.INSTRUCTIONS = new ArrayList<>(instructionCount);

            for (int i = 0; i < instructionCount; i++) {
                this.INSTRUCTIONS.add(input.readUnsignedShort());
            }
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            this.OBJECTNAME.write(output);
            this.STATENAME.write(output);
            this.FUNCNAME.write(output);
            output.writeByte(this.FUNCTYPE);
            output.writeShort(this.INSTRUCTIONS.size());

            for (int instr : this.INSTRUCTIONS) {
                output.writeShort(instr);
            }
        }

        /**
         * Collects all of the strings used by the DebugFunction and adds them
         * to a set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            strings.add(this.OBJECTNAME);
            strings.add(this.STATENAME);
            strings.add(this.FUNCNAME);
        }

        /**
         * Generates a qualified name for the object of the form
         * "OBJECT.FUNCTION".
         *
         * @return A qualified name.
         *
         */
        public IString getFullName() {
            return IString.format("%s.%s", this.OBJECTNAME, this.FUNCNAME);
        }

        /**
         * Pretty-prints the DebugFunction.
         *
         * @return A string representation of the DebugFunction.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append(String.format("%s %s.%s (type %d): ", this.OBJECTNAME, this.STATENAME, this.FUNCNAME, this.FUNCTYPE));
            this.INSTRUCTIONS.forEach(instr -> buf.append(String.format("%04x ", instr)));
            return buf.toString();
        }

        final private TString OBJECTNAME;
        final private TString STATENAME;
        final private TString FUNCNAME;
        final private byte FUNCTYPE;
        final private List<Integer> INSTRUCTIONS;

    }

}
