/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.pex;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import restringer.IString;

/**
 * An abstraction describing a string table.
 *
 * @author Mark Fairchild
 * @version 2016/07/04
 */
final public class StringTable extends ArrayList<StringTable.TString> {

    /**
     * Creates a new <code>TString</code> by reading from a
     * <code>DataInput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @return The new <code>TString</code>.
     * @throws IOException
     */
    public TString read(DataInput input) throws IOException {
        Objects.requireNonNull(input);

        int index = input.readUnsignedShort();

        if (index < 0 || index >= this.size()) {
            throw new IOException(String.format("Invalid TString index: %d (size %d)", index, this.size()));
        }

        TString newString = this.get(index);
        return newString;
    }

    /**
     * @return Returns a reusable instance of a blank <code>TString</code>.
     */
    public TString blank() {
        return this.addString(IString.BLANK);
    }
    
    /**
     * Creates a new <code>DataInput</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @throws IOException
     */
    public StringTable(DataInput input) throws IOException {
        int strCount = input.readUnsignedShort();
        this.ensureCapacity(strCount);
        
        for (int i = 0; i < strCount; i++) {
            try {
                final String STR = input.readUTF();
                final TString TSTR = new TString(STR, i);
                this.add(TSTR);
            } catch (IOException ex) {
                throw new IOException("Error reading string #" + i, ex);
            }
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    public void write(DataOutput output) throws IOException {
            output.writeShort(this.size());

        for (TString tstr : this) {
            try {
                tstr.writeFull(output);
            } catch (IOException ex) {
                throw new IOException("Error writing string #" + tstr.INDEX, ex);
            }
        }
    }

    /**
     * Rebuilds the string table. This is necessary if ANY strings in ANY of the
     * Pex's members has changed at all. Otherwise, writing the Pex will produce
     * an invalid file.
     * 
     * @param inUse The <code>Set</code> of strings that are still in use.
     * 
     */
    public void rebuildStringTable(Set<TString> inUse) {
        this.retainAll(this);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    /*@Override
    public int calculateSize() {
        int sum = 0;

        if (this.size() > 0xFFF0) {
            sum += 6;
        } else {
            sum += 2;
        }

        sum += this.parallelStream().mapToInt(v -> v.calculateFullSize()).sum();
        return sum;
    }*/
    
    /**
     * Adds a new string to the <code>StringTable</code> and returns the
     * corresponding <code>TString</code>.
     *
     * @param val The value of the new string.
     * @return The new <code>TString</code>, or the existing one if the
     * <code>StringTable</code> already contained a match.
     */
    public TString addString(IString val) {
        Optional<TString> match = this.stream().filter(v -> v.equals(val)).findFirst();
        if (match.isPresent()) {
            return match.get();
        }

        TString tstr = new TString(val, this.size());
        this.add(tstr);
        return tstr;
    }

    /**
     * A case-insensitive string with value semantics that reads and writes as a
     * two-byte index into a string table.
     *
     * @author Mark Fairchild
     * @version 2016/07/04
     */
    static public class TString extends restringer.IString {

        /**
         * Creates a new <code>TString</code> from a character sequence and an
         * index.
         *
         * @param cs The <code>CharSequence</code>.
         * @param index The index of the <code>TString</code>.
         */
        private TString(CharSequence cs, int index) {
            super(cs);
            this.INDEX = index;
        }

        /**
         * @see WString#write(DataOutput)
         * @param output The output stream.
         * @throws IOException
         */
        public void writeFull(DataOutput output) throws IOException {
            output.writeUTF(super.toString());
        }

        /**
         * @see restringer.ess.Element#write(DataOutput)
         * @param output The output stream.
         * @throws IOException
         */
        public void write(DataOutput output) throws IOException {
            output.writeShort(this.INDEX);
        }

        final private int INDEX;

    }

}
