local AI = require "util/aicommon"

local Eve = Behavior("Eve")

Eve.editorIcon("eve/eve_down.json")

local OnHitQuips = {
  "Not cool.",
  "Dude, what the hell?"
}

local OnInteractQuips = {
  "Well, what are you waiting for? Get us out of here.",
  "You should probably look around."
}

function Eve:initialize()
  self.walkSpeed = 110.0

  self:addComponent("Character", {
    faction = "Heroes",
    hp = 100.0,
    movementSpeed = self.walkSpeed,
    collisionRadius = 24.0,
    colliderDisplacement = Vector(0.0, 12.0),
    colliderDensity = 128.0,
    skeleton = {
      s = "eve/eve_down.json",
      n = "eve/eve_up.json",
      w = "eve/eve_left.json",
      e = "eve/eve_right.json"
    }
  })

  self.critterTimeout = 0.0

  self:setupColliders()
  local collider = self.physicsCollider
  collider.categories = flags(World.layer("MovementCollider"), World.layer("Interactable"))

  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object
  })

  self.object:on("use", function()
    self:say(table.pickOne(OnInteractQuips))
  end)

  self.character:setSkeleton("n")
  CreateObject("FootstepsRenderer", Vector.zero(), self.object):addComponent("FootstepsRenderer")

  self.startIndex = 1 + math.floor(Random.value() * 3)
end

function Eve:awake()
  self:setupAI()
  self.heroRef = BehaviorUtil.findFirstOfType("Hero")
end

function Eve:setupAI()
  local lookAt = Co.create(function(transform)
    while true do
      self.character:lookAt(transform.worldPosition)
      Co.yield()
    end
  end)

  local idle = Co.create(function()
    self.character:setAnimation("idle", true)
    self.movable.velocity = Vector.zero()

    while true do
      local ally = self.character:findEntityInRange(400.0, function (object)
        if not object.character or object == self.object then
          return false
        end

        if object.character.faction == self.character.faction then
          return true
        end

        return false
      end)

      if not ally then
        Co.sleep(0.5)
      else
        Co.yieldRunWhile(function ()
          return Physics.lineOfSight(self.transform.worldPosition, ally.transform.worldPosition)
        end, Co.concurrent(
          lookAt(ally.transform),
          function()
            local position = self.transform.worldPosition

            while true do
              local diff = self.transform.worldPosition - position
              position = self.transform.worldPosition

              if diff:len() > 2.0 then
                self.character:setAnimation("walk", true)
              else
                self.character:setAnimation("idle", true)
              end

              Co.sleep(0.0)          
            end
          end
        ))
      end
    end
  end)

  self.idle = idle

  self.getHitFn = Co.create(function(enemy)
    self.soundEmitter:postEvent("EveGetHit")
    self.walkCo = nil

    if enemy.character.faction == self.character.faction then
      local quip = table.pickOne(OnHitQuips)
      self:say(quip)
    end

    self.movable.velocity = Vector.zero()
    self.character:setAnimation("get_hit", false)
    AI.yieldWaitForAnimation(self.object, "get_hit")

    self.character:setAnimation("walk", true)

    self.coMain = self.hubWorldRoute
    self.walkCo = self.walkAnimFn
  end)

  self.walkAnimFn = Co.create(function()
    local lastPos = self.transform.worldPosition

    while true do
      local pos = self.transform.worldPosition
      local diff = pos - lastPos
      lastPos = pos

      if diff:len() > 0.0 then
        self.character:setAnimation("walk", true)
      else
        self.character:setAnimation("idle", true)
      end

      Co.yield()
    end
  end)

  local cp1 = WorldUtil.getLandmarkByName("Eve1")
  local cp2 = WorldUtil.getLandmarkByName("Eve2")
  local cp3 = WorldUtil.getLandmarkByName("Eve3")
  local cp4 = WorldUtil.getLandmarkByName("Eve4")

  local index = self.startIndex

  self.hubWorldRoute = Co.create(function()
    while true do
      local critter = nil

      Co.yieldRunWhile(function()
        critter = self:findCritterInRange(500.0)

        return Vector.distance(self.transform.worldPosition, self.heroRef.transform.worldPosition) > 120.0
        and index <= 4 and (not critter or self.critterTimeout > 0.0)
      end, function()
        if index == 4 then
          self.movable.velocity = Vector.zero()
          self.character:setSkeleton("s")

          self.walkCo = nil
          self.character:setAnimation("cold", false)
          self.character:addAnimation("idle", true)
          Co.sleep(0.5, 1.0)
          self:say("I wish it was spring already.")
          Co.sleep(3.0, 6.0)
          self.walkCo = self.walkAnimFn
        elseif index == 3 then
          Co.yield(AI.goTo(self.movable, cp2.position))
          self.movable.velocity = Vector.zero()
          self.character:setSkeleton("s")
          Co.sleep(0.5, 1.0)
          self:say("Do you ever wonder what's out there?")
          Co.sleep(6.0, 8.0)
        elseif index == 2 then
          Co.yield(AI.goTo(self.movable, cp3.position))
          self.movable.velocity = Vector.zero()
          self.character:setSkeleton("n")
          Co.sleep(0.5, 1.0)
          self:say("You can restock at Joe's.. if he doesn't eat all the supplies.")
          Co.sleep(4.0)
          BehaviorUtil.findFirstOfType("TraderJoe"):say("Watch your tongue, young lady.")
        elseif index == 1 then
          Co.yield(AI.goTo(self.movable, cp4.position))
          self.movable.velocity = Vector.zero()
          self.character:setSkeleton("n")
          Co.sleep(0.5, 1.0)
          self:say("I think this is the portal to the arena they were talking about.")
          Co.sleep(4.0)
          self:say("I'd prepare well before going in there.")
          Co.sleep(4.0)
        end

        index = index + 1
        Co.sleep(5.0, 7.0)
      end)

      if critter and self.critterTimeout <= 0.0 then
        self.bodyHitbox:turnOff()
        critter.bodyHitbox:turnOff()
        critter.coMain = Co.create(function()
          while true do
            Co.yield()
          end
        end)
        critter.character:setAnimation("idle", true)

        Co.yield(AI.chase(self.movable, critter.transform, 64.0))
        self.movable.collisionAvoidance = false
        local mask = critter.movable.physicsCollider.mask
        critter.movable.physicsCollider.mask = 0
        critter.movable.velocity = Vector.zero()
        critter.character:setSkeleton("e")
        critter.skeletonRenderer.zOrder = 99

        local targetPosition = critter.transform.worldPosition + Vector(-32.0, 22.0)

        local blocked = false

        Co.yield(Co.concurrent(
          AI.goTo(self.movable, targetPosition, 4.0),
          function()
            while true do
              if self.movable.blocked then
                blocked = true
                break
              end

              Co.yield()
            end
          end
        ))

        if not blocked then 
          self.movable.velocity = Vector.zero()
          targetPosition = self.transform.worldPosition

          self.character:setSkeleton("e")
          self.movable.velocity = Vector.zero()
          self.walkCo = nil
          self.character:setAnimation("pet_start")
          self.character:addAnimation("pet_loop", true)

          Co.yieldConcurrent(
            Co.waitFor(4.0, 8.0),
            function ()
              while Vector.distance(self.transform.worldPosition, targetPosition) < 4.0 do
                Co.yield()
              end
            end
          )

          self.character:setAnimation("pet_end")
          AI.yieldWaitForAnimation(self.object, "pet_end")
          self.walkCo = self.walkAnimFn
        end

        critter.movable.physicsCollider.mask = mask
        critter.skeletonRenderer.zOrder = 100
        self.movable.collisionAvoidance = true
        self.critterTimeout = 30.0        
        self.bodyHitbox:turnOn()
        critter.bodyHitbox:turnOn()
      end

      index = 1
      self.character:lookAt(self.heroRef.transform.worldPosition)
      self.movable.velocity = Vector.zero()

      while Vector.distance(self.transform.worldPosition, self.heroRef.transform.worldPosition) < 120.0 do
        self.character:lookAt(self.heroRef.transform.worldPosition)
        Co.sleep(0.1)
      end
    end
  end)

  self.walkCo = self.walkAnimFn
  self.coMain = self.idle
end

function Eve:fixedUpdate()
  if self.critterTimeout > 0.0 then
    self.critterTimeout = self.critterTimeout - Time.fixedDeltaTime
  end

  if self.coMain then
    if not self.coMain:update() then
      self.hubWorldRoute:reset()
      self.coMain = self.hubWorldRoute
    end
  end

  if self.textCo then
    if not self.textCo:update() then
      self.textCo = nil
    end
  end

  if self.walkCo then
    if not self.walkCo:update() then
      self.walkAnimFn:reset()
      self.walkCo = self.walkAnimFn
    end
  end
end

function Eve:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end

  self.bodyHitbox.onDealDamage = function(damage)
  end
end

function Eve:createGoToCo(position)
  return Co.create(function()
    self.character:setAnimation("walk", true)
    Co.yield(AI.goTo(self.movable, position))
    self.character:setAnimation("idle", true)
    self.movable.velocity = Vector.zero()
  end)
end

function Eve:say(text)
  if self.bubble then
    self.bubble.object:destroy()
  end

  self.bubble = CreateObject("SpeechBubble", Vector(8.0, 114.0), self.object):addComponent("SpeechBubble")
  self.bubble:setText(text)

  self.textCo = Co.create(function ()
    while not self.bubble.done do
      Co.yield()
    end

    Co.sleep(3.0)

    self.bubble.object:destroy()
    self.bubble = nil
  end)
end

function Eve:findCritterInRange(range)
  local object = self.character:findEntityInRange(range, function(object)
    if not object.forestCritter then
      return false
    end

    return true
  end)

  if not object then
    return nil
  end

  return object.forestCritter
end