local LockOnWidget = Behavior("LockOnWidget")

function LockOnWidget:initialize()
  self.lockOn = nil
  self.lockOnRange = 300.0
  self.lockOnLastPosition = Vector()

  self.lockOnImage = AssetUtil.find("default/lockon_ring.png")
  self.lockOnInnerImage = AssetUtil.find("default/lockon_ring_inner.png")

  self.lockMoveDir = Vector()
  Kernel.inputController:on("moveLockOn", function (event)
    self.lockMoveDir = event.value

    if math.epsCompare(event.value:len(), 0.0, 0.1) then
      self.lockWait = false
    end

    event.handled = true
  end)

  Kernel.inputController:on("lockOn", function (event)
    if self.lockOn then
      self.lockOn = nil
      return
    end

    self.lockOn = self:findLockOnTarget()
    event.handled = true
  end)

  Kernel.inputController:on("lockOnClosest", function (event)
    self.lockOn = self:findLockOnTarget(true)
    event.handled = true
  end)
end

function LockOnWidget:fixedUpdate()
  if self.lockOn then
    if self.lockOn.character and not self.lockOn.character.canLock then
      self.lockOn = nil
    elseif not self.lockOn.health or not self.lockOn.health.isAlive then
      self.lockOn = self:findLockOnTarget()
    else
      self.lockOnLastPosition = self.lockOn.transform.worldPosition
      local dist = Vector.distance(self.transform.worldPosition, self.lockOnLastPosition)
      if dist > self.lockOnRange then
        self.lockOn = self:findLockOnTarget()
        if self.lockOn then
          self.lockOnLastPosition = self.lockOn.transform.worldPosition
        end
      end
    end
  else
    self.lockOnLastPosition = self.transform.worldPosition
  end
end

function LockOnWidget:update()
  if self.lockOn then
    self:drawWidget()
  end
end

function LockOnWidget:findLockOnTarget(onlyClosestDistance)
  local closestDistanceTarget = nil
  local closestDistance = self.lockOnRange * 2.0

  local closestAngleTarget = nil
  local closestAngle = 1.0

  local myPos = self.transform.worldPosition

  self.character:forEachEntityInRange(self.lockOnRange, function(object)
    local pos = object.transform.worldPosition
    if not Physics.lineOfSight(myPos, pos) then
      return true
    end

    if not object.character or object.character.faction == self.character.faction or not object.character.canLock then
      return true
    end

    if not object.health or not object.health.isAlive then
      return true
    end

    local angle = (pos - myPos):normalized():dot(self.character.lookDir)
    if math.epsCompare(angle, 1.0, 0.05) and math.abs(angle - 1.0) < math.abs(closestAngle) and not onlyClosestDistance then
      closestAngle = angle
      closestAngleTarget = object
    end

    local distance = Vector.distance(pos, myPos)
    if distance < closestDistance then
      closestDistance = distance
      closestDistanceTarget = object
    end

    return true
  end, World.layer("MovementCollider"))

  return closestAngleTarget or closestDistanceTarget
end

function LockOnWidget:drawWidget()
  local size = Vector(64.0, 32.0)
  local scale = 1.0

    if self.lockOn.character.properties.isLarge then
    scale = 2.0
  end

  Graphics.drawImage(self.lockOnImage, {
    position = self.lockOn.transform.worldPosition - size * 0.5 * scale,
    size = size * scale,
    tint = Color(1.0, 0.0, 0.0, 0.6),
    zOrder = 0
  })

  local lockLength = self.lockMoveDir:len()
  if lockLength < 0.25 or self.lockWait then
    return
  end

  local lockPos = self.lockOn.transform.worldPosition

  local smallestAngle = 100.0
  local newTarget = nil
  local myPosition = self.transform.worldPosition
  local targetDistance = 1000.0

  self.character:forEachEntityInRange(self.lockOnRange, function(object)
    if not object.character or object.character.faction == self.character.faction 
      or object == self.lockOn then
      return true
    end

    local pos = object.transform.worldPosition
    local distance = Vector.distance(myPosition, pos)
    if distance > self.lockOnRange then
      return true
    end

    local dir = (pos - lockPos):normalized()
    local angle = self.lockMoveDir:dot(dir * -1.0)

    if math.epsCompare(angle, smallestAngle, 0.1) then
      if distance < targetDistance then
        smallestAngle = angle
        newTarget = object
        targetDistance = distance
      end
    elseif angle < smallestAngle then
      smallestAngle = angle
      newTarget = object
      targetDistance = distance
    end

    return true
  end, World.layer("MovementCollider"))

  if not newTarget then
    return
  end

  local cursorPosition = lockPos + (newTarget.transform.worldPosition - lockPos) * (lockLength - 0.25) * 1.33

  if lockLength >= 0.95 then
    self.lockOn = newTarget
    self.lockWait = true
  end

  Graphics.drawImage(self.lockOnInnerImage, {
    position = cursorPosition - size * 0.5,
    size = size,
    tint = Color(1.0, 0.0, 0.0, 0.6),
    zOrder = 0
  })
end
