local SaveGameStats = require "util/savegamestats"

local ConsumableHelper = require "util/consumablehelper"
local Runes = require "util/runes"
local SaveGameManager = require "util/savegamemanager"

local Hero = Behavior("Hero")

Hero.property("spawnDirection", "s")

Hero.editorIcon("hero/hero_down.json")

function Hero:initialize(properties)
  self.properties = Hero.defaultProperties(properties)

  Kernel:addComponent("HUD")
  Kernel:addComponent("InventoryView")
  Kernel:addComponent("ContainerView")
  Kernel:addComponent("TradingView")
  Kernel:addComponent("AltarView")
  Kernel:addComponent("PauseMenu")

  self:addComponent("Actor")
  self:addComponent("Inventory")
  self:addComponent("QuickBar")
  self:addComponent("LockOnWidget")

  self.indicator = CreateObject("InteractionIndicator", Vector.zero()):addComponent("InteractionIndicator")
  self.cameraFollower = CreateObject("CameraFollow", Vector.zero()):addComponent("CameraFollow", self.transform)
  self.footsteps = CreateObject("FootstepsRenderer", Vector.zero(), self.object):addComponent("FootstepsRenderer")
  self.footsteps.maxFootsteps = 128

  self:setupColliders()

  self.defaultWalkSpeed = 130.0
  self.walkSpeed = self.defaultWalkSpeed

  self.baseWeaponDamage = 65.0
  self.canChangeRunes = false

  local skeletons = {
    s = "hero/hero_down.json",
    sw = "hero/hero_right_down.json",
    w = "hero/hero_left.json",
    nw = "hero/hero_right_up.json",
    n = "hero/hero_up.json",
    ne = "hero/hero_left_up.json",
    e = "hero/hero_right.json",
    se = "hero/hero_left_down.json"
  }

  self.defaultHp = 200.0

  self:addComponent("Character", {
    isEightSided = true,
    skeleton = skeletons,
    faction = "Heroes",
    hp = self.defaultHp,
    movementSpeed = self.walkSpeed,
    collisionAvoidance = false,
    collisionRadius = 24.0,
    colliderDisplacement = Vector(0.0, 12.0),
    colliderDensity = 128.0,
    colliders = {
      bb_body = self.bodyHitbox.object,
      bb_sword = self.sword.object
    }
  })

  self.physicsCollider.categories = flags(World.layer("MovementCollider"), World.layer("Hero"))

  self.object:on("animationEvent", function(trackId, animationName, event)
    if event == "EvadeStopMovement" then
      self.movable.velocity = Vector.zero()
    end
  end)

  self.soundEmitter:setSwitch("Type", "Hero")
  self.character:setSkeleton(self.properties.spawnDirection)
  self.character.lookDir = Vector(0.0, 1.0)

  self:setupActor()

  self.moveDir = Vector()
  Kernel.inputController:on("movement", function (event)
    self.moveDir = event.value
    event.handled = true
  end)

  self.wantsToBlock = false
  Kernel.inputController:on("block", function (event)
    if Kernel.menuEvents.lock then
      return
    end

    self.wantsToBlock = event.value
    event.handled = true
  end)

  Kernel.inputController:on("attack", function (event)
    if Kernel.menuEvents.lock then
      return
    end

    self.queuedAction = "attack"
    event.handled = true
  end)

  Kernel.inputController:on("evade", function (event)
    if Kernel.menuEvents.lock then
      return
    end

    self.queuedAction = "evade"
    event.handled = true
  end)

  Kernel.inputController:on("interact", function (event)
    if Kernel.menuEvents.lock then
      return
    end

    self.queuedAction = "use"
    event.handled = true
  end)

  self.totalStamina = 150.0
  self.stamina = self.totalStamina

  self.staminaRegenTimer = 0.0
  self.staminaRegenRate = 1.0
  self.staminaCostMultiplier = 1.0
  self.canAttack = true

  if _EDITOR then
    Input.onKeyDown(KEYS.KEY_G, function()
      self.godMode = not self.godMode
    end)
  end

  local pos = self.transform.worldPosition
  for i=1,32 do
    _L._visibilityMapUpdate(pos.x, pos.y, 400.0)
  end

  self.object:on("useConsumable", function (itemId)
    if not self.inventory.items[itemId] then
      return
    end

    if not self.usingConsumable then
      self.actor:stopAllScripts()
      self.actor:playScript("useConsumable", itemId)
    end
  end)

  self:registerSaveGameHandlers()
end

function Hero:awake()
  self.cameraFollower:snapToPosition()
end

function Hero:setupColliders()
  self.sword = CreateObject("Sword", Vector.zero(), self.object):addComponent("Weapon")
  self.sword:listenForAnimationEvents(self)

  self.sword.onEnvironmentHit = function(collisionInfo)
    self.soundEmitter:postEvent("HeroHitEnvironment")
  end

  self.sword.onApplyEffects = function (object)
    local statusEffects = object.statusEffects
    if not statusEffects then
      return
    end

    if Runes.isEquipped("Venom") then
      local rune = Runes.getRune("Venom")
      local dmg = self:calculateWeaponDamage()
      dmg = dmg * (rune.dmgPercent / 100.0) * rune.duration
      statusEffects:applyByName("Poison", { amount = dmg, duration = rune.duration })
    end

    if Runes.isEquipped("ElsasTouch") then
      if Random.value() <= 0.25 then
        statusEffects:applyByName("Frozen", { duration = 3.0 })
      end
    end
  end

  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:on("collisionStart", function(collisionInfo)
    self:handleBodyCollision(collisionInfo)
  end)
end

function Hero:setupActor()
  local actor = self.actor

  actor:addScript("idle", function()
    self.character:setAnimation("idle", true)
    self.movable.velocity = Vector.zero()
    self.soundEmitter:setSwitch("State", "Idle")

    while true do
      while Kernel.menuEvents.lock do
        WaitForNumberOfFrames(0)
      end

      if self.wantsToBlock then
        self.queuedAction = "block"
      end

      if self.queuedAction then
        if self:executeQueuedAction() then
          return
        end
      end

      if self.lockOnWidget.lockOn then
        self.character.lookDir = (self.lockOnWidget.lockOn.transform.worldPosition - self.transform.worldPosition):normalized()
        self.character:setOrientedSkeleton()
      end

      if self.moveDir:len() > 0.0 then
        local position = self.transform.worldPosition + Vector(0.0, 12.0)
        local velocity = self.moveDir * 26.0
        local hitWall = false
        Physics.raycast(position, position + velocity, function(hitInfo)
          if hitInfo.object == self.object then
            return -1.0
          end

          hitWall = true
          return 0.0
        end, flags(World.layer("World"), World.layer("MovementCollider")))

        if not hitWall then
          return actor:playScript("walk")
        end
      end

      WaitForNumberOfFrames(0)
    end
  end)

  actor:addScript("runeFound", function()
    self.movable.velocity = Vector.zero()
    self.soundEmitter:postEvent("RuneFound")

    self.character:setSkeleton("s")

    self.skeletonRenderer.timeScale = 0.75
    self.character:setAnimation("rune")
    WaitForAnimation("rune")
    self.skeletonRenderer.timeScale = 1.0

    return actor:playScript("idle")
  end)

  actor:addScript("useConsumable", function(itemId)
    self.usingConsumable = true

    self.movable.velocity = Vector.zero()
    self.soundEmitter:setSwitch("State", "UsingConsumable")
    self.soundEmitter:postEvent("QuickUseConsumable")

    if self.character.lookDir.x > 0.0 then
      self.character.lookDir = Vector(1.0, 0.0)
    else
      self.character.lookDir = Vector(-1.0, 0.0)
    end

    self.character:setOrientedSkeleton()

    self.skeletonRenderer.timeScale = 1.65
    self.character:setAnimation("potion")
    WaitForAnimation("potion")
    self.skeletonRenderer.timeScale = 1.0

    if self.inventory:removeItems(itemId, 1) < 1 then
      self.usingConsumable = false
      return
    end

    ConsumableHelper.useConsumable(self, itemId)
    Kernel.inventoryView:refreshItems()
    self.usingConsumable = false
    return actor:playScript("idle")
  end)

  actor:addScript("walk", function()
    self.soundEmitter:setSwitch("State", "Walking")
    local nextFootStepTimer = 0.0

    while true do
      if Kernel.menuEvents.lock then
        return actor:playScript("idle")
      end

      if self.wantsToBlock then
        self.queuedAction = "block"
      end

      if self.queuedAction then
        if self:executeQueuedAction() then
          self.skeletonRenderer:setTimeScale(1.0)
          return
        end
      end

      if self.moveDir.x == 0.0 and self.moveDir.y == 0.0 then
        self.skeletonRenderer:setTimeScale(1.0)
        return actor:playScript("idle")
      end

      self.movable.velocity = self.moveDir:clone()

      if not self.lockOnWidget.lockOn then
        if self.moveDir.x ~= 0.0 or self.moveDir.y ~= 0.0 then
          self.character.lookDir = self.moveDir:normalized()
          self.character:setOrientedSkeleton()
        end
      else
        self.character.lookDir = (self.lockOnWidget.lockOn.transform.worldPosition - self.transform.worldPosition):normalized()
        self.character:setOrientedSkeleton()
      end

      local multiplier = 1.0

      if Runes.isEquipped("SprintBurst") then
        local rune = Runes.getRune("SprintBurst")
        if rune.active then
          multiplier = rune.multiplier
        end
      end

      if Runes.isEquipped("Lighting") then
        self.stamina = self.stamina - Time.fixedDeltaTime * 14.0
        self.staminaRegenTimer = 1.0
        if self.stamina < 0.0 then
          multiplier = multiplier * 0.25
        else
          multiplier = multiplier * 1.4
        end
      end

      local position = self.transform.worldPosition + Vector(0.0, 12.0)
      local velocity = self.movable.velocity * 26.0
      local hitWall = false
      Physics.raycast(position, position + velocity, function(hitInfo)
        if hitInfo.object == self.object then
          return -1.0
        end

        hitWall = true
        return 0.0
      end, World.layer("World"))

      if hitWall then
        self.character:setAnimation("idle", true)
        self.movable.velocity = Vector.zero()
      else
        self.character:setAnimation("walk", true)
      end

      local factor = self.movable.velocity:dot(self.character.lookDir)
      self.movable.movementSpeed = self.walkSpeed * math.clamp(factor, 0.75, 1.0) * multiplier

      local speed = self.movable.velocity:len()

      nextFootStepTimer = nextFootStepTimer - Time.fixedDeltaTime
      if nextFootStepTimer < 0.0 and speed > 0.0 then
        self.soundEmitter:postEvent("Footstep")
        nextFootStepTimer = 0.39 / speed
      end

      if not hitWall then
        self.skeletonRenderer:setTimeScale(speed)
      end

      WaitForNumberOfFrames(0)
    end
  end)

  actor:addScript("dragonshout", function()
    self.movable.velocity = Vector.zero()

    local lookDir = self.character.lookDir

    if math.abs(lookDir.x) > math.abs(lookDir.y) then
      if lookDir.x < 0.0 then
        lookDir = Vector(-1.0, 0.0)
      else
        lookDir = Vector(1.0, 0.0)
      end
    else
      if lookDir.y < 0.0 then
        lookDir = Vector(0.0, -1.0)
      else
        lookDir = Vector(0.0, 1.0)
      end
    end

    self.stamina = self.stamina - 75.0 * self.staminaCostMultiplier

    self.character.lookDir = lookDir
    self.character:setOrientedSkeleton()
    self.character:setAnimation("jump_back")

    self.character:forEachEntityInRange(300.0, function (object)
      if object == self.object then
        return true
      end

      local statusEffects = object.statusEffects
      if not statusEffects then
        return true
      end

      local dir = (object.transform.worldPosition - self.transform.worldPosition):normalized()
      statusEffects:applyByName("Knockback", { impulse = dir * 6000.0 })
      return true
    end, World.layer("MovementCollider"))

    WaitForAnimation("jump_back")
    self.staminaRegenTimer = 1.0

    return actor:playScript("idle")
  end)

  actor:addScript("evade", function()
    if Runes.isEquipped("Dragonshout") then
      return actor:playScript("dragonshout")
    end

    self.isBlocking = false

    local lookDir = self.character.lookDir

    if math.abs(lookDir.x) > math.abs(lookDir.y) then
      if lookDir.x < 0.0 then
        lookDir = Vector(-1.0, 0.0)
      else
        lookDir = Vector(1.0, 0.0)
      end
    else
      if lookDir.y < 0.0 then
        lookDir = Vector(0.0, -1.0)
      else
        lookDir = Vector(0.0, 1.0)
      end
    end

    self.stamina = self.stamina - 20.0 * self.staminaCostMultiplier

    self.movable.velocity = self.moveDir
    if self.movable.velocity:len() == 0.0 then
      self.movable.velocity = -1.0 * self.character.lookDir
    end

    self.character.lookDir = lookDir
    self.character:setOrientedSkeleton()
    self.character:setAnimation("jump_back")
    self.skeletonRenderer:setTimeScale(1.5)
    self.soundEmitter:postEvent("HeroEvade")

    self.movable.movementSpeed = 180.0

    self.bodyHitbox:turnOff()

    WaitForAnimation("jump_back")

    self.bodyHitbox:turnOn()

    self.skeletonRenderer:setTimeScale(1.0)
    self.movable.movementSpeed = self.walkSpeed
    self.movable.velocity = Vector.zero()
    self.staminaRegenTimer = 0.75
    return actor:playScript("idle")
  end)

  actor:addScript("block", function()
    self.movable.velocity = Vector.zero()

    self.isBlocking = true
    self.skeletonRenderer.timeScale = 2.5
    self.character:setAnimation("block_start", false)
    self.character:addAnimation("block_active", true)
    WaitForAnimation("block_start")
    self.skeletonRenderer.timeScale = 1.0
    self.soundEmitter:setSwitch("State", "Blocking")

    local moving = false
    local nextFootStepTimer = 0.0
    local changeDirTimer = 0.05

    while true do
      self.movable.velocity = self.moveDir
      local factor = self.movable.velocity:dot(self.character.lookDir)
      self.movable.movementSpeed = self.walkSpeed * 0.7 * math.clamp(factor, 0.7, 1.0)

      if self.queuedAction then
        if self:executeQueuedAction() then
          self.skeletonRenderer:setTimeScale(1.0)
          return
        end
      end

      changeDirTimer = changeDirTimer - Time.fixedDeltaTime
      if changeDirTimer > 0.0 and self.moveDir:len() > 0.0 then
        self.character.lookDir = self.moveDir
        self.character:setOrientedSkeleton()
      end

      if self.lockOnWidget.lockOn then
        self.character.lookDir = (self.lockOnWidget.lockOn.transform.worldPosition - self.transform.worldPosition):normalized()
        self.character:setOrientedSkeleton()
      end

      local speed = self.moveDir:len()

      if speed > 0.0 and not moving then
        self.character:setAnimation("block_active_move", true)
        moving = true
      elseif speed == 0.0 and moving then
        self.character:setAnimation("block_active", true)
        moving = false
      end

      if not self.wantsToBlock then
        self.skeletonRenderer.timeScale = 1.5
        self.character:setAnimation("block_release", false)
        WaitForAnimation("block_release")
        self.skeletonRenderer.timeScale = 1.0
        self.queuedAction = nil
        self.isBlocking = false
        return actor:playScript("idle")
      end

      nextFootStepTimer = nextFootStepTimer - Time.fixedDeltaTime
      if nextFootStepTimer < 0.0 and speed > 0.0 then
        self.soundEmitter:postEvent("Footstep")
        nextFootStepTimer = 0.75 / speed
      end
        
      WaitForNumberOfFrames(0)
    end

    self.isBlocking = false
  end)

  actor:addScript("attack", function()
    self.soundEmitter:setSwitch("State", "Attacking")

    local dmg = self:calculateWeaponDamage()

    self.sword:startAttack(dmg)

    local hitSomething = false
    self.sword.onHitConnected = function(object, damage)
      hitSomething = true
      if Runes.isEquipped("Vampirism") then
        self.health.hp = math.min(self.health.hp + damage * 0.10, self.health.totalHp)
      end
    end

    self.movable.velocity = Vector.zero()
    self.soundEmitter:postEvent("HeroSwordAttack")

    local lookDir = self.moveDir

    if not self.lockOnWidget.lockOn then
      if lookDir.x == 0.0 and lookDir.y == 0.0 then
        lookDir = self.character.lookDir
      end
    else
      lookDir = (self.lockOnWidget.lockOn.transform.worldPosition - self.transform.worldPosition):normalized()
    end

    lookDir:normalize()

    self.character.lookDir = lookDir
    self.character:setOrientedSkeleton()

    local animName = "hit"
    local skeleton = self.character.currentSkeleton

    if (skeleton == "w" or skeleton == "e") and Random.value() < 0.3 then
      animName = "hit_var1"
    end

    self.character:setAnimation(animName, false)
    self.movable.velocity = lookDir
    self.movable.movementSpeed = self.walkSpeed * 0.5

    local pushForwardTimer = 0.25
    while pushForwardTimer > 0.0 do
      pushForwardTimer = pushForwardTimer - Time.fixedDeltaTime
      WaitForNumberOfFrames(0)
    end

    self.movable.velocity = Vector.zero()

    WaitForAnimation(animName)
    self.skeletonRenderer:setTimeScale(1.0)

    self.character:addAnimation("idle", true)
    Co.sleep(0.1)

    if Runes.isEquipped("Bloodthirst") then
      if not hitSomething then
        self.health:takeDamage(self:calculateWeaponDamage() * 0.5)
        if not self.health.isAlive then
          return actor:playScript("getHit")
        end
      end
    end

    self.sword.onHitConnected = nil
    actor:playScript("idle")
  end)

  actor:addScript("failedAttack", function()
    self.movable.velocity = Vector.zero()
    self.soundEmitter:postEvent("HeroFailedAttack")

    local lookDir = self.character.lookDir

    self.character:setAnimation("hit_fail")
    WaitForAnimation("hit_fail")
    actor:playScript("idle")
  end)

  self.gettingHit = false

  actor:addScript("getHit", function()
    self.queuedAction = nil

    if Runes.isEquipped("SprintBurst") then
      local rune = Runes.getRune("SprintBurst")
      rune:onGetHit()
    end

    self.soundEmitter:postEvent("HeroGotHit")
    self.movable.velocity = Vector.zero()

    if self.health.isAlive then
      self.character:setAnimation("get_hit", false, true)
      WaitForAnimation("get_hit")
      return actor:playScript("idle")
    end

    self.soundEmitter:postEvent("HeroOnDeath")
    self.soundEmitter:setSwitch("State", "Dying")
    self.indicator:hide()
    SaveGameStats.increase("timesDied")

    WaitUntilFadeOut(self.skeletonRenderer)
  end)

  actor:playScript("idle")

  self.updateVisibilityCo = Co.create(function()
    local acc = 0.0

    while true do
      local range = 400.0
      if Runes.isEquipped("Farsight") then
        local rune = Runes.getRune("Farsight")
        range = range * rune.multiplier
      end

      local pos = self.transform.worldPosition
      _L._visibilityMapUpdate(pos.x, pos.y, range)
      acc = Co.sleep(1.0 / 60.0 + acc)
    end
  end)

  self.staminaCo = Co.create(function()
    while true do
      while not self.health.isAlive do
        Co.yield()
      end

      if self.staminaRegenTimer > 0.0 then
        self.staminaRegenTimer = self.staminaRegenTimer - Time.fixedDeltaTime
        if self.staminaRegenTimer < 0.0 then
          self.staminaRegenTimer = 0.0
        end
      else
        if self.stamina < self.totalStamina then
          local regenRate = 128.0 * math.max(self.stamina / self.totalStamina, 0.3) * self.staminaRegenRate
          self.stamina = self.stamina + Time.fixedDeltaTime * regenRate

          if self.stamina > self.totalStamina then
            self.stamina = self.totalStamina
          end

          if self.stamina >= 25.0 then
            self.canAttack = true
          end
        end
      end

      Co.yield()
    end
  end)

  self.checkForActiveObjectCo = Co.create(function()
    while true do
      local activeObject = nil

      local lookDir = self.character.lookDir * 24.0
      local size = Vector(96.0, 96.0)
      local position = self.transform.worldPosition + Vector(0.0, size.y * 0.5) + lookDir

      local bottomLeft = position - size * 0.5
      local topRight = position + size * 0.5
      
      --Debug.drawRect(bottomLeft, topRight - bottomLeft, 0xFF00FFAA, true)

      Physics.aabbQuery(bottomLeft, topRight, function(object)
        if not Physics.lineOfSight(self.transform.worldPosition, object.transform.worldPosition) then
          return true
        end

        activeObject = object
        return false
      end, World.layer("Interactable"))

      if self.activeObject ~= activeObject then
        if self.activeObject ~= nil then
          if self.activeObject:hasAnyListeners("deactivate") then
            self.activeObject:emit("deactivate", self)
          end

          self.indicator:hide()
        end

        self.activeObject = activeObject
        if self.activeObject ~= nil then
          if self.activeObject:hasAnyListeners("activate") then
            self.activeObject:emit("activate", self)
          end

          local small = false
          if activeObject.itemPouch then
            small = true
          end
          
          self.indicator:show(small)
        end
      end

      if self.activeObject then
        local bounds = self.activeObject.transform.worldBounds
        local center = Vector((bounds.bottomLeft.x + bounds.topRight.x) * 0.5, bounds.topRight.y)
        center = center + Vector(0.0, 24.0)

        if self.activeObject.traderJoe then
          center = center + Vector(10.0, 4.0)
        elseif self.activeObject.altar then
          center = center + Vector(-16.0, 4.0)
        elseif self.activeObject.eve then
          center = center + Vector(0.0, 4.0)
        end

        self.indicator.transform:setLocalPositionAndReset(center)
      end

      Co.yield()
    end
  end)
end

function Hero:fixedUpdate()
  Runes.update()
  Sound.setListenerPosition(self.transform.worldPosition, self.character.lookDir)

  if self.updateVisibilityCo then
    if not self.updateVisibilityCo:update() then
      self.updateVisibilityCo = nil
    end
  end

  if self.checkForActiveObjectCo then
    if not self.checkForActiveObjectCo:update() then
      self.checkForActiveObjectCo = nil
    end
  end

  if not self.staminaCo:update() then
    self.staminaCo:reset()
  end
end

function Hero:update()
  if self.textCo then
    if not self.textCo:update() then
      self.textCo = nil
    end
  end
end

function Hero:executeQueuedAction()
  local action = self.queuedAction
  self.queuedAction = nil

  self.actionTimeout = true
  setTimeout(function()
    self.actionTimeout = nil
  end, 0.25)

  if action == "attack" then
    local staminaCost = 35.0 * self.staminaCostMultiplier

    if Runes.isEquipped("Focused") and Random.value() <= 0.5 then
      staminaCost = 0.0
    end

    if Runes.isEquipped("Bloodthirst") then
      staminaCost = 0.0
    end

    if self.stamina < staminaCost and Runes.isEquipped("SecondWind") and Random.value() <= 0.25 then
      self.stamina = self.totalStamina
    end

    if Runes.isEquipped("Relentless") and self.stamina < staminaCost and self.health.hp > staminaCost then
      self.health:takeDamage(staminaCost)
      self.actor:playScript("attack")
      self.staminaRegenTimer = 1.0
    elseif self.stamina >= staminaCost and self.canAttack then
      self.stamina = self.stamina - staminaCost
      
      self.staminaRegenTimer = 1.0

      if self.stamina < 0.0 then
        self.stamina = 0.0
        self.staminaRegenTimer = 1.5
        self.canAttack = false
      end

      self.actor:playScript("attack")
    else
      self.actor:playScript("failedAttack")
    end
  elseif action == "block" then
    if self.stamina >=0 and self.canAttack then
      self.actor:playScript("block")
    else
      self.actionTimeout = nil
      return false
    end
  elseif action == "use" then
    if self.activeObject and self.activeObject:hasAnyListeners("use") then
      self.character:lookAt(self.activeObject.transform.worldPosition)
      self.activeObject:emit("use", self)
    end

    return false
  elseif action == "evade" then
    local staminaCost = 20.0
    if Runes.isEquipped("Dragonshout") then
      staminaCost = 60.0
    end

    if self.stamina >= staminaCost then
      self.staminaRegenTimer = 1.0
      self.actor:playScript("evade")
      return true
    end

    return false
  else
    return false
  end

  return true
end

function Hero:calculateWeaponDamage()
  local dmg = self.baseWeaponDamage
  if self.berserk then
    dmg = dmg + 25
  end

  if Runes.isEquipped("EnergySurge") then
    if self.health.totalHp > 0.0 and self.health.hp / self.health.totalHp <= 0.3 then
      dmg = dmg * 1.5
    end
  end

  return dmg
end

function Hero:handleBodyCollision(collisionInfo)
  if not collisionInfo.object.weapon or not self.health.isAlive then
    return
  end

  if self.statusEffects:hasStatusEffect("Frozen") then
    return
  end

  local weapon = collisionInfo.object.weapon
  if weapon:shouldConnect(self.object) then    
    local blockSuccess = false

    local myPosition = self.transform.worldPosition

    local hitDir
    local parent = weapon.transform.parent

    if weapon.movementDirection then
      hitDir = weapon.movementDirection * -1.0
    elseif parent then
      hitDir = (parent.worldPosition - myPosition):normalized()
    else
      hitDir = (weapon.transform.worldPosition - myPosition):normalized()
    end

    if self.isBlocking then
      local lookDir = self.character.lookDir:normalized()
      local angle = Vector.angle(lookDir, hitDir)
      blockSuccess = angle < 0.75
    end

    if not blockSuccess then
      local bloodDirection = hitDir * -1.0
      CreateObject("BloodParticles", collisionInfo.pointSelf):addComponent("BloodParticles", bloodDirection)
      
      if not self.godMode then
        local dmg = weapon.damage
        if self.berserk then
          dmg = dmg * 1.5
        end

        self.health:takeDamage(dmg)
      end

      if weapon.onApplyEffects then
        weapon.onApplyEffects(self.object)
      end

      if Runes.isEquipped("EldrichInsanity") then
        local parent = weapon.transform.parent.object
        if parent and parent.character then
          if Random.value() < 0.05 then
            parent.character.faction = self.character.faction
            parent:emit("resetTarget")
          end
        end
      end

      local parent = nil
      if weapon.transform.parent then
        parent = weapon.transform.parent.object
      else
        parent = weapon.object
      end

      if not (Runes.isEquipped("Guzzler") and self.usingConsumable) then
        self.actor:stopAllScripts()
        self.actor:playScript("getHit")
        self.usingConsumable = false

        local dir = (self.transform.worldPosition - parent.transform.worldPosition):normalized()
        self.statusEffects:applyByName("Knockback", { impulse = dir * 1500.0 })
      end

      if Runes.isEquipped("Coldblooded") then
        if parent.statusEffects then
          parent.statusEffects:applyByName("Slow", { amount = 0.95, duration = 4.0 })
        end
      end
    else
      local parent = nil
      if weapon.transform.parent then
        parent = weapon.transform.parent.object
      end

      if not parent then
        parent = weapon.object
      end

      local dir = (self.transform.worldPosition - parent.transform.worldPosition):normalized()
      self.statusEffects:applyByName("Knockback", { impulse = dir * 800.0 })

      self.soundEmitter:postEvent("HeroBlockedAttack")

      local staminaCost = weapon.damage * weapon.staminaMultiplier
      if Runes.isEquipped("Knighted") then
        staminaCost = staminaCost * 0.5
      end

      self.stamina = self.stamina - staminaCost
      self.staminaRegenTimer = 1.0

      if self.stamina < 0.0 then
        local dmg = weapon.damage + self.stamina
        if dmg > 0.0 then
          self.health:takeDamage(dmg)

          local dir = (self.transform.worldPosition - parent.transform.worldPosition):normalized()
          self.statusEffects:applyByName("Knockback", { impulse = dir * 1500.0 })

          self.actor:stopAllScripts()
          self.actor:playScript("getHit")
        else
          self.actor:playScript("idle")
        end

        self.usingConsumable = false
        self.stamina = 0.0
        self.staminaRegenTimer = 1.5
        self.canAttack = false
        self.isBlocking = false
      end
    end
  end
end

function Hero:registerSaveGameHandlers()
  SaveGameManager:on("saveGame", function(saveGame)
    local inventory = {
      items = self.inventory.items,
      coins = self.inventory.coins
    }

    saveGame:setKey("inventory", inventory)

    local equippedRunes = {}
    for runeName, _ in pairs(Runes.equipped) do
      equippedRunes[#equippedRunes+1] = runeName
    end
    saveGame:setKey("equippedRunes", equippedRunes)

    saveGame:setKey("offeredRunes", Runes.offered)

    local quickbar = self.quickBar.slots
    saveGame:setKey("quickbar", quickbar)

    local statusEffects = self.statusEffects:serialize()
    saveGame:setKey("statusEffects", statusEffects)

    local arenaInfo = self.arenaInfo or { current = 0 }
    saveGame:setKey("arenaInfo", arenaInfo)
  end)

  SaveGameManager:on("loadGame", function(saveGame)
    local inventory = saveGame:getKey("inventory")
    if inventory then
      self.inventory.items = inventory.items
      self.inventory.coins = inventory.coins
    end

    local equippedRunes = saveGame:getKey("equippedRunes")
    if equippedRunes then
      for i, runeName in ipairs(equippedRunes) do
        Runes.equip(runeName, self)
      end
    end

    local offeredRunes = saveGame:getKey("offeredRunes")
    if offeredRunes then
      Runes.offered = offeredRunes
    end

    local quickbar = saveGame:getKey("quickbar")
    if quickbar then
      self.quickBar.slots = quickbar
    end

    local statusEffects = saveGame:getKey("statusEffects")
    if statusEffects then
      self.statusEffects:deserialize(statusEffects)
    end

    local arenaInfo = saveGame:getKey("arenaInfo")
    self.arenaInfo = arenaInfo or { current = 0 }
  end)
end

function Hero:say(text)
  if self.bubble then
    self.bubble.object:destroy()
  end

  self.bubble = CreateObject("SpeechBubble", Vector(8.0, 114.0), self.object):addComponent("SpeechBubble")
  self.bubble:setText(text)

  self.textCo = Co.create(function ()
    while not self.bubble.done do
      Co.yield()
    end

    Co.sleep(3.0)

    self.bubble.object:destroy()
    self.bubble = nil
  end)
end
