local AI = require "util/aicommon"

local HeroAvatar = Behavior("HeroAvatar")

HeroAvatar.editorIcon("hero/hero_down.json")

function HeroAvatar:initialize()
  self.walkSpeed = 138.0

  self.maxHeroDistance = 600.0

  self:addComponent("Character", {
    faction = "Heroes",
    hp = 200.0 / 3.0,
    movementSpeed = self.walkSpeed,
    collisionRadius = 24.0,
    colliderDisplacement = Vector(0.0, 12.0),
    colliderDensity = 128.0,
    isEightSided = true,
    skeleton = {
      s = "hero/hero_down.json",
      sw = "hero/hero_right_down.json",
      w = "hero/hero_left.json",
      nw = "hero/hero_right_up.json",
      n = "hero/hero_up.json",
      ne = "hero/hero_left_up.json",
      e = "hero/hero_right.json",
      se = "hero/hero_left_down.json"
    }
  })

  self:setupColliders()
  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object,
    bb_sword = self.sword.object
  })

  self.physicsCollider.categories = flags(World.layer("MovementCollider"), World.layer("Hero"))
  self.character:setSkeleton("s")
  self.skeletonRenderer.tintColor = Color(0.4, 1.2, 1.4, 0.65)

  self.heroRef = BehaviorUtil.findFirstOfType("Hero")
  self.health.totalHp = self.heroRef.health.totalHp * 0.5
  self.health.hp = self.health.totalHp

  self:addComponent("Healthbar", Vector(0.0, 80.0))

  local renderer = self.skeletonRenderer
  local character = self.character
  local soundEmitter = self.soundEmitter
  local movable = self.movable
  local transform = self.transform

  self.chaseFn = Co.create(function(enemy)
    self.character:setAnimation("walk", true)
    Co.yield(AI.chase(movable, enemy.transform, 80.0))
  end)

  self.goToFn = Co.create(function(target)
    local position = self.transform.worldPosition
    self.character:setAnimation("walk", true)

    if Physics.lineOfSight(position, target) then
      Co.yieldRunWhile(function()
        return Vector.distance(self.transform.worldPosition, target) > 4.0
      end, AI.moveTowards(self.movable, target))
    else
      local path = {}
      AI.yieldFindPath(self.transform.worldPosition, target, path)
      if not path.found then
        return
      end

      AI.yieldFollowPath(self.movable, path)
    end

    self.movable.velocity = Vector.zero()
  end)

  local sideStep = Co.create(function (enemy)
    character:setAnimation("walk", true)
    movable.movementSpeed = self.walkSpeed

    local direction = math.sign(Random.value() - 0.5)
    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized():perpendicular() * direction
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local stepBack = Co.create(function (enemy)
    movable.movementSpeed = self.walkSpeed
    character:setAnimation("walk", true)

    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized()
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  self.attackFn = Co.create(function(enemy)
    if not enemy.transform then
      return
    end

    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 40.0
    end, stepBack(enemy))
    movable.velocity = Vector.zero()

    Co.yield(AI.alignForAttack(movable, enemy.transform))

    movable.velocity = Vector.zero()
    character:setAnimation("idle")

    movable.velocity = Vector.zero()
    self.sword:startAttack(self.heroRef.hero:calculateWeaponDamage())
    character:setAnimation("hit")
    AI.yieldWaitForAnimation(self.object, "hit")
    self.sword:deactivate()
    character:setAnimation("idle")
    Co.sleep(0.15 + Random.value() * 0.25)
  end)

  self.getHitFn = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    self.sword:deactivate()
    soundEmitter:postEvent("HeroGotHit")

    if not self.health.isAlive then
      soundEmitter:postEvent("HeroOnDeath")
      self.sword:deactivate()
      self.skeletonRenderer:setFlag(self.skeletonRenderer.Flags.Dissolve)
      Co.yield(AI.fadeOut(renderer, 2.0))
      self.object:destroy()
      return
    end
    
    renderer.timeScale = 3.0
    character:setAnimation("get_hit", false)
    renderer.timeScale = 1.0
    AI.yieldWaitForAnimation(self.object, "get_hit")
  end)

  self.idle = Co.create(function()
    while true do
      character:setAnimation("idle", true)
      movable.velocity = Vector.zero()

      if not character.target then
        character.target = character:findVisibleEnemyInRange(300.0)
      end

      local enemy = character.target
      if not enemy then
        if Vector.distance(transform.worldPosition, self.heroRef.transform.worldPosition) > 240.0 then
          Co.yield(self.chaseFn(self.heroRef.movable))
        end

        return
      end

      Co.yieldRunWhile(function ()
        return enemy.health and enemy.health.isAlive
        and Vector.distance(enemy.transform.worldPosition, transform.worldPosition) < self.maxHeroDistance * 0.85
      end, Co.series(
        self.chaseFn(enemy),
        self.attackFn(enemy),
        Co.maybe(
          Co.concurrent(
            sideStep(enemy, 90.0),
            Co.waitFor(1.0, 1.75)
          ),
          0.25
        )
      ))

      self.sword:deactivate()
      character.target = nil
      Co.yield()
    end
  end)

  self.coMain = self.idle

  self.teleportToHeroCo = Co.create(function()
    local distanceSq = self.maxHeroDistance * self.maxHeroDistance

    while true do
      if Vector.distance2(self.heroRef.transform.worldPosition, self.transform.worldPosition) > distanceSq then
        local alpha = 1.0
        local color = renderer.tintColor
        while alpha > 0.0 do
          alpha = alpha - Time.fixedDeltaTime * 4.0
          color.a = alpha
          renderer.tintColor = color
        end

        local collision = true

        while collision do
          local pos = self.heroRef.transform.worldPosition
          pos = pos + Random.insideUnitCircle() * 128.0
          local size = Vector(72.0, 72.0)

          local collision = false
          Physics.aabbQuery(pos - size, pos + size, function(object)
            collision = true
            return false
          end, flags(World.layer("World"), World.layer("MovementCollider")))

          if collision then
            self.transform.localPosition = pos
            break
          end

          Co.yield()
        end

        while alpha < 0.65 do
          alpha = alpha + Time.fixedDeltaTime * 4.0
          color.a = alpha
          renderer.tintColor = color
        end

        color.a = 0.65
        renderer.tintColor = color
      end

      Co.sleep(1.0)
    end
  end)
end

function HeroAvatar:fixedUpdate()
  if not self.coMain:update() then
    self.idle:reset()
    self.coMain = self.idle
  end

  if not self.teleportToHeroCo:update() then
    self.teleportToHeroCo:reset()
  end
end

function HeroAvatar:setupColliders()
  self.sword = CreateObject("Sword", Vector.zero(), self.object):addComponent("Weapon")
  self.sword:listenForAnimationEvents(self.object)

  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy, hitDir)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end
end
