local BINDINGS_PATH = PathUtil.join(PathUtil.getApplicationData(), "bindings.json")

local InputEvents = require "lib/inputevents"

local GameplayContext = require "util/gameplaycontext"
local UIContext = require "util/uicontext"

local InputController = Behavior("InputController")

function InputController:initialize()
  self.device = "Controller"
  self.contexts = {}

  InputEvents:on("event", function(event)
    for i, context in ipairs(self.contexts) do
      self.device = event.source

      local handled = false

      context:evaluate(event, function (type, name, value)
        if type == "action" then
          if self:emitAction(name) then
            handled = true
          end
        elseif type == "value" then
          if self:emitState(name, value) then
            handled = true
          end
        elseif type == "range" then
          if self:emitRange(name, value) then
            handled = true
          end
        end
      end)

      if handled then
        break
      end
    end
  end)

  self.uiContext = UIContext()
  self.gameplayContext = GameplayContext()

  self:registerContext(self.uiContext)
  self:registerContext(self.gameplayContext)

  self:load()
end

function InputController:save()
  local serialized = {}
  for i, context in ipairs(self.contexts) do
    serialized[#serialized+1] = context:serialize()
  end

  local json = JSON:encode_pretty(serialized)
  local file = io.open(BINDINGS_PATH, "w")
  file:write(json)
  io.close(file)
end

function InputController:load()
  local file = io.open(BINDINGS_PATH, "r")
  if not file then
    return
  end

  local json = file:read("*all")

  local serialized = JSON:decode(json)
  for i, serializedContext in ipairs(serialized) do
    self.contexts[i]:deserialize(serializedContext)
  end
end

function InputController:resetDefaults()
  self.contexts = {}
  self.uiContext = UIContext()
  self.gameplayContext = GameplayContext()

  self:registerContext(self.uiContext)
  self:registerContext(self.gameplayContext)
  self:save()
end

function InputController:registerContext(context)
  context.priority = context.priority or 0
  self.contexts[#self.contexts+1] = context
end

function InputController:sortContextsByPriority()

end

function InputController:emitAction(name)
  local event = {
    type = "action",
    name = name,
    handled = false
  }

  local listeners = self.object.listeners[name]
  if not listeners then
    return false
  end

  for i, listener in ipairs(listeners) do
    listener(event)

    if event.handled then
      return true
    end
  end

  return false
end

function InputController:emitState(name, value)
  local event = {
    type = "value",
    name = name,
    handled = false,
    value = value
  }

  local listeners = self.object.listeners[name]
  if not listeners then
    return false
  end

  for i, listener in ipairs(listeners) do
    listener(event)
    
    if event.handled then
      return true
    end
  end

  return false
end

function InputController:emitRange(name, value)
  local event = {
    type = "range",
    name = name,
    handled = false,
    value = value
  }

  local listeners = self.object.listeners[name]
  if not listeners then
    return false
  end

  for i, listener in ipairs(listeners) do
    listener(event)
    
    if event.handled then
      return true
    end
  end

  return false
end

function InputController:getKeyboardBindings()
  local bindings = {}

  for name, binds in pairs(self.uiContext.actions) do
    for i, bind in ipairs(binds) do
      if bind.device == self.device and bind.device == "Keyboard" then
        bindings[name] = "keyboard-button " .. KEYS_INDEX[bind.button]
        break
      elseif bind.device == self.device and bind.device == "Controller" then
        bindings[name] = "controller-button " .. bind.button
        break
      end
    end
  end

  for name, binds in pairs(self.gameplayContext.actions) do
    for i, bind in ipairs(binds) do
      if bind.device == self.device and bind.device == "Keyboard" then
        bindings[name] = "keyboard-button " .. KEYS_INDEX[bind.button]
        break
      elseif bind.device == self.device and bind.device == "Controller" then
        bindings[name] = "controller-button " .. bind.button
        break
      end
    end
  end

  return bindings
end
