local SaveGameStats = require "util/savegamestats"

local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local Draugur = Behavior("Draugur")

Draugur.property("hp", 180.0)
Draugur.property("detectionRadius", 360.0)
Draugur.property("socialRadius", 600.0)
Draugur.property("damage", 60.0)
Draugur.property("aoeDamage", 50.0)
Draugur.property("walkSpeed", 138.0)
Draugur.property("spawnDirection", "s")
Draugur.property("faction", "Monks")
Draugur.property("lootTable", "Draugur")
Draugur.property("patrolRoute", "", Metatype.Route)

Draugur.editorIcon("draugur/draugur_down.json")

function Draugur:initialize(properties)
  self.properties = Draugur.defaultProperties(properties)

  local skeleton = {
    s = "draugur/draugur_down.json",
    n = "draugur/draugur_up.json",
    w = "draugur/draugur_left.json",
    e = "draugur/draugur_right.json"
  }

  self:addComponent("Character", {
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 24.0,
    colliderDisplacement = Vector(0.0, 14.0),
    colliderDensity = 128.0,
    skeleton = skeleton,
    movementSpeed = self.properties.walkSpeed
  })

  self:setupColliders()

  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object,
    bb_hit = self.weaponBox.object
  })

  self.character:setSkeleton(self.properties.spawnDirection)

  self:addComponent("Healthbar", Vector(0.0, 100.0))
  self.spawnPosition = self.transform.worldPosition
  self:setupAI()
end

function Draugur:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  local returnToSpawn = Co.create(function ()
    character:setAnimation("walk", true)
    Co.yield(AI.goTo(movable, self.spawnPosition, 24.0))
    movable.velocity = Vector.zero()
    character:setAnimation("idle", true)
  end)

  local chase = Co.create(function (enemy)
    character.isAlert = true
    character:setAnimation("walk", true)

    Co.yield(AI.chase(movable, enemy.transform, 100.0))

    if movable.blocked then
      Co.yield(self.doAttack())
    end
  end)

  local sideStep = Co.create(function (enemy)
    character:setAnimation("walk", true)
    movable.movementSpeed = properties.walkSpeed

    local direction = math.sign(Random.value() - 0.5)
    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized():perpendicular() * direction
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local stepBack = Co.create(function (enemy)
    movable.movementSpeed = properties.walkSpeed
    character:setAnimation("walk", true)

    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized()
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local doAttack = Co.create(function()
    renderer.timeScale = 3.75

    character:setAnimation("telegraph")
    AI.yieldWaitForAnimation(self.object, "telegraph")
    self.weaponBox:startAttack(properties.damage)

    renderer.timeScale = 1.5

    character:setAnimation("hit")
    AI.yieldWaitForAnimation(self.object, "hit")

    renderer.timeScale = 1.0
  end)

  self.doAttack = doAttack

  local attack = Co.create(function(enemy)
    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 60.0
    end, stepBack(enemy, 60.0))

    Co.yield(AI.alignForAttack(movable, enemy.transform))

    soundEmitter:postEvent("DraugarAttack")
    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    Co.yield(self.doAttack())
  end)

  self.attack = attack

  local aoeAttack = Co.create(function(enemy)
    if character:findAllyInRange(300.0) then
      Co.yield(attack(enemy))
      return
    end

    soundEmitter:postEvent("DraugarAttack2")
    movable.velocity = Vector.zero()

    renderer.timeScale = 2.0
    character:setAnimation("telegraph")
    AI.yieldWaitForAnimation(self.object, "telegraph")
    renderer.timeScale = 0.75

    self.aoeHitBox:startAttack(self.properties.aoeDamage)
    character:setAnimation("hit_aoe")

    Co.sleep(0.3)
    self.aoeHitBox:deactivate()

    AI.yieldWaitForAnimation(self.object, "hit_aoe")
    renderer.timeScale = 1.0
  end)

  local idle = Co.create(function()
    while true do
      renderer.timeScale = 1.0
      movable.velocity = Vector.zero()
      character:setAnimation("idle", true)

      Co.yieldRunWhile(function ()
        if character.target then
          return false
        end

        character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
        return not character.target 
      end,
        function()
          if character.target then
            character:warnAllies(properties.socialRadius)
          elseif Vector.distance(transform.worldPosition, self.spawnPosition) > 40.0 then
            Co.yield(returnToSpawn())
            character:setAnimation("idle", true)
          else
            Co.sleep(0.25)
          end
        end
      )

      local enemy = character.target
      if not enemy then
        return
      end

      Co.yieldRunWhile(function ()
        return enemy.health and enemy.health.isAlive
      end, Co.series(
        chase(enemy),
        Co.pickOne(
          attack(enemy),
          aoeAttack(enemy),
          { 0.75, 0.25 }
        ),
        Co.maybe(
          Co.concurrent(
            sideStep(enemy, 90.0),
            Co.waitFor(1.0, 1.75)
          ),
          0.45
        )
      ))

      character.target = nil
    end
  end)

  self.idle = idle

  self.followPatrolRoute = Co.create(function()
    local patrolRoute = WorldUtil.getRouteByName(properties.patrolRoute)
    if not patrolRoute then
      print("ERROR - Invalid patrol route name \"" .. properties.patrolRoute .. "\"")
      return
    end

    character:setAnimation("walk", true)

    Co.yieldRunWhile(function ()
      if character.target then
        return
      end
      
      character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
      return not character.target 
    end, AI.followPatrolRoute(movable, patrolRoute))

    if character.target then
      character:warnAllies(properties.socialRadius)
    end

    Co.yield(idle())
  end)

  self.getHitFn = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    renderer.timeScale = 1.0
    soundEmitter:postEvent("DraugarGetHit")
    character:setAnimation("get_hit", false)

    if not self.health.isAlive then
      self.statusEffects:clearAll()
      soundEmitter:postEvent("DraugarOnDeath")
      self.weaponBox:deactivate()
      
      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0
      self:removeComponent("Healthbar")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      SaveGameStats.increase("enemiesSlain")
      return
    end
   
    AI.yieldWaitForAnimation(self.object, "get_hit")
    
    if enemy.character and enemy.character.faction ~= character.faction then
      character.target = enemy
    end

    self.coMain = Co.maybe(
      Co.pickOne(
        attack(enemy),
        Co.concurrent(
          stepBack(enemy),
          Co.waitFor(0.45, 0.55)
        ),
        Co.concurrent(
          sideStep(enemy),
          Co.waitFor(0.35, 0.45)
        ),
        { 0.3, 0.35, 0.35 }
      )
    , 0.5)
  end)

  self.object:on("resetTarget", function()
    self.idle:reset()
    self.coMain = self.idle
  end)
end

function Draugur:fixedUpdate()
  self.aoeHitBox.transform.localPosition = Vector.zero()

  if not self.coMain or not self.coMain:update() then
    if self.properties.patrolRoute == "" then
      self.idle:reset()
      self.coMain = self.idle
    else
      self.followPatrolRoute:reset()
      self.coMain = self.followPatrolRoute
    end
  end
end

function Draugur:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end

  self.weaponBox = CreateObject("Hand", Vector.zero(), self.object):addComponent("Weapon")
  self.weaponBox:listenForAnimationEvents(self)

  self.aoeHitBox = CreateObject("AoEHitbox", Vector.zero(), self.object):addComponent("Weapon")
  self.aoeHitBox:listenForAnimationEvents(self, "AoeColliderOn", "AoeColliderOff")
  local aoeCollider = self.aoeHitBox.physicsCollider
  aoeCollider:addRectangleShape(Vector(0.0, 16.0), Vector(232.0, 160.0), { isSensor = true })
end
