local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local ForestCritter = Behavior("ForestCritter")

ForestCritter.property("hp", 180.0)
ForestCritter.property("walkSpeed", 60.0)
ForestCritter.property("runSpeed", 120.0)
ForestCritter.property("spawnDirection", "e")
ForestCritter.property("faction", "Neutral")
ForestCritter.property("lootTable", "ForestCritter")

ForestCritter.editorIcon("forest_critter/forest_critter_right.json")

function ForestCritter:initialize(properties)
  self.properties = ForestCritter.defaultProperties(properties)

  self:addComponent("Character", {
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 18.0,
    colliderDisplacement = Vector(0.0, 8.0),
    colliderDensity = 128.0,
    skeleton = {
      w = "forest_critter/forest_critter_left.json",
      e = "forest_critter/forest_critter_right.json"
    },
    isTwoSided = true
  })

  self:setupColliders()
  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object
  })

  self.character:setSkeleton(self.properties.spawnDirection)

  self:addComponent("Healthbar", Vector(0.0, 48.0))

  self.spawnPosition = self.transform.worldPosition
  self.isRunning = false

  self.movable.movementSpeed = self.properties.walkSpeed

  self:setupAI()
end

function ForestCritter:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  self:setRunning(false)

  self.idle = Co.create(function()
    while true do
      movable.velocity = Vector.zero()

      local dir = nil

      while true do
        dir = Random.unitVector()
        dir.y = dir.y * 0.5
        dir:normalize()

        if Physics.lineOfSight(transform.worldPosition, transform.worldPosition + dir * 128.0) then
          break
        end

        Co.sleep(0.1)
      end

      self:setRunning(self.isRunning)
      character.lookDir = dir
      character:setOrientedSkeleton()

      Co.yieldConcurrent(
        AI.moveTowards(movable, transform.worldPosition + dir * 512.0),
        Co.waitFor(2.0, 6.0),
        function()
          while true do
            local hitWall = false
            local p0 = transform.worldPosition
            local p1 = p0 + dir * 64.0
            if not Physics.lineOfSight(p0, p1) then
              break
            end

            Co.sleep(0.1)
          end
        end
      )

      if not self.isRunning then
        movable.velocity = Vector.zero()

        if Random.value() <= 0.5 then
          character:setAnimation("idle_2")
          character:addAnimation("idle", true)
        elseif Random.value() <= 0.05 then
          character:setAnimation("idle_jump")
          character:addAnimation("idle", true)
        else
          character:setAnimation("idle", true)
        end

        Co.sleep(4.0, 8.0)
      else
        character:setAnimation("idle", true)
      end
    end
  end)

  self.getHitFn = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    soundEmitter:postEvent("ForestCritterGetHit")
    character:setAnimation("get_hit", false)

    if not self.health.isAlive then
      self.statusEffects:clearAll()
      soundEmitter:postEvent("ForestCritterOnDeath")

      if self.properties.lootTable ~= "" then
        LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)
      end

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0
      self:removeComponent("Healthbar")
      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      return
    end
   
    AI.yieldWaitForAnimation(self.object, "get_hit")
    self.isRunning = true
  end)
end

function ForestCritter:fixedUpdate()
  if not self.coMain or not self.coMain:update() then
    self.idle:reset()
    self.coMain = self.idle
  end
end

function ForestCritter:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy, hitDir)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end
end

function ForestCritter:setRunning(running)
  if running then
    self.movable.movementSpeed = self.properties.runSpeed
  else
    self.movable.movementSpeed = self.properties.walkSpeed
  end

  if not running then
    self.character:setAnimation("walk", true)
    self.isRunning = false
  elseif running then
    self.character:setAnimation("run", true)
    self.isRunning = true
  end
end
