local SaveGameStats = require "util/savegamestats"

local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local Knight = Behavior("Knight")

Knight.property("hp", 250.0)
Knight.property("detectionRadius", 360.0)
Knight.property("socialRadius", 600.0)
Knight.property("shortAttackDamage", 65.0)
Knight.property("longAttackDamage", 50.0)
Knight.property("walkSpeed", 120.0)
Knight.property("runSpeed", 300.0)
Knight.property("spawnDirection", "s")
Knight.property("faction", "Knights")
Knight.property("lootTable", "Knight")
Knight.property("patrolRoute", "", Metatype.Route)

Knight.editorIcon("knight/knight_down.json")

function Knight:initialize(properties)
  self.properties = Knight.defaultProperties(properties)

  self:addComponent("Character", {
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 28.0,
    colliderDisplacement = Vector(0.0, 8.0),
    colliderDensity = 128.0,
    skeleton = {
      s = "knight/knight_down.json",
      n = "knight/knight_up.json",
      w = "knight/knight_left.json",
      e = "knight/knight_right.json"
    }
  })

  self:setupColliders()
  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object,
    bb_weapon = self.halberd.object
  })

  self.character:setSkeleton(self.properties.spawnDirection)

  self:addComponent("Healthbar", Vector(0.0, 88.0))
  CreateObject("FootstepsRenderer", Vector.zero(), self.object):addComponent("FootstepsRenderer")

  self.spawnPosition = self.transform.worldPosition
  self.isRunning = false

  self.walkFootstepSpeed = 0.45
  self.runFootstepSpeed = 0.35
  self.footstepSpeed = self.walkFootstepSpeed
end

function Knight:awake()
  self:setupAI()
end

function Knight:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  local returnToSpawn = Co.create(function ()
    movable.velocity = Vector.zero()
    self:setRunning(false)
    Co.yield(AI.goTo(movable, self.spawnPosition, 28.0))
    movable.velocity = Vector.zero()
    character:setAnimation("idle", true)
  end)

  local chase = Co.create(function (enemy)
    character.isAlert = true
    self:setRunning(false)

    Co.yieldConcurrent(
      AI.chase(movable, enemy.transform, 120.0, 28.0),
      AI.playFootstepSounds(movable, "KnightFootstep", self.footstepSpeed)
    )
  end)

  local sideStep = Co.create(function (enemy)
    character:setAnimation("walk", true)
    self:setRunning(false)

    local direction = math.sign(Random.value() - 0.5)
    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized():perpendicular() * direction
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local stepBack = Co.create(function (enemy)
    self:setRunning(false)
    character:setAnimation("walk", true)

    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized()
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local attack = Co.create(function(enemy)
    if not enemy.transform then
      return
    end

    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    if not self.isBlocking then
      Co.yieldRunWhile(function ()
        return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 100.0
      end, stepBack(enemy))
      movable.velocity = Vector.zero()

      Co.yield(AI.alignForAttack(movable, enemy.transform))

      movable.velocity = Vector.zero()
      character:setAnimation("idle")
      Co.sleep(0.15 + Random.value() * 0.25)
    else
      self.isBlocking = false
    end

    movable.velocity = Vector.zero()
    renderer.timeScale = 0.85

    soundEmitter:postEvent("KnightAttack")
    if Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 80.0 then
      self.halberd:startAttack(properties.shortAttackDamage)
      character:setAnimation("hit_1")
      AI.yieldWaitForAnimation(self.object, "hit_1")
    else
      self.halberd:startAttack(properties.longAttackDamage)
      character:setAnimation("hit")
      AI.yieldWaitForAnimation(self.object, "hit")
    end

    renderer.timeScale = 1.0
  end)

  self.attack = attack

  local chargeAttack = Co.create(function(enemy)
    self:setRunning(false)
    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    local dir = (transform.worldPosition - enemy.transform.worldPosition):normalized()
    if not Physics.lineOfSight(transform.worldPosition, dir * 360.0) then
      return
    end

    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 340.0
      and Physics.lineOfSight(transform.worldPosition, enemy.transform.worldPosition)
    end,
      Co.concurrent(
        stepBack(enemy),
        AI.playFootstepSounds(self.movable, "KnightFootstep", self.footstepSpeed)
      )
    )
    
    movable.velocity = Vector.zero()
    self:setRunning(true)

    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) > 120.0
      and Physics.lineOfSight(transform.worldPosition, enemy.transform.worldPosition)
    end, Co.concurrent(
      function ()
        while true do
          local otherPos = enemy.transform.worldPosition
          local myPos = movable.transform.worldPosition
          local diff = myPos - otherPos
          diff = diff:abs()

          local distance = Vector.distance(otherPos, myPos)
          if distance ~= 0.0 then
            diff = diff / distance
          end

          local dir = (myPos - otherPos) / distance

          local target = otherPos + dir * 120.0
          if diff.x < diff.y then
            target.x = otherPos.x
          else
            target.y = otherPos.y
          end

          local dist = Vector.distance(myPos, target)
          movable.velocity = (target - myPos) / dist

          if movable.character then
            movable.character:lookAt(target)
          end
          
          Co.yield()
        end
      end,
      AI.playFootstepSounds(self.movable, "KnightFootstep", self.footstepSpeed)
    ))

    movable.velocity = Vector.zero()
    Co.yield(AI.alignForAttack(movable, enemy.transform))
    character:lookAt(enemy.transform.worldPosition)

    movable.velocity = Vector.zero()
    renderer.timeScale = 1.5
    self.halberd:startAttack(properties.longAttackDamage)
    character:setAnimation("hit")
    AI.yieldWaitForAnimation(self.object, "hit")
    renderer.timeScale = 1.0
  end)

  local block = Co.create(function (enemy)
    movable.velocity = Vector.zero()
    self.isBlocking = true

    renderer.timeScale = 3.0
    character:setAnimation("block_start")
    AI.yieldWaitForAnimation(self.object, "block_start")
    renderer.timeScale = 1.0
    character:setAnimation("block_active", true)

    local time = Time.globalTime + 0.85

    while Time.globalTime < time do
      character:lookAt(enemy.transform.worldPosition)
      Co.yield()
    end

    character:setAnimation("block_release")  
    AI.yieldWaitForAnimation(self.object, "block_release")
    self.isBlocking = false
    character:setAnimation("idle")
  end)

  local idle = Co.create(function()
    while true do
      movable.velocity = Vector.zero()
      character:setAnimation("idle", true)

      Co.yieldRunWhile(function ()
        if character.target then
          return false
        end

        character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
        return not character.target 
      end,
        function()
          if character.target then
            character:warnAllies(properties.socialRadius)
          elseif Vector.distance(transform.worldPosition, self.spawnPosition) > 40.0 then
            Co.yield(returnToSpawn())
            character:setAnimation("idle", true)
          else
            Co.sleep(0.25)
          end
        end
      )

      local enemy = character.target
      if not enemy then
        return
      end

      Co.yieldRunWhile(function ()
        return enemy.health and enemy.health.isAlive
      end, Co.series(
        chase(enemy),
        Co.pickOne(
          attack(enemy),
          chargeAttack(enemy),
          { 0.8, 0.2 }
        ),
        Co.maybe(
          Co.concurrent(
            sideStep(enemy, 90.0),
            Co.waitFor(1.0, 1.75)
          ),
          0.5
        )
      ))

      character.target = nil
    end
  end)

  self.idle = idle

  self.followPatrolRoute = Co.create(function()
    local patrolRoute = WorldUtil.getRouteByName(properties.patrolRoute)
    if not patrolRoute then
      print("ERROR - Invalid patrol route name \"" .. properties.patrolRoute .. "\"")
      return
    end

    self:setRunning(false)
    character:setAnimation("walk", true)

    Co.yieldRunWhile(function ()
      if character.target then
        return
      end

      character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
      return not character.target 
    end, Co.concurrent(
      AI.followPatrolRoute(movable, patrolRoute),
      AI.playFootstepSounds(self.movable, "KnightFootstep", self.footstepSpeed)
    ))

    if character.target then
      character:warnAllies(properties.socialRadius)
    end

    Co.yield(idle())
  end)

  self.getHitFn = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    self.statusEffects:clearAll()
    soundEmitter:postEvent("KnightGetHit")
    character:setAnimation("get_hit", false)

    if not self.health.isAlive then
      self.halberd:deactivate()
      soundEmitter:postEvent("KnightOnDeath")

      if properties.lootTable ~= "" then
        LootSpawner.spawnPouch(self.transform.worldPosition, properties.lootTable)
      end

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0
      self:removeComponent("Healthbar")

      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      SaveGameStats.increase("enemiesSlain")
      return
    end
   
    AI.yieldWaitForAnimation(self.object, "get_hit")
    
    if enemy.character and enemy.character.faction ~= character.faction then
      character.target = enemy
    end

    self.coMain = Co.pickOne(
      block(enemy),
      Co.concurrent(
        sideStep(enemy, 90.0),
        Co.waitFor(0.75, 1.25)
      ),
      Co.runWhile(function ()
        return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 100.0
      end, stepBack(enemy)),
      { 0.5, 0.3, 0.2 }
    )
  end)

  self.object:on("resetTarget", function()
    self.idle:reset()
    self.coMain = self.idle
  end)
end

function Knight:fixedUpdate()
  if not self.coMain or not self.coMain:update() then
    if self.properties.patrolRoute == "" then
      self.idle:reset()
      self.coMain = self.idle
    else
      self.followPatrolRoute:reset()
      self.coMain = self.followPatrolRoute
    end
  end
end

function Knight:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy, hitDir)
    if self.isBlocking then
      local dot = Vector.dot(self.character.lookDir, hitDir)
      local blockSuccess = dot > 0.5
      if blockSuccess then
        self.attack:reset()
        self.coMain = self.attack(enemy)
        return false
      end
    end

    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end

  self.halberd = CreateObject("Sword", Vector.zero(), self.object):addComponent("Weapon")
  self.halberd:listenForAnimationEvents(self)
end

function Knight:setRunning(running)
  if running then
    self.movable.movementSpeed = self.properties.runSpeed
  else
    self.movable.movementSpeed = self.properties.walkSpeed
  end

  if not running then
    self.character:setAnimation("walk", true)
    self.isRunning = false
    self.footstepSpeed = self.walkFootstepSpeed
  elseif running then
    self.character:setAnimation("run", true)
    self.isRunning = true
    self.footstepSpeed = self.runFootstepSpeed
  end
end
