local SaveGameStats = require "util/savegamestats"

local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local LittleAssassin = Behavior("LittleAssassin")

LittleAssassin.property("hp", 145.0)
LittleAssassin.property("detectionRadius", 360.0)
LittleAssassin.property("socialRadius", 600.0)
LittleAssassin.property("damage", 35.0)
LittleAssassin.property("walkSpeed", 170.0)
LittleAssassin.property("spawnDirection", "s")
LittleAssassin.property("faction", "Assassins")
LittleAssassin.property("lootTable", "LittleAssassin")
LittleAssassin.property("patrolRoute", "", Metatype.Route)

LittleAssassin.editorIcon("little_assassin/little_assassin_down.json")

function LittleAssassin:initialize(properties)
  self.properties = LittleAssassin.defaultProperties(properties)

  self:addComponent("Character", {
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 24.0,
    colliderDisplacement = Vector(0.0, 14.0),
    colliderDensity = 64.0,
    skeleton = {
      s = "little_assassin/little_assassin_down.json",
      n = "little_assassin/little_assassin_up.json",
      w = "little_assassin/little_assassin_left.json",
      e = "little_assassin/little_assassin_right.json"
    }
  })

  self:setupColliders()
  self.character:setSkeleton(self.properties.spawnDirection)

  self:addComponent("Healthbar", Vector(0.0, 100.0))
  self.spawnPosition = self.transform.worldPosition

  self.movable.movementSpeed = self.properties.walkSpeed
end

function LittleAssassin:awake()
  self:setupAI()
end

function LittleAssassin:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  local returnToSpawn = Co.create(function ()
    character:setAnimation("walk", true)
    Co.yield(AI.goTo(movable, self.spawnPosition))
    movable.velocity = Vector.zero()
    character:setAnimation("idle", true)
  end)

  local chase = Co.create(function (enemy)
    movable.movementSpeed = properties.walkSpeed
    character.isAlert = true
    character:setAnimation("walk", true)
    Co.yield(AI.chase(movable, enemy.transform, 150.0))
  end)

  local sideStep = Co.create(function (enemy)
    character:setAnimation("walk", true)
    movable.movementSpeed = properties.walkSpeed

    local direction = math.sign(Random.value() - 0.5)
    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized():perpendicular() * direction
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local stepBack = Co.create(function (enemy)
    movable.movementSpeed = properties.walkSpeed
    character:setAnimation("walk", true)

    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized()
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local attack = Co.create(function(enemy)
    local doAlign = true

    if Vector.distance(transform.worldPosition, enemy.transform.worldPosition) > 80.0 then
      Co.yieldRunWhile(function ()
        return Vector.distance2(transform.worldPosition, enemy.transform.worldPosition) < 120.0
      end, stepBack(enemy))

      local target = enemy.transform.worldPosition
      local myPos = transform.worldPosition
      local diff = target - myPos

      if math.abs(diff.x) < math.abs(diff.y) then
        target.y = myPos.y
      else
        target.x = myPos.x
      end

      Co.yieldRunWhile(function ()
        return Vector.distance(target, enemy.transform.worldPosition) < 32.0
      end, AI.moveTowards(movable, target))
    end

    movable.collisionAvoidance = false
    soundEmitter:postEvent("LittleAssassinAttack")
    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    local dir = (enemy.transform.worldPosition - transform.worldPosition):normalized()

    renderer.timeScale = 1.75
    character:setAnimation("telegraph")
    AI.yieldWaitForAnimation(self.object, "telegraph")
    renderer.timeScale = 1.0
    character:setAnimation("hit")

    self.dagger1:startAttack(properties.damage)
    self.dagger2:startAttack(properties.damage)

    movable.velocity = dir
    movable.movementSpeed = properties.walkSpeed * 1.35
    AI.yieldWaitForAnimation(self.object, "hit")

    movable.velocity = Vector.zero()
    movable.collisionAvoidance = true
    self.dagger1:deactivate()
    self.dagger2:deactivate()

    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 120.0
    end, stepBack(enemy))
  end)

  self.idle = Co.create(function()
    renderer.timeScale = 1.0

    while true do
      movable.movementSpeed = properties.walkSpeed
      movable.velocity = Vector.zero()
      character:setAnimation("idle", true)

      --if not character.target then
      --  character.target = BehaviorUtil.findFirstOfType("Hero")
      --end

      Co.yieldRunWhile(function ()
        if character.target then
          return false
        end

        character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
        return not character.target 
      end,
        function()
          if character.target then
            character:warnAllies(properties.socialRadius)
          elseif Vector.distance(transform.worldPosition, self.spawnPosition) > 40.0 then
            Co.yield(returnToSpawn())
            character:setAnimation("idle", true)
          else
            Co.sleep(0.25)
          end
        end
      )

      local enemy = character.target
      if not enemy then
        return
      end

      Co.yieldRunWhile(function ()
        return enemy.health
        and enemy.health.isAlive
      end, Co.series(
        chase(enemy),
        attack(enemy),
        Co.maybe(
          Co.concurrent(
            sideStep(enemy, 150.0),
            Co.waitFor(1.0, 1.75)
          ),
          0.5
        )
      ))

      renderer.timeScale = 1.0
      character.target = nil

      self.dagger1:deactivate()
      self.dagger2:deactivate()
    end
  end)

  self.followPatrolRoute = Co.create(function()
    local patrolRoute = WorldUtil.getRouteByName(properties.patrolRoute)
    if not patrolRoute then
      print("ERROR - Invalid patrol route name \"" .. properties.patrolRoute .. "\"")
      return
    end

    character:setAnimation("walk", true)

    Co.yieldRunWhile(function ()
      if character.target then
        return
      end
      
      character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
      return not character.target 
    end, AI.followPatrolRoute(movable, patrolRoute))

    if character.target then
      character:warnAllies(properties.socialRadius)
    end

    Co.yield(self.idle())
  end)

  self.getHitFn = Co.create(function(enemy)
    if self.projectile then
      self.projectile:selfDestruct()
      self.projectile = nil
    end

    movable.velocity = Vector.zero()
    soundEmitter:stopAllSounds()
    soundEmitter:postEvent("LittleAssassinGetHit")
    character:setAnimation("get_hit", false)

    if not self.health.isAlive then
      self.dagger1:deactivate()
      self.dagger2:deactivate()
      
      self.statusEffects:clearAll()
      soundEmitter:postEvent("LittleAssassinOnDeath")

      LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0
      self:removeComponent("Healthbar")
      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      SaveGameStats.increase("enemiesSlain")
      return
    end

    AI.waitForAnimation(self.object, "get_hit")
    
    if enemy.character and enemy.character.faction ~= character.faction then
      character.target = enemy
    end

    self.idle:reset()
    self.coMain = self.idle
  end)

  self.object:on("resetTarget", function()
    self.idle:reset()
    self.coMain = self.idle
  end)
end

function LittleAssassin:fixedUpdate()
  if not self.coMain or not self.coMain:update() then
    if self.properties.patrolRoute == "" then
      self.idle:reset()
      self.coMain = self.idle
    else
      self.followPatrolRoute:reset()
      self.coMain = self.followPatrolRoute
    end
  end
end

function LittleAssassin:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end

  self.dagger1 = CreateObject("Dagger1", Vector.zero(), self.object):addComponent("Weapon")
  self.dagger1:listenForAnimationEvents(self, "AttackColliderOn1", "AttackColliderOff1")

  self.dagger2 = CreateObject("Dagger2", Vector.zero(), self.object):addComponent("Weapon")
  self.dagger2:listenForAnimationEvents(self, "AttackColliderOn2", "AttackColliderOff2")

  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object,
    bb_sword = self.dagger1.object,
    bb_sword2 = self.dagger2.object
  })
end
