local SaveGameStats = require "util/savegamestats"

local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local MiniBoss = Behavior("MiniBoss")

MiniBoss.property("hp", 1250.0)
MiniBoss.property("detectionRadius", 800.0)
MiniBoss.property("socialRadius", 800.0)
MiniBoss.property("damage", 65.0)
MiniBoss.property("walkSpeed", 140.0)
MiniBoss.property("variant", "default")
MiniBoss.property("spawnDirection", "e")
MiniBoss.property("faction", "Monks")
MiniBoss.property("lootTable", "MiniBoss")

MiniBoss.editorIcon("mini_boss/mini_boss_right.json")

function MiniBoss:initialize(properties)
  self.properties = MiniBoss.defaultProperties(properties)

  local skeleton = {
    w = "mini_boss/mini_boss_left.json",
    e = "mini_boss/mini_boss_right.json"
  }

  self:addComponent("Character", {
    isTwoSided = true,
    isLarge = true,
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 32.0,
    colliderDisplacement = Vector(0.0, 24.0),
    colliderDensity = 4096.0,
    skeleton = skeleton
  })

  self.movable.collisionAvoidanceStrength = 6.0

  self:setupColliders()

  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object,
    bb_sword = self.sword.object
  })

  self.character:setSkeleton(self.properties.spawnDirection)

  self:addComponent("Healthbar", Vector(0.0, 220.0))
  self.spawnPosition = self.transform.worldPosition
  self:setupAI()

  self.movable.movementSpeed = self.properties.walkSpeed
end

function MiniBoss:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  local returnToSpawn = Co.create(function ()
    if self.holdingBarrel then
      character:setAnimation("walk_barrel", true)
    else
      character:setAnimation("walk", true)
    end

    if Physics.lineOfSight(transform.worldPosition, self.spawnPosition) then
      Co.yield(AI.moveTowards(movable, self.spawnPosition))
    else
      local path = {}
      Co.yield(AI.findPath(transform.worldPosition, self.spawnPosition, path))

      if not path.found then
        return
      end

      Co.yield(AI.followPath(movable, path))
    end

    movable.velocity = Vector.zero()
  end)

  local chase = Co.create(function (enemy)
    character.isAlert = true

    if self.holdingBarrel then
      character:setAnimation("walk_barrel", true)
    else
      character:setAnimation("walk", true)
    end

    local distance = 160.0
    if self.holdingBarrel then
      distance = 500.0
    end

    Co.yield(AI.chase(movable, enemy.transform, distance))
  end)

  local stepBack = Co.create(function (enemy)
    if self.holdingBarrel then
      character:setAnimation("walk_barrel", true)
    else
      character:setAnimation("walk", true)
    end

    local enemyPos = enemy.transform.worldPosition
    local myPos = self.transform.worldPosition
    local dist = Vector.distance(enemyPos, myPos)

    while true do
      enemyPos = enemy.transform.worldPosition
      myPos = self.transform.worldPosition
      dist = Vector.distance(enemyPos, myPos)

      movable.velocity = (myPos - enemyPos) / dist

      character:lookAt(enemyPos)

      if movable.blocked then
        return
      end

      Co.yield()
    end
  end)

  local throwBarrelAttack = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    character:setAnimation("walk_barrel", true)

    while true do
      character:lookAt(enemy.transform.worldPosition)

      local enemyPos = enemy.transform.worldPosition + Vector(0.0, -64.0)
      local myPos = movable.transform.worldPosition

      local diff = myPos - enemyPos
      diff = diff:abs()

      if diff.y < 8.0 then
        break
      end

      local target = myPos:clone()
      target.y = enemyPos.y

      movable.velocity = (target - myPos):normalized()
      if movable.blocked then
        break
      end

      Co.yield()
    end
    
    movable.velocity = Vector.zero()
    character:setAnimation("idle_barrel", true)
    Co.sleep(0.15, 0.25)

    soundEmitter:postEvent("MiniBossThrowBarrel")
    character:setAnimation("throw")
    AI.yieldWaitForAnimation(self.object, "throw")
    self.holdingBarrel = false
  end)

  local swordAttack = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    character:setAnimation("walk", true)

    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 110.0
    end, stepBack(enemy))

    while true do
      character:lookAt(enemy.transform.worldPosition)

      local enemyPos = enemy.transform.worldPosition + Vector(0.0, -8.0)
      local myPos = movable.transform.worldPosition

      local diff = myPos - enemyPos
      diff = diff:abs()

      if diff.y < 8.0 then
        break
      end

      local target = myPos:clone()
      target.y = enemyPos.y

      movable.velocity = (target - myPos):normalized()
      if movable.blocked then
        break
      end

      Co.yield()
    end

    movable.velocity = Vector.zero()
    self.sword:startAttack(self.properties.damage)

    if Random.value() <= 0.5 then
      soundEmitter:postEvent("MiniBossAttack1")
      character:setAnimation("hit")
      AI.yieldWaitForAnimation(self.object, "hit")
    else
      soundEmitter:postEvent("MiniBossAttack2")
      character:setAnimation("hit2")
      AI.yieldWaitForAnimation(self.object, "hit2")
    end
  end)

  self.isTired = false

  local tired = Co.create(function(time)
    soundEmitter:postEvent("MiniBossTired")

    self.isTired = true
    self.tiredTimer = time
    while self.tiredTimer > 0.0 do
      self.tiredTimer = self.tiredTimer - Time.fixedDeltaTime
    end

    character:addAnimation("tired_stop")
    AI.yieldWaitForAnimation(self.object, "tired_stop")
    self.isTired = false
  end)

  local swordAttackCombo = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    character:setAnimation("walk", true)

    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 110.0
    end, stepBack(enemy))

    while true do
      character:lookAt(enemy.transform.worldPosition)

      local enemyPos = enemy.transform.worldPosition + Vector(0.0, -8.0)
      local myPos = movable.transform.worldPosition

      local diff = myPos - enemyPos
      diff = diff:abs()

      if diff.y < 8.0 then
        break
      end

      local target = myPos:clone()
      target.y = enemyPos.y

      movable.velocity = (target - myPos):normalized()
      if movable.blocked then
        break
      end

      Co.yield()
    end

    movable.velocity = Vector.zero()

    for i=1,3 do
      soundEmitter:postEvent("MiniBossAttack1")
      self.sword:startAttack(self.properties.damage)
      character:setAnimation("hit")
      AI.yieldWaitForAnimation(self.object, "hit")
    end

    if Random.value() < 0.15 then
      soundEmitter:postEvent("MiniBossAttack2")
      character:setAnimation("hit2")
      AI.yieldWaitForAnimation(self.object, "hit2")
    end

    character:setAnimation("tired_start")
    character:addAnimation("tired_loop", true)
    AI.yieldWaitForAnimation(self.object, "tired_start")
    Co.yield(tired(3.0 + Random.value() * 3.0))
  end)

  local attack = Co.create(function(enemy)
    movable.velocity = Vector.zero()

    if self.holdingBarrel then
      Co.yield(throwBarrelAttack(enemy))
    else
      local hpDelta = self.health.hp / self.health.totalHp

      if hpDelta <= 0.75 then
        Co.yield(Co.pickOne(
          swordAttack(enemy),
          swordAttackCombo(enemy),
          { 0.7, 0.3 }
        ))
      else
        Co.yield(swordAttack(enemy))
      end
    end

    if Random.value() <= 0.15 then
      soundEmitter:postEvent("MiniBossTaunt")
      character:setAnimation("taunt")
      AI.yieldWaitForAnimation(self.object, "taunt")
    end

    movable.velocity = Vector.zero()
    Co.sleep(0.45, 0.75)
  end)

  self:on("animationEvent", function(trackId, animationName, eventName)
    if eventName == "BarrelGrab" and self.barrelRef then
      self.barrelRef.object:destroy()
      self.holdingBarrel = true
    end
  end)

  self:on("animationEvent", function(trackId, animationName, eventName)
    if eventName == "BarrelThrow" then
      local lookDir = self.character.lookDir

      local pos = self.transform.worldPosition + Vector(math.sign(lookDir.x) * 96.0, 96.0)
      local velocity = lookDir:clone()
      if math.abs(velocity.x) > math.abs(velocity.y) then
        velocity.y = 0.0
      else
        velocity.x = 0.0
      end

      velocity:normalize()

      CreateObject("MiniBossProjectile", pos):addComponent("MiniBossProjectile", {
        velocity = velocity,
        speed = 1024.0
      })
    end
  end)

  local goToBarrel = Co.create(function(barrel)
    self.barrelRef = barrel

    local barrelPosition = barrel.transform.worldPosition
    local p0 = barrelPosition - Vector(160.0, 32.0)
    local p1 = barrelPosition + Vector(0.0, 32.0)

    --Debug.drawRect(p0, p1 - p0, 0xFF00FFFF, true)

    local collideLeft = false
    Physics.aabbQuery(p0, p1, function (object)
      if object == self.object then
        return true
      end

      if object == barrel.object then
        return true
      end

      collideLeft = true
      return false
    end, flags(World.layer("World"), World.layer("MovementCollider")))

    p0 = barrelPosition - Vector(0.0, 32.0)
    p1 = barrelPosition + Vector(160.0, 32.0)

    local collideRight = false
    Physics.aabbQuery(p0, p1, function (object)
      if object == barrel.object then
        return true
      end

      collideRight = true
      return false
    end, flags(World.layer("World"), World.layer("MovementCollider")))

    if collideLeft and collideRight then
      return
    end

    local target = Vector.zero()

    if not collideLeft then
      target = Vector(-64.0, -16.0)
    else
      target = Vector(64.0, -16.0)
    end

    target = target + barrel.transform.worldPosition

    character:setAnimation("walk", true)
    Co.yield(AI.goTo(movable, target))

    character:lookAt(barrel.transform.worldPosition)

    movable.velocity = Vector.zero()
    character:setAnimation("pick")
    AI.yieldWaitForAnimation(self.object, "pick")

    self.barrelRef = nil

    character:setAnimation("walk_barrel", true)
  end)

  local idle = Co.create(function()
    while true do
      movable.velocity = Vector.zero()

      if self.isTired then
        Co.yield(tired(self.tiredTimer))
      elseif self.holdingBarrel then
        character:setAnimation("idle_barrel", true)
      else
        character:setAnimation("idle", true)
      end

      Co.yieldRunWhile(function ()
        if character.target then
          return false
        end

        character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
        return not character.target 
      end,
        function()
          if character.target then
            character:warnAllies(properties.socialRadius)
          elseif Vector.distance(transform.worldPosition, self.spawnPosition) > 40.0 then
            Co.yield(returnToSpawn())
            character:setAnimation("idle", true)
          else
            Co.sleep(0.25)
          end
        end
      )

      local enemy = character.target
      if not enemy then
        return
      end

      Co.yieldRunWhile(function ()
        return enemy.health and enemy.health.isAlive and (not self.barrelRef or self.holdingBarrel)
      end, Co.concurrent(
        Co.series(
          chase(enemy),
          attack(enemy)
        ),
        function()
          while true do
            if not self.holdingBarrel then
              local barrel = self:findBarrelInRange(320.0)
              if barrel then
                if Random.value() <= 0.15 then
                  self.barrelRef = barrel
                  break
                end
              end
            end

            Co.sleep(5.0)
          end
        end
      ))

      if self.barrelRef then
        Co.yield(goToBarrel(self.barrelRef))
      end

      character.target = nil
    end
  end)

  self.idle = idle

  self.getHitFn = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    soundEmitter:postEvent("MiniBossGetHit")
    self.sword:deactivate()

    if self.holdingBarrel then
      character:setAnimation("get_hit_barrel", false)
    elseif self.isTired then
      character:setAnimation("tired_get_hit")
      character:addAnimation("tired_loop", true)
    else
      character:setAnimation("get_hit", false)
    end

    if not self.health.isAlive then
      self.statusEffects:clearAll()
      soundEmitter:postEvent("MiniBossOnDeath")
      
      LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0
      self:removeComponent("Healthbar")

      Co.yield(AI.fadeOut(renderer, 1.0))

      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      SaveGameStats.increase("enemiesSlain")
      return
    end
   
    if self.holdingBarrel then
      AI.yieldWaitForAnimation(self.object, "get_hit_barrel")
    elseif self.isTired then
      AI.yieldWaitForAnimation(self.object, "tired_get_hit")
    else
      AI.yieldWaitForAnimation(self.object, "get_hit")
    end

    if enemy.character and enemy.character.faction ~= character.faction then
      character.target = enemy
    end
  end)

  self.coMain = idle

  self.object:on("resetTarget", function()
    self.idle:reset()
    self.coMain = self.idle
  end)
end

function MiniBoss:fixedUpdate()
  if not self.coMain or not self.coMain:update() then
    self.idle:reset()
    self.coMain = self.idle
  end
end

function MiniBoss:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end

  self.sword = CreateObject("Sword", Vector.zero(), self.object):addComponent("Weapon")
  self.sword:listenForAnimationEvents(self)
  self.sword.staminaMultiplier = 1.5

  self.sword.onApplyEffects = function(object)
    if object.statusEffects then
      local dir = (object.transform.worldPosition - self.transform.worldPosition):normalized()
      object.statusEffects:applyByName("Knockback", { impulse = dir * 3000.0 })
    end
  end
end

function MiniBoss:findBarrelInRange(range)
  local position = self.transform.worldPosition
  local p0 = position - Vector(range, range)
  local p1 = position + Vector(range, range)

  local barrel = nil

  Physics.aabbQuery(p0, p1, function (object)
    if object.barrel then
      local collideLeft, collideRight = self:checkFreeSpaceAroundBarrel(object.barrel)
      if collideLeft and collideRight then
        return true
      end

      barrel = object.barrel
      return false
    end

    return true
  end, World.layer("MovementCollider"))

  return barrel
end

function MiniBoss:checkFreeSpaceAroundBarrel(barrel)
  local barrelPosition = barrel.transform.worldPosition
  p0 = barrelPosition - Vector(160.0, 32.0)
  p1 = barrelPosition + Vector(0.0, 32.0)
  --Debug.drawRect(p0, p1 - p0, 0xFF00FFFF, true)

  local collideLeft = false
  Physics.aabbQuery(p0, p1, function (object)
    if object == barrel.object then
      return true
    end

    collideLeft = true
    return false
  end, flags(World.layer("World"), World.layer("MovementCollider")))

  p0 = barrelPosition - Vector(0.0, 32.0)
  p1 = barrelPosition + Vector(160.0, 32.0)
  --Debug.drawRect(p0, p1 - p0, 0xFF00FFFF, true)

  local collideRight = false
  Physics.aabbQuery(p0, p1, function (object)
    if object == barrel.object then
      return true
    end

    collideRight = true
    return false
  end, flags(World.layer("World"), World.layer("MovementCollider")))

  return collideLeft, collideRight
end
