local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local Mori = Behavior("Mori")

Mori.property("hp", 100.0)
Mori.property("detectionRadius", 500.0)
Mori.property("walkSpeed", 80.0)
Mori.property("healAmount", 25.0)
Mori.property("healDuration", 2.0)
Mori.property("projectileSpeed", 800.0)
Mori.property("aoeDamage", 200.0)
Mori.property("faction", "Monks")
Mori.property("lootTable", "Mori")

Mori.editorIcon("draugar_mini/draugar_mini_down.json")

function Mori:initialize(properties)
  self.properties = Mori.defaultProperties(properties)

  self:addComponent("Character", {
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 30.0,
    colliderDisplacement = Vector(0.0, 8.0),
    colliderDensity = 128.0,
    skeleton = {
      s = "draugar_mini/draugar_mini_down.json",
      n = "draugar_mini/draugar_mini_up.json",
      w = "draugar_mini/draugar_mini_left.json",
      e = "draugar_mini/draugar_mini_right.json"
    }
  })

  self:setupColliders()
  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object
  })

  self.character:setSkeleton("e")
  self.skeletonRenderer:setFlag(self.skeletonRenderer.Flags.Dissolve)

  self:addComponent("Healthbar", Vector(0.0, 100.0))
  self.spawnPosition = self.transform.worldPosition

  self.movable.movementSpeed = self.properties.walkSpeed

  self:setupAI()
end

function Mori:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  local returnToSpawn = Co.create(function ()
    character:setAnimation("walk", true)
    Co.yield(AI.goTo(movable, self.spawnPosition))
    movable.velocity = Vector.zero()
    character:setAnimation("idle", true)
  end)

  local chase = Co.create(function (enemy)
    character.isAlert = true
    character:setAnimation("walk", true)
    Co.yield(AI.chase(movable, enemy.transform, 400.0, 32.0))
  end)

  local sideStep = Co.create(function (enemy, minDistance)
    character:setAnimation("walk", true)

    local enemyPos = enemy.transform.worldPosition
    local myPos = self.transform.worldPosition
    local dist = Vector.distance(enemyPos, myPos)

    sideways = -1.0

    if Random.value() <= 0.5 then
      sideways = 1.0
    end

    while true do
      enemyPos = enemy.transform.worldPosition
      myPos = self.transform.worldPosition
      dist = Vector.distance(enemyPos, myPos)

      movable.velocity = (myPos - enemyPos) / dist
      if sideways ~= 0.0 then
        movable.velocity = movable.velocity:perpendicular() * sideways
      end

      character:lookAt(enemyPos)
      Co.yield()
    end
  end)

  local stepBack = Co.create(function (enemy, minDistance)
    character:setAnimation("walk", true)

    local enemyPos = enemy.transform.worldPosition
    local myPos = self.transform.worldPosition
    local dist = Vector.distance(enemyPos, myPos)

    while true do
      enemyPos = enemy.transform.worldPosition
      myPos = self.transform.worldPosition
      dist = Vector.distance(enemyPos, myPos)

      movable.velocity = (myPos - enemyPos) / dist

      character:lookAt(enemyPos)
      Co.yield()
    end
  end)

  local attack = Co.create(function(enemy)
    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 150.0
    end, stepBack(enemy, 150.0))

    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    character:setAnimation("walk", true)
    Co.sleep(0.15 + Random.value() * 0.25)
    
    renderer.timeScale = 2.0

    self.movable.velocity = Vector.zero()

    local position = self.transform.worldPosition + Vector(0.0, 16.0)
    local target = enemy.movable:predictPosition(0.3) + Vector(0.0, 16.0)
    local dir = (target - position):normalized()
    position = position + dir * 36.0

    self.projectile = CreateObject("MoriProjectile", position):addComponent("MoriProjectile", {
      velocity = (target - position):normalized(),
      parent = self.object,
      speed = self.properties.projectileSpeed,
      parentHitbox = self.bodyHitbox.object,
      healAmount = self.properties.healAmount,
      healDuration = self.properties.healDuration
    })

    while not self.projectile.telegraphDone do
      character:lookAt(enemy.transform.worldPosition)
      position = self.transform.worldPosition + Vector(0.0, 16.0)
      target = enemy.movable:predictPosition(0.3) + Vector(0.0, 16.0)
      dir = (target - position):normalized()
      position = position + dir * 38.0

      self.projectile.transform.localPosition = position
      Co.yield()
    end

    position = self.transform.worldPosition + Vector(0.0, 16.0)
    target = enemy.movable:predictPosition(0.3) + Vector(0.0, 16.0)
    self.projectile.properties.velocity = (target - position):normalized()

    renderer.timeScale = 1.0
  end)

  self.isSuiciding = false

  local suicide = Co.create(function()
    self:removeComponent("Healthbar")

    character.canLock = false
    self.isSuiciding = true
    self.physicsCollider.type = self.physicsCollider.Types.Static
    self.aoeHitBox:startAttack(self.properties.aoeDamage)
    character:setSkeleton("s")
    character:setAnimation("explode")
    AI.yieldWaitForAnimation(self.object, "explode")
    self.object:destroy()
  end)

  local idle = Co.create(function()
    while true do
      movable.velocity = Vector.zero()
      character:setAnimation("idle", true)

      local ally = nil

      Co.yieldThrottle(0.25, function ()
        ally = character:findEntityInRange(self.properties.detectionRadius, function(object)
          if object == self.object then
            return false
          end

          local character = object.character
          local health = object.health

          return character and character.faction == self.properties.faction
            and health and health.isAlive
        end)
      end)

      if not ally then
        local enemy = character:findVisibleEnemyInRange(256.0)
        if enemy then
          Co.yield(suicide())
        end

        return
      end

      local health = ally.health

      Co.yieldRunWhile(function ()
        return health and health.isAlive and health.hp < health.totalHp
      end, Co.series(
        chase(ally),
        attack(ally),
        Co.maybe(
          Co.concurrent(
            sideStep(ally, 150.0),
            Co.waitFor(1.0, 1.75)
          ),
          0.5
        )
      ))
    end
  end)

  self.idle = idle

  self.getHitFn = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    soundEmitter:postEvent("DraugarGetHit")
    character:setAnimation("get_hit", false)

    if not self.health.isAlive then
      soundEmitter:postEvent("DraugarOnDeath")
      self.statusEffects:clearAll()

      LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0

      self:removeComponent("Healthbar")
      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      return
    end
   
    AI.yieldWaitForAnimation(self.object, "get_hit")

    idle:reset()
    self.coMain = idle
  end)

  self.coMain = idle
end

function Mori:fixedUpdate()
  if not self.coMain or not self.coMain:update() then
    self.idle:reset()
    self.coMain = self.idle
  end

  self.aoeHitBox.transform.localPosition = Vector.zero()
end

function Mori:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy)
    if self.isSuiciding then
      return false
    end

    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end

  self.aoeHitBox = CreateObject("AoEHitbox", Vector.zero(), self.object):addComponent("Weapon")
  self.aoeHitBox:listenForAnimationEvents(self)
  local aoeCollider = self.aoeHitBox.physicsCollider
  aoeCollider:addRectangleShape(Vector(0.0, 16.0), Vector(232.0, 140.0), { isSensor = true })
end
