local SaveGameStats = require "util/savegamestats"

local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local Preacher = Behavior("Preacher")

Preacher.property("hp", 175.0)
Preacher.property("detectionRadius", 400.0)
Preacher.property("socialRadius", 600.0)
Preacher.property("damage", 75.0)
Preacher.property("attackRange", 400.0)
Preacher.property("walkSpeed", 85.0)
Preacher.property("projectileSpeed", 500.0)
Preacher.property("spawnDirection", "s")
Preacher.property("faction", "Monks")
Preacher.property("assholeMode", true)
Preacher.property("lootTable", "Preacher")
Preacher.property("patrolRoute", "", Metatype.Route)

Preacher.editorIcon("preacher/preacher_down.json")

function Preacher:initialize(properties)
  self.properties = Preacher.defaultProperties(properties)

  local skeleton = {
    s = "preacher/preacher_down.json",
    n = "preacher/preacher_up.json",
    w = "preacher/preacher_left.json",
    e = "preacher/preacher_right.json"
  }

  self:addComponent("Character", {
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 24.0,
    colliderDisplacement = Vector(0.0, 14.0),
    colliderDensity = 128.0,
    skeleton = skeleton
  })

  self:setupColliders()

  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object
  })

  self.character:setSkeleton(self.properties.spawnDirection)

  CreateObject("FootstepsRenderer", Vector.zero(), self.object):addComponent("FootstepsRenderer")

  self:addComponent("Healthbar", Vector(0.0, 100.0))
  self.spawnPosition = self.transform.worldPosition
  self:setupAI()

  self.movable.movementSpeed = self.properties.walkSpeed
end

function Preacher:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  local returnToSpawn = Co.create(function ()
    character:setAnimation("walk", true)
    Co.yield(AI.goTo(movable, self.spawnPosition))
    movable.velocity = Vector.zero()
    character:setAnimation("idle", true)
  end)

  local chase = Co.create(function (enemy)
    character.isAlert = true
    character:setAnimation("walk", true)
    Co.yield(AI.chase(movable, enemy.transform, self.properties.attackRange, 32.0, 24.0))
  end)

  local sideStep = Co.create(function (enemy)
    character:setAnimation("walk", true)
    movable.movementSpeed = properties.walkSpeed

    local direction = math.sign(Random.value() - 0.5)
    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized():perpendicular() * direction
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local stepBack = Co.create(function (enemy)
    movable.movementSpeed = properties.walkSpeed
    character:setAnimation("walk", true)

    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized()
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local attack = Co.create(function(enemy)
    local dist = Vector.distance(transform.worldPosition, enemy.transform.worldPosition)
    if dist > self.properties.detectionRadius then
      return
    end

    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 150.0
    end, stepBack(enemy, 150.0))

    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    character:setAnimation("walk", true)
    Co.sleep(0.15 + Random.value() * 0.25)
    
    renderer.timeScale = 2.0

    local lineOfSight = false

    Co.yieldConcurrent(function()
      while not lineOfSight do
        character:lookAt(enemy.transform.worldPosition)

        local position = self.transform.worldPosition
        local target = enemy.movable:predictPosition(0.3) + Vector(0.0, 16.0)
        local dir = (target - position):normalized()
        local perp = dir:perpendicular()
        local dist = (target - position):len()

        local v0 = position + dir * 16.0 + perp * 24.0
        local v1 = v0 + dir * dist

        local lineOfSightLeft = true

        Physics.raycast(v0, v1, function (collisionInfo)
          if collisionInfo.object._objectId == 0 then
            lineOfSightLeft = false
            return 0.0
          end

          local character = collisionInfo.object.character 
          local hitbox = collisionInfo.object.hitbox
          if hitbox then
            character = hitbox.transform.parent.object.character
          end

          if character and character.faction == self.character.faction then
            lineOfSightLeft = false
          end

          return 0.0
        end, flags(World.layer("World"), World.layer("Hitbox")))

        v0 = position + dir * 16.0 - perp * 24.0
        v1 = v0 + dir * dist

        local lineOfSightRight = true

        Physics.raycast(v0, v1, function (collisionInfo)
          if collisionInfo.object._objectId == 0 then
            lineOfSightRight = false
            return 0.0
          end

          local character = collisionInfo.object.character 
          local hitbox = collisionInfo.object.hitbox
          if hitbox then
            character = hitbox.transform.parent.object.character
          end

          if character and character.faction == self.character.faction then
            lineOfSightRight = false
          end

          return 0.0
        end, flags(World.layer("World"), World.layer("Hitbox")))

        lineOfSight = lineOfSightLeft and lineOfSightRight

        if lineOfSightLeft and not lineOfSightRight then
          self.movable.velocity = perp
        elseif not lineOfSightLeft then
          self.movable.velocity = -perp
        else
          self.movable.velocity = Vector.zero()
        end

        Co.yield()
      end
    end, Co.waitFor(1.0, 2.0))

    if not lineOfSight and not properties.assholeMode then
      return
    end

    self.movable.velocity = Vector.zero()
    character:setAnimation("telegraph", true)
    soundEmitter:postEvent("PreacherCastProjectile")

    local position = self.transform.worldPosition + Vector(0.0, 16.0)
    local target = enemy.movable:predictPosition(0.3) + Vector(0.0, 16.0)
    local dir = (target - position):normalized()
    position = position + dir * 36.0

    self.projectile = CreateObject("PreacherProjectile", position):addComponent("PreacherProjectile", {
      velocity = (target - position):normalized(),
      parent = self.object,
      speed = self.properties.projectileSpeed,
      parentHitbox = self.bodyHitbox.object,
      damage = self.properties.damage
    })

    while not self.projectile.telegraphDone do
      character:lookAt(enemy.transform.worldPosition)
      position = self.transform.worldPosition + Vector(0.0, 16.0)
      target = enemy.movable:predictPosition(0.3) + Vector(0.0, 16.0)
      dir = (target - position):normalized()
      position = position + dir * 38.0

      self.projectile.transform.localPosition = position
      Co.yield()
    end

    soundEmitter:postEvent("PreacherThrowProjectile")
    position = self.transform.worldPosition + Vector(0.0, 16.0)
    target = enemy.movable:predictPosition(0.3) + Vector(0.0, 16.0)
    self.projectile.properties.velocity = (target - position):normalized()

    renderer.timeScale = 1.0

    character:addAnimation("idle", true)
    Co.sleep(2.0)

    if not Physics.lineOfSight(self.transform.worldPosition, enemy.transform.worldPosition) then
      Co.yield(chase(enemy))
    end
  end)

  self.idle = Co.create(function()
    while true do
      movable.velocity = Vector.zero()
      character:setAnimation("idle", true)

      Co.yieldRunWhile(function ()
        if character.target then
          return false
        end

        character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
        return not character.target 
      end,
        function()
          if character.target then
            character:warnAllies(properties.socialRadius)
          elseif Vector.distance(transform.worldPosition, self.spawnPosition) > 40.0 then
            Co.yield(returnToSpawn())
            character:setAnimation("idle", true)
          else
            Co.sleep(0.25)
          end
        end
      )

      local enemy = character.target
      if not enemy then
        return
      end

      Co.yieldRunWhile(function ()
        return enemy.health
        and enemy.health.isAlive
      end, Co.series(
        chase(enemy),
        attack(enemy),
        Co.maybe(
          Co.concurrent(
            sideStep(enemy, 150.0),
            Co.waitFor(1.0, 1.75)
          ),
          0.5
        )
      ))

      character.target = nil
    end
  end)

  self.followPatrolRoute = Co.create(function()
    local patrolRoute = WorldUtil.getRouteByName(properties.patrolRoute)
    if not patrolRoute then
      print("ERROR - Invalid patrol route name \"" .. properties.patrolRoute .. "\"")
      return
    end

    character:setAnimation("walk", true)

    Co.yieldRunWhile(function ()
      if character.target then
        return
      end
      
      character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
      return not character.target 
    end, AI.followPatrolRoute(movable, patrolRoute))

    if character.target then
      character:warnAllies(properties.socialRadius)
    end

    Co.yield(self.idle())
  end)

  self.getHitFn = Co.create(function(enemy)
    if self.projectile then
      self.projectile:selfDestruct()
      self.projectile = nil
    end

    movable.velocity = Vector.zero()
    soundEmitter:postEvent("PreacherGetHit")
    character:setAnimation("get_hit", false)

    if not self.health.isAlive then
      self.statusEffects:clearAll()
      soundEmitter:postEvent("PreacherOnDeath")

      LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0
      self:removeComponent("Healthbar")
      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      SaveGameStats.increase("enemiesSlain")
      return
    end
    
    AI.yieldWaitForAnimation(self.object, "get_hit")
    
    if enemy.character and enemy.character.faction ~= character.faction then
      character.target = enemy
    end

    self.idle:reset()
    self.coMain = self.idle
  end)

  self.object:on("resetTarget", function()
    self.idle:reset()
    self.coMain = self.idle
  end)
end

function Preacher:fixedUpdate()
  if not self.coMain or not self.coMain:update() then
    if self.properties.patrolRoute == "" then
      self.idle:reset()
      self.coMain = self.idle
    else
      self.followPatrolRoute:reset()
      self.coMain = self.followPatrolRoute
    end
  end
end

function Preacher:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end
end
