local SaveGameStats = require "util/savegamestats"

local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"

local SnowDraugur = Behavior("SnowDraugur")

SnowDraugur.property("hp", 160.0)
SnowDraugur.property("detectionRadius", 300.0)
SnowDraugur.property("socialRadius", 600.0)
SnowDraugur.property("damage", 50.0)
SnowDraugur.property("walkSpeed", 104.0)
SnowDraugur.property("variant", "default")
SnowDraugur.property("spawnDirection", "s")
SnowDraugur.property("faction", "Monks")
SnowDraugur.property("lootTable", "SnowDraugur")
SnowDraugur.property("patrolRoute", "", Metatype.Route)

SnowDraugur.editorIcon("draugur_snow/draugur_snow_down.json")

function SnowDraugur:initialize(properties)
  self.properties = SnowDraugur.defaultProperties(properties)

  local skeleton = {
    s = "draugur_snow/draugur_snow_down.json",
    n = "draugur_snow/draugur_snow_up.json",
    w = "draugur_snow/draugur_snow_left.json",
    e = "draugur_snow/draugur_snow_right.json"
  }

  self:addComponent("Character", {
    faction = self.properties.faction,
    hp = self.properties.hp,
    collisionRadius = 24.0,
    colliderDisplacement = Vector(0.0, 14.0),
    colliderDensity = 128.0,
    skeleton = skeleton
  })

  self:setupColliders()

  self.character:setLinkedColliders({
    bb_body = self.bodyHitbox.object,
    bb_hand = self.weaponBox.object
  })

  self.character:setSkeleton(self.properties.spawnDirection)

  self:addComponent("Healthbar", Vector(0.0, 100.0))
  self.spawnPosition = self.transform.worldPosition
  self:setupAI()

  self.movable.movementSpeed = self.properties.walkSpeed
end

function SnowDraugur:setupAI()
  local renderer = self.skeletonRenderer
  local character = self.character
  local properties = self.properties
  local transform = self.transform
  local movable = self.movable
  local soundEmitter = self.soundEmitter

  local returnToSpawn = Co.create(function ()
    character:setAnimation("walk", true)
    Co.yield(AI.goTo(movable, self.spawnPosition))
    movable.velocity = Vector.zero()
    character:setAnimation("idle", true)
  end)

  local chase = Co.create(function (enemy)
    character.isAlert = true
    character:setAnimation("walk", true)
    Co.yield(AI.chase(movable, enemy.transform, 64.0))
  end)

  local sideStep = Co.create(function (enemy)
    character:setAnimation("walk", true)
    movable.movementSpeed = properties.walkSpeed

    local direction = math.sign(Random.value() - 0.5)
    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized():perpendicular() * direction
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local stepBack = Co.create(function (enemy)
    character:setAnimation("walk", true)
    movable.movementSpeed = properties.walkSpeed

    while true do
      local enemyPos = enemy.transform.worldPosition
      local myPos = self.transform.worldPosition
      movable.velocity = (myPos - enemyPos):normalized()
      character:lookAt(enemyPos)
      Co.yield()

      if movable.blocked then
        return
      end
    end
  end)

  local attack = Co.create(function(enemy)
    Co.yieldRunWhile(function ()
      return Vector.distance(transform.worldPosition, enemy.transform.worldPosition) < 55.0
    end, stepBack(enemy))

    character:setAnimation("walk", true)
    Co.yield(AI.alignForAttack(movable, enemy.transform))

    soundEmitter:postEvent("SnowDraugarAttack")
    movable.velocity = Vector.zero()
    character:lookAt(enemy.transform.worldPosition)

    self.weaponBox:startAttack(properties.damage)

    character:setAnimation("hit")
    AI.yieldWaitForAnimation(self.object, "hit")

    character:setAnimation("idle")
    Co.sleep(0.35, 0.85)
  end)

  local idle = Co.create(function()
    while true do
      movable.velocity = Vector.zero()
      character:setAnimation("idle", true)

      Co.yieldRunWhile(function ()
        if character.target then
          return false
        end

        character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
        return not character.target 
      end,
        function()
          if character.target then
            character:warnAllies(properties.socialRadius)
          elseif Vector.distance(transform.worldPosition, self.spawnPosition) > 40.0 then
            Co.yield(returnToSpawn())
            character:setAnimation("idle", true)
          else
            Co.sleep(0.25)
          end
        end
      )

      local enemy = character.target
      if not enemy then
        return
      end

      Co.yieldRunWhile(function ()
        return enemy.health and enemy.health.isAlive
      end, Co.series(
        chase(enemy),
        attack(enemy),
        Co.maybe(
          Co.concurrent(
            sideStep(enemy),
            Co.waitFor(0.85, 1.45)
          ),
          0.45
        )
      ))

      character.target = nil
    end
  end)

  self.idle = idle

  self.followPatrolRoute = Co.create(function()
    local patrolRoute = WorldUtil.getRouteByName(properties.patrolRoute)
    if not patrolRoute then
      print("ERROR - Invalid patrol route name \"" .. properties.patrolRoute .. "\"")
      return
    end

    character:setAnimation("walk", true)

    Co.yieldRunWhile(function ()
      if character.target then
        return
      end
      
      character.target = character:findVisibleEnemyInRange(properties.detectionRadius)
      return not character.target 
    end, AI.followPatrolRoute(movable, patrolRoute))

    if character.target then
      character:warnAllies(properties.socialRadius)
    end

    Co.yield(self.idle())
  end)

  self.getHitFn = Co.create(function(enemy)
    movable.velocity = Vector.zero()
    soundEmitter:postEvent("DraugarGetHit")
    character:setAnimation("get_hit", false)

    if not self.health.isAlive then
      self.statusEffects:clearAll()
      soundEmitter:postEvent("DraugarOnDeath")
      self.weaponBox:deactivate()
      
      LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)

      self.physicsCollider.mask = 0
      self.physicsCollider.categories = 0
      self:removeComponent("Healthbar")
      CreateObject("Puff", self.transform.worldPosition + Vector(0.0, 24.0)):addComponent("Puff")
      self:removeComponent("SkeletonRenderer")

      Co.sleep(2.0)
      self.object:destroy()
      SaveGameStats.increase("enemiesSlain")
      return
    end
   
    AI.yieldWaitForAnimation(self.object, "get_hit")
    
    if enemy.character and enemy.character.faction ~= character.faction then
      character.target = enemy
    end

    self.coMain = Co.maybe(
      Co.pickOne(
        attack(enemy),
        Co.concurrent(
          stepBack(enemy),
          Co.waitFor(0.65, 1.25)
        ),
        Co.concurrent(
          sideStep(enemy),
          Co.waitFor(0.65, 1.25)
        ),
        { 0.3, 0.35, 0.35 }
      )
    , 0.5)
  end)
  
  self.object:on("resetTarget", function()
    self.idle:reset()
    self.coMain = self.idle
  end)
  
  self.coMain = idle
end

function SnowDraugur:fixedUpdate()
  if not self.coMain or not self.coMain:update() then
    if self.properties.patrolRoute == "" then
      self.idle:reset()
      self.coMain = self.idle
    else
      self.followPatrolRoute:reset()
      self.coMain = self.followPatrolRoute
    end
  end
end

function SnowDraugur:setupColliders()
  self.bodyHitbox = CreateObject("BodyHitbox", Vector.zero(), self.object):addComponent("Hitbox")
  self.bodyHitbox:enableEventHandling()

  self.bodyHitbox.onHit = function (enemy)
    self.getHitFn:reset()
    self.coMain = self.getHitFn(enemy)
    return true
  end

  self.weaponBox = CreateObject("Hand", Vector.zero(), self.object):addComponent("Weapon")
  self.weaponBox:listenForAnimationEvents(self)
  self.weaponBox.onApplyEffects = function (object)
    local statusEffects = object.statusEffects
    if not statusEffects then
      return
    end

    if Random.value() >= 0.5 then
      return
    end

    statusEffects:applyByName("Frozen", { duration = 3.0 })
  end
end
