local Runes = require "util/runes"
local SaveGameManager = require "util/savegamemanager"

local ItemDefs = require "data/itemdefs"

local AltarView = Behavior("AltarView")

function AltarView:initialize()
  self:on("destroy", function()
    if self.visible then
      UI.detachView("AltarView")
      Kernel.menuEvents:releaseLock("Altar")
    end
  end)

  self.visible = false
  self.selectedIndex = 1
  self.offers = {}
  self.fused = false

  Kernel.inputController:on("menuBack", function (event)
    if not self.visible then
      return
    end

    event.handled = true
    self:hide()
  end)

  Kernel.inputController:on("menuSelect", function (event)
    if not self.visible then
      return
    end

    self:fuse()
    event.handled = true
  end)

  Kernel.inputController:on("menuOption", function (event)
    if not self.visible then
      return
    end

    self:reroll()
    event.handled = true
  end)

  Kernel.inputController:on("menuUp", function()
    if not self.visible then
      return
    end

    if self.selectedIndex > 1 then
      self.selectedIndex = self.selectedIndex - 1
      self:updateBindings()
    end
  end)

  Kernel.inputController:on("menuDown", function()
    if not self.visible then
      return
    end

    if self.selectedIndex < #self.offers then
      self.selectedIndex = self.selectedIndex + 1
      self:updateBindings()
    end
  end)
end

function AltarView:awake()
  self.heroRef = BehaviorUtil.findFirstOfType("Hero")
end

function AltarView:updateBindings()
  local offers = {}

  for i, runeName in ipairs(Runes.offered) do
    local def = ItemDefs[runeName]

    offers[#offers+1] = {
      id = runeName,
      def = def
    }
  end

  self.offers = offers

  local shards = 0
  if self.heroRef.inventory.items.RuneShard then
    shards = self.heroRef.inventory.items.RuneShard
  end

  local bindings = {
    inputDevice = Kernel.inputController.device,
    buttons = Kernel.inputController:getKeyboardBindings(),
    offers = offers,
    selectedIndex = self.selectedIndex,
    selectedRune = offers[self.selectedIndex],
    canReroll = self.heroRef.inventory.coins >= 1000.0,
    shards = shards
  }

  UI.updateBinding("AltarView", bindings)
end

function AltarView:show()
  if not self.visible and Kernel.menuEvents:takeLock("Altar") then
    UI.attachView("AltarView", "ui/altar.rml")
    self.visible = true
    self.fused = false
    Kernel.soundEmitter:postEvent("WindowShow")

    if #Runes.offered < 3 then
      self:refreshOffers()
    end

    self:updateBindings()
  end
end

function AltarView:hide()
  if not self.visible then
    return
  end

  UI.detachView("AltarView")
  Kernel.menuEvents:releaseLock("Altar")
  Kernel.soundEmitter:postEvent("WindowHide")

  if self.fused then
    self.heroRef.actor:stopAllScripts()
    self.heroRef.actor:playScript("runeFound")
    self.fused = false
  end

  self.visible = false
end

function AltarView:refreshOffers()
  local uncollected = Runes.getUncollected(self.heroRef)
  local offers = {}

  if #uncollected <= 3 then
    for i, rune in ipairs(uncollected) do
      offers[rune] = true
    end
  else
    local count = 0
    while count < 3 do
      local rune = table.pickOne(uncollected)
      if not offers[rune] then
        offers[rune] = true
        count = count + 1
      end
    end
  end

  offers = table.keys(offers)
  Runes.offered = offers
  SaveGameManager.save()
end

function AltarView:fuse()
  local cost = self.offers[self.selectedIndex].def.shardCost
  if not self.heroRef.inventory:containsItems("RuneShard", cost) then
    return
  end

  self.heroRef.inventory:removeItems("RuneShard", cost)

  local runeName = self.offers[self.selectedIndex].id
  self.heroRef.inventory:addItems(runeName, 1)

  local uncollected = Runes.getUncollected(self.heroRef)
  if #uncollected > 0 then
    local newRune = table.pickOne(uncollected)
    Runes.offered[self.selectedIndex] = newRune
  else
    local offered = {}
    for i, r in ipairs(Runes.offered) do
      if runeName ~= r then
        offered[#offered+1] = runeName
      end
    end

    Runes.offered = offered
  end

  Kernel.soundEmitter:postEvent("ItemBought")
  SaveGameManager.save()
  self:updateBindings()
  self.fused = true
end

function AltarView:reroll()
  local inventory = self.heroRef.inventory
  if inventory.coins >= 1000.0 then
    inventory.coins = inventory.coins - 1000.0
  else
    return
  end

  self:refreshOffers()
  Kernel.soundEmitter:postEvent("ItemBought")
  self:updateBindings()
end
