local SaveGameStats = require "util/savegamestats"

local ItemDefs = require "data/itemdefs"

local ContainerView = Behavior("ContainerView")

function ContainerView:initialize()
  self:on("destroy", function()
    if self.visible then
      UI.detachView("ContainerView")
      Kernel.menuEvents:releaseLock("Container")
    end
  end)

  self.visible = false

  self.items = {}
  self.selectedIndex = 1

  Kernel.inputController:on("menuBack", function (event)
    if not self.visible then
      return
    end

    event.handled = true
    self:hide()
  end)

  Kernel.inputController:on("menuSelect", function (event)
    if not self.visible then
      return
    end

    event.handled = true
    self:takeAll()
  end)

  Kernel.inputController:on("menuOption", function (event)
    if not self.visible then
      return
    end

    event.handled = true
    self:takeSelectedItem()
  end)

  Kernel.inputController:on("menuUp", function()
    if not self.visible then
      return
    end

    if self.selectedIndex > 1 then
      self.selectedIndex = self.selectedIndex - 1
      self:updateBindings()
      Kernel.soundEmitter:postEvent("MenuClick")
    end
  end)

  Kernel.inputController:on("menuDown", function()
    if not self.visible then
      return
    end

    if self.selectedIndex < #self.items then
      self.selectedIndex = self.selectedIndex + 1
      self:updateBindings()
      Kernel.soundEmitter:postEvent("MenuClick")
    end
  end)

  self.scroll = 0.0
end

function ContainerView:awake()
  self.heroRef = BehaviorUtil.findFirstOfType("Hero")
  self:refreshItems()
end

function ContainerView:update(deltaTime)
  if not self.visible then
    return
  end

  local targetScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 then
    targetScroll = 0.0
  end

  if self.scroll ~= targetScroll then
    self:updateBindings()
  end
end

function ContainerView:updateBindings()
  local targetScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 then
    targetScroll = 0.0
  end

  self.scroll = self.scroll + (targetScroll - self.scroll) * Time.deltaTime * 8.0

  local bindings = {
    inputDevice = Kernel.inputController.device,
    buttons = Kernel.inputController:getKeyboardBindings(),
    items = self.items,
    selectedIndex = self.selectedIndex,
    selectedItem = self.items[self.selectedIndex],
    itemCount = #self.items,
    itemListScroll = self.scroll
  }

  UI.updateBinding("ContainerView", bindings)
end

function ContainerView:show(inventoryRef, closeCallback)
  self.inventoryRef = inventoryRef
  self.closeCallback = closeCallback

  if not self.inventoryRef then
    return
  end

  if not self.visible and Kernel.menuEvents:takeLock("Container") then
    UI.attachView("ContainerView", "ui/containermenu.rml")
    self.visible = true
    self:refreshItems()
    Kernel.soundEmitter:postEvent("WindowShow")

    self.runePickedUp = false
  end
end

function ContainerView:hide()
  if not self.visible then
    return
  end

  self.inventoryRef = nil

  UI.detachView("ContainerView")
  Kernel.menuEvents:releaseLock("Container")
  Kernel.soundEmitter:postEvent("WindowHide")

  self.visible = false
  if self.closeCallback then
    self.closeCallback()
    self.closeCallback = nil
  end

  if self.runePickedUp then
    self.heroRef.actor:stopAllScripts()
    self.heroRef.actor:playScript("runeFound")

    self.runePickedUp = false
  end
end

function ContainerView:refreshItems()
  if not self.inventoryRef then
    return
  end

  local itemsByType = {}
  for itemId, qty in pairs(self.inventoryRef.items) do
    local def = ItemDefs[itemId]

    local items = itemsByType[def.type]
    if not items then
      itemsByType[def.type] = {}
      items = itemsByType[def.type]
    end

    items[#items+1] = {
      def = def,
      id = itemId,
      qty = qty
    }
  end

  self.items = {}

  if self.inventoryRef.coins > 0 then
    self.items[#self.items+1] = {
      def = {
        name = "Gold",
        icon = "item-icon gold",
        type = "Gold"
      },
      id = 0,
      qty = self.inventoryRef.coins
    }
  end

  for type, items in pairs(itemsByType) do
    for i, item in ipairs(items) do
      self.items[#self.items + 1] = item
    end
  end

  if self.selectedIndex >= #self.items then
    self.selectedIndex = #self.items
  elseif self.selectedIndex == 0 then
    self.selectedIndex = 1
  end

  self:updateBindings()
end

function ContainerView:takeSelectedItem()
  local item = self.items[self.selectedIndex]
  if not item or not self.heroRef then
    return
  end

  if item.id == 0 then
    SaveGameStats.increase("goldCollected", self.inventoryRef.coins)
    self.inventoryRef:transferCoins(self.heroRef.inventory, self.inventoryRef.coins)
  else
    self.inventoryRef:transferItems(self.heroRef.inventory, item.id, item.qty)
  end

  if item.def.type == "Rune" then
    self.runePickedUp = true
  end

  self:refreshItems()

  if self.inventoryRef:isEmpty() then
    self:hide()
  end
end

function ContainerView:takeAll()
  if not self.heroRef then
    return
  end

  for i, item in ipairs(self.items) do
    if item.id == 0 then
      SaveGameStats.increase("goldCollected", self.inventoryRef.coins)
      self.inventoryRef:transferCoins(self.heroRef.inventory, self.inventoryRef.coins)
    else
      self.inventoryRef:transferItems(self.heroRef.inventory, item.id, item.qty)

      if item.def.type == "Rune" then
        self.runePickedUp = true
      end
    end
  end

  self:hide()
end
