local SaveGameStats = require "util/savegamestats"

local ItemDefs = require "data/itemdefs"
local Runes = require "util/runes"

local InventoryView = Behavior("InventoryView")

function InventoryView:initialize()
  self:on("destroy", function()
    if self.visible then
      UI.detachView("InventoryView")
      Kernel.menuEvents:releaseLock("Inventory")
    end
  end)

  self.visible = false

  self.items = {}
  self.selectedIndex = 0

  self.tabs = { "rune", "potion", "skull" }
  self.tabItemTypes = { "None", "Potion", "KeyItem" }
  self.tabIndex = 1

  self.runesPerRow = 10

  Kernel.inputController:on("openInventory", function(event)
    if not self.heroRef.health.isAlive then
      return
    end

    if self.visible then
      self:hide()
    else
      self:show()
    end

    event.handled = true
  end)

  Kernel.inputController:on("menuBack", function(event)
    if not self.visible then
      return
    end

    self:hide()
    event.handled = true
  end)

  Kernel.inputController:on("menuUp", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.mapToQuickbar then
      self:mapSelectedItemToQuickbar("up")
      return
    end

    if self.tabIndex == 1 then
      if self.selectedIndex == 1 then
        self.selectedIndex = 0
      else
        self.selectedIndex = self.selectedIndex - self.runesPerRow
        if self.selectedIndex < 0 then
          self.selectedIndex = 0
        end
      end
    else
      if self.selectedIndex > 1 then
        self.selectedIndex = self.selectedIndex - 1
        self:updateBindings()
        Kernel.soundEmitter:postEvent("MenuClick")
      end
    end
  end)

  Kernel.inputController:on("menuDown", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.mapToQuickbar then
      self:mapSelectedItemToQuickbar("down")
      return
    end

    if self.tabIndex == 1 then
      if self.selectedIndex == 0 and #self.runes > 0 then
        self.selectedIndex = 1
      elseif self.selectedIndex + self.runesPerRow <= #self.runes then
        self.selectedIndex = self.selectedIndex + self.runesPerRow
        if self.selectedIndex >= #self.runes then
          self.selectedIndex = #self.runes
        end
      end
    else
      if self.selectedIndex < #self.items then
        self.selectedIndex = self.selectedIndex + 1
      end
    end

    self:updateBindings()
    Kernel.soundEmitter:postEvent("MenuClick")
  end)

  Kernel.inputController:on("menuLeft", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.mapToQuickbar then
      self:mapSelectedItemToQuickbar("left")
      return
    end

    if self.tabIndex == 1 then
      if self.selectedIndex > 1 then
        self.selectedIndex = self.selectedIndex - 1
      end
    elseif self.tabIndex > 1 then
      self.alpha = 0.0
      self.tabIndex = self.tabIndex - 1
      if self.tabIndex == 1 then
        self.selectedIndex = 0
      end

      self:refreshItems()
    end

    Kernel.soundEmitter:postEvent("MenuClick")
    self:updateBindings()
  end)

  Kernel.inputController:on("menuRight", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.mapToQuickbar then
      self:mapSelectedItemToQuickbar("right")
      return
    end

    if self.tabIndex == 1 then
      if self.selectedIndex == 0 then
        self.alpha = 0.0
        self.tabIndex = self.tabIndex + 1
        self:refreshItems()
      else
        self.selectedIndex = self.selectedIndex + 1
        if self.selectedIndex > #self.runes then
          self.selectedIndex = #self.runes
        end
      end
    elseif self.tabIndex < #self.tabs then
      self.alpha = 0.0
      self.tabIndex = self.tabIndex + 1
      self:refreshItems()
    end
    
    Kernel.soundEmitter:postEvent("MenuClick")
    self:updateBindings()
  end)

  self.mapToQuickbar = false
  Kernel.inputController:on("menuOption", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.items[self.selectedIndex].def.type == "Potion" then
      self.mapToQuickbar = true
      self:updateBindings()
    end
  end)

  Kernel.inputController:on("menuSelect", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.tabIndex == 1 then
      if self.selectedIndex > 0 and self.heroRef.canChangeRunes then
        local index = self.selectedIndex

        local runeName = self.runes[index].id
        if Runes.isEquipped(runeName) then
          Runes.unequip(runeName, self.heroRef)
        else
          Runes.equip(runeName, self.heroRef)
        end
      end

      self:updateBindings()
    else
      self:useConsumable()
    end
  end)

  self.scroll = 0.0
end

function InventoryView:awake()
  self.heroRef = BehaviorUtil.findFirstOfType("Hero")
  self:refreshItems()
end

function InventoryView:update(deltaTime)
  if not self.heroRef or not self.visible then
    return
  end

  if not self.heroRef.health.isAlive then
    self:hide()
    return
  end

  local targetScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 then
    targetScroll = 0.0
  end

  if self.scroll ~= targetScroll then
    self:updateBindings()
  end
end

function InventoryView:updateBindings()
  local targetScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 then
    targetScroll = 0.0
  end

  self.scroll = self.scroll + (targetScroll - self.scroll) * Time.deltaTime * 8.0

  local collectedRunes = Runes.getCollected(self.heroRef)
  local runes = {}
  for i, runeName in ipairs(collectedRunes) do
    local rune = Runes.getRune(runeName)

    runes[#runes+1] = {
      id = rune.name,
      def = ItemDefs[runeName],
      equipped = Runes.isEquipped(runeName)
    }
  end

  self.runes = {}
  for i, rune in ipairs(runes) do
    self.runes[i] = runes[i]
  end

  for i=#runes,19 do
    runes[#runes+1] = {
      id = "NotFound",
      def = { icon = "rune-icon not-found" },
      equipped = false
    }
  end

  local inventory = self.heroRef.inventory

  local shards = 0
  if inventory.items.RuneShard then
    shards = inventory.items.RuneShard
  end

  local bindings = {
    inputDevice = Kernel.inputController.device,
    buttons = Kernel.inputController:getKeyboardBindings(),
    items = self.items,
    selectedIndex = self.selectedIndex,
    selectedItem = self.items[self.selectedIndex],
    tabs = self.tabs,
    tabIndex = self.tabIndex,
    mapToQuickbar = self.mapToQuickbar,
    coins = inventory.coins,
    itemListScroll = self.scroll,
    runes = runes,
    selectedRune = self.runes[self.selectedIndex],
    hp = self.heroRef.health.hp,
    totalHp = self.heroRef.health.totalHp,
    dmg = self.heroRef.baseWeaponDamage,
    canChangeRunes = self.heroRef.canChangeRunes,
    shards = shards,
    stats = SaveGameStats.getStats()
  }

  UI.updateBinding("InventoryView", bindings)
end

function InventoryView:show()
  if not self.visible and Kernel.menuEvents:takeLock("Inventory") then
    UI.attachView("InventoryView", "ui/inventory.rml")
    self.visible = true
    self:refreshItems()
    Kernel.soundEmitter:postEvent("WindowShow")
  end
end

function InventoryView:hide()
  if not self.visible then
    return
  end

  UI.detachView("InventoryView")
  Kernel.menuEvents:releaseLock("Inventory")
  Kernel.soundEmitter:postEvent("WindowHide")

  self.visible = false
  self.mapToQuickbar = false
end

function InventoryView:refreshItems()
  if not self.heroRef then
    return
  end

  self.items = {}

  for itemId, qty in pairs(self.heroRef.inventory.items) do
    local def = ItemDefs[itemId]

    if def.type ~= "Rune" then
      self.items[#self.items + 1] = {
        def = def,
        id = itemId,
        qty = qty
      }
    end
  end

  table.sort(self.items, function (a, b)
    return a.id < b.id
  end)

  if self.tabIndex == 2 then
    if self.selectedIndex >= #self.items then
      self.selectedIndex = #self.items
    elseif self.selectedIndex == 0 then
      self.selectedIndex = 1
    end
  end

  self:updateBindings()
end

function InventoryView:mapSelectedItemToQuickbar(slot)
  self.mapToQuickbar = false

  if self.selectedIndex == 0 then
    return
  end

  local slots = self.heroRef.quickBar.slots
  local itemId = self.items[self.selectedIndex].id

  for slot, slotItem in pairs(slots) do
    if slotItem == itemId then
      slots[slot] = nil
    end
  end

  if slots[slot] == itemId then
    slots[slot] = nil
  else
    slots[slot] = itemId
  end

  self:updateBindings()
end

function InventoryView:useConsumable()
  if self.selectedIndex == 0 then
    return
  end

  local itemId = self.items[self.selectedIndex].id
  if self.items[self.selectedIndex].def.type ~= "Potion" then
    return
  end

  local inventory = self.heroRef.inventory
  local qty = inventory.items[itemId]
  if not qty or qty == 0 then
    return
  end

  self.heroRef.character:useConsumable(itemId)
  self:hide()
end
