local InputEvents = require "lib/inputevents"

local OptionsMenu = Behavior("OptionsMenu")

function OptionsMenu:initialize()
  self:on("destroy", function()
    if self.visible then
      UI.detachView("OptionsMenu")
      Kernel.menuEvents:releaseLock("OptionsMenu")
    end
  end)

  self.visible = false
  self.tabIndex = 1
  self.itemIndex = 1

  Kernel.inputController:on("menuSelect", function(event)
    if not self.visible or self.rebindingIndex ~= 0 then
      return
    end

    event.handled = true
    Kernel.soundEmitter:postEvent("MenuClick")

    if self.tabIndex == 1 then
      self:handleGraphicsTabEvent("menuSelect")
    elseif self.tabIndex == 4 then
      self:handleKeyboardTabEvent("menuSelect")
    end
  end)

  Kernel.inputController:on("menuBack", function(event)
    if not self.visible or self.rebindingIndex ~= 0 then
      return
    end

    event.handled = true
    self:hide()
  end)

  Kernel.inputController:on("menuLeft", function(event)
    if not self.visible then
      return
    end

    event.handled = true
    Kernel.soundEmitter:postEvent("MenuClick")

    if self.itemIndex == 1 and self.tabIndex > 1 then
      self.tabIndex = self.tabIndex - 1
      self:updateBindings()
      Kernel.soundEmitter:postEvent("MenuClick")
    elseif self.tabIndex == 1 then
      self:handleGraphicsTabEvent("menuLeft")
    elseif self.tabIndex == 2 then
      self:handleSoundTabEvent("menuLeft")
    end
  end)

  Kernel.inputController:on("menuRight", function(event)
    if not self.visible then
      return
    end

    event.handled = true
    Kernel.soundEmitter:postEvent("MenuClick")

    if self.itemIndex == 1 and self.tabIndex < 4 then
      self.tabIndex = self.tabIndex + 1
      self:updateBindings()
      Kernel.soundEmitter:postEvent("MenuClick")
    elseif self.tabIndex == 1 then
      self:handleGraphicsTabEvent("menuRight")
    elseif self.tabIndex == 2 then
      self:handleSoundTabEvent("menuRight")
    end
  end)

  Kernel.inputController:on("menuUp", function(event)
    if not self.visible or self.rebindingIndex ~= 0 then
      return
    end

    event.handled = true
    Kernel.soundEmitter:postEvent("MenuClick")

    if self.itemIndex > 1 then
      self.itemIndex = self.itemIndex - 1
      self:updateBindings()
      Kernel.soundEmitter:postEvent("MenuClick")
    end
  end)

  Kernel.inputController:on("menuDown", function(event)
    if not self.visible or self.rebindingIndex ~= 0 then
      return
    end

    event.handled = true
    Kernel.soundEmitter:postEvent("MenuClick")

    local maxIndex = 1
    if self.tabIndex == 1 then
      maxIndex = 4
    elseif self.tabIndex == 2 then
      maxIndex = 5
    elseif self.tabIndex == 4 then
      maxIndex = #self.keyBindings + 2
    end

    if self.itemIndex < maxIndex then
      self.itemIndex = self.itemIndex + 1
      self:updateBindings()
      Kernel.soundEmitter:postEvent("MenuClick")
    end
  end)

  self.screenSize = Vector(
    Config.getKey("screen", "width", "int"),
    Config.getKey("screen", "height", "int")
  )

  self.fullscreen = Config.getKey("screen", "fullscreen", "bool")

  local videoModes = Screen.getAvailableVideoModes()
  self.videoModes = {}
  local c = {}

  for i, mode in ipairs(videoModes) do
    local vm = tostring(mode.width) .. "x" .. tostring(mode.height)
    if not c[vm] then
      c[vm] = true
      self.videoModes[#self.videoModes+1] = mode
    end
  end

  self.videoModeIndex = 1
  for i, mode in ipairs(self.videoModes) do
    if self.screenSize.x == mode.width and self.screenSize.y == mode.height then
      self.videoModeIndex = i
    end
  end

  self.masterVolume = Config.getKey("sound", "masterVolume", "float")
  self.musicVolume = Config.getKey("sound", "musicVolume", "float")
  self.sfxVolume = Config.getKey("sound", "sfxVolume", "float")
  self.ambientVolume = Config.getKey("sound", "ambientVolume", "float")

  self.keyBindings = {
    {
      name = "Pause",
      bindingType = "action",
      bindingName = "pause"
    },
    {
      name = "Walk left",
      bindingType = "range",
      bindingName = "movement",
      bindingValue = Vector(-1.0, 0.0)
    },
    {
      name = "Walk right",
      bindingType = "range",
      bindingName = "movement",
      bindingValue = Vector(1.0, 0.0)
    },
    {
      name = "Walk up",
      bindingType = "range",
      bindingName = "movement",
      bindingValue = Vector(0.0, 1.0)
    },
    {
      name = "Walk down",
      bindingType = "range",
      bindingName = "movement",
      bindingValue = Vector(0.0, -1.0)
    },
    {
      name = "Attack",
      bindingType = "action",
      bindingName = "attack"
    },
    {
      name = "Block",
      bindingType = "state",
      bindingName = "block"
    },
    {
      name = "Evade",
      bindingType = "action",
      bindingName = "evade"
    },
    {
      name = "Toggle Lock On",
      bindingType = "action",
      bindingName = "lockOn"
    },
    {
      name = "Lock On Closest Enemy",
      bindingType = "action",
      bindingName = "lockOnClosest"
    },
    {
      name = "Inventory",
      bindingType = "action",
      bindingName = "openInventory"
    },
    {
      name = "Interact",
      bindingType = "action",
      bindingName = "interact"
    }
  }

  self.rebindingIndex = 0
  self.timeout = 0.0

  InputEvents:on("event", function (event)
    if self.rebindingIndex == 0 or self.timeout > 0.0 then
      return
    end

    if event.source ~= "Keyboard" or event.type ~= "ButtonPressed" then
      return
    end

    Kernel.soundEmitter:postEvent("MenuClick")
    self:rebindKey(event.button)
    self:updateBindings()

    self.rebindingIndex = 0
  end)

  self:refreshKeyBindings()
end

function OptionsMenu:rebindKey(key)
  local binding = self.keyBindings[self.rebindingIndex]
  if not binding then
    return
  end

  local context = Kernel.inputController.gameplayContext
  if binding.bindingType == "action" then
    local bindings = context.actions[binding.bindingName]
    if bindings then
      local item = table.find(bindings, function (item)
        return item.device == "Keyboard"
      end)

      item.button = key
    end
  elseif binding.bindingType == "state" then
    local bindings = context.states[binding.bindingName]
    if bindings then
      local item = table.find(bindings, function (item)
        return item.device == "Keyboard"
      end)

      item.button = key
    end
  elseif binding.bindingType == "range" then
    local bindings = context.ranges[binding.bindingName]
    if bindings then
      local item = table.find(bindings, function (item)
        return item.device == "Keyboard" and item.value == binding.bindingValue
      end)

      item.button = key
    end
  end

  self:refreshKeyBindings()
  Kernel.inputController:save()
end

function OptionsMenu:keyIdToName(key)
  for keyName, id in pairs(KEYS) do
    if id == key then
      return keyName:sub(5)
    end
  end

  return "Unknown key"
end

function OptionsMenu:refreshKeyBindings()
  local context = Kernel.inputController.gameplayContext

  for i, binding in ipairs(self.keyBindings) do
    if binding.bindingType == "action" then
      local bindings = context.actions[binding.bindingName]
      if bindings then
        local item = table.find(bindings, function (item)
          return item.device == "Keyboard"
        end)

        if item then
          binding.button = self:keyIdToName(item.button)
        end
      end
    elseif binding.bindingType == "state" then
      local bindings = context.states[binding.bindingName]
      if bindings then
        local item = table.find(bindings, function (item)
          return item.device == "Keyboard"
        end)

        if item then
          binding.button = self:keyIdToName(item.button)
        end
      end      
    elseif binding.bindingType == "range" then
      local bindings = context.ranges[binding.bindingName]
      if bindings then
        local item = table.find(bindings, function (item)
          return item.device == "Keyboard" and item.value == binding.bindingValue
        end)

        if item then
          binding.button = self:keyIdToName(item.button)
        end
      end
    end
  end
end

function OptionsMenu:update(deltaTime)
  self.timeout = self.timeout - deltaTime
  if self.timeout < 0.0 then
    self.timeout = 0.0
  end

  self:updateBindings()
end

function OptionsMenu:updateBindings()
  local videoMode = self.videoModes[self.videoModeIndex]

  local bindings = {
    tabIndex = self.tabIndex,
    itemIndex = self.itemIndex,
    itemIndexS = tostring(self.itemIndex),
    screenSize = tostring(videoMode.width) .. "x" .. tostring(videoMode.height),
    fullscreen = self.fullscreen,
    masterVolume = math.floor(self.masterVolume * 10.0),
    musicVolume = math.floor(self.musicVolume * 10.0),
    sfxVolume = math.floor(self.sfxVolume * 10.0),
    ambientVolume = math.floor(self.ambientVolume * 10.0),
    keyBindings = self.keyBindings,
    rebindingIndex = self.rebindingIndex,
    numKeyBindings = #self.keyBindings
  }

  UI.updateBinding("OptionsMenu", bindings)
end

function OptionsMenu:show()
  if not self.visible and Kernel.menuEvents:takeLock("OptionsMenu") then
    UI.attachView("OptionsMenu", "ui/optionsmenu.rml")
    self.visible = true
    Kernel.soundEmitter:postEvent("WindowShow")
    self:updateBindings()
  end
end

function OptionsMenu:hide()
  if not self.visible then
    return
  end

  UI.detachView("OptionsMenu")
  Kernel.menuEvents:releaseLock("OptionsMenu")
  Kernel.menuEvents:takeLock("MainMenu")
  Kernel.soundEmitter:postEvent("WindowHide")

  self.visible = false
end

function OptionsMenu:handleGraphicsTabEvent(event)
  if self.itemIndex == 2 then
    local videoMode = self.videoModes[self.videoModeIndex]

    if event == "menuLeft" then
      if self.videoModeIndex > 1 then
        self.videoModeIndex = self.videoModeIndex - 1
        self:updateBindings()
      else
        self.videoModeIndex = #self.videoModes
        self:updateBindings()
      end
    elseif event == "menuRight" then
      if self.videoModeIndex < #self.videoModes then
        self.videoModeIndex = self.videoModeIndex + 1
        self:updateBindings()
      else
        self.videoModeIndex = 1
        self:updateBindings()
      end
    end
  elseif self.itemIndex == 3 then
    self.fullscreen = not self.fullscreen
    self:updateBindings()
  elseif self.itemIndex == 4 then
    if event == "menuSelect" then
      local videoMode = self.videoModes[self.videoModeIndex]
      Config.setKey("screen", "width", "int", videoMode.width)
      Config.setKey("screen", "height", "int", videoMode.height)
      Config.setKey("screen", "fullscreen", "bool", self.fullscreen)

      Screen.setVideoMode(videoMode.width, videoMode.height, self.fullscreen)
    end
  end
end

function OptionsMenu:handleSoundTabEvent(event)
  if self.itemIndex == 2 then
    if event == "menuLeft" then
      self.masterVolume = self.masterVolume - 0.1
    elseif event == "menuRight" then
      self.masterVolume = self.masterVolume + 0.1
    end

    self.masterVolume = math.clamp(self.masterVolume, 0.0, 1.0)
    Config.setKey("sound", "masterVolume", "float", self.masterVolume)
  elseif self.itemIndex == 3 then
    if event == "menuLeft" then
      self.sfxVolume = self.sfxVolume - 0.1
    elseif event == "menuRight" then
      self.sfxVolume = self.sfxVolume + 0.1
    end

    self.sfxVolume = math.clamp(self.sfxVolume, 0.0, 1.0)
    Config.setKey("sound", "sfxVolume", "float", self.sfxVolume)
  elseif self.itemIndex == 4 then
    if event == "menuLeft" then
      self.musicVolume = self.musicVolume - 0.1
    elseif event == "menuRight" then
      self.musicVolume = self.musicVolume + 0.1
    end

    self.musicVolume = math.clamp(self.musicVolume, 0.0, 1.0)
    Config.setKey("sound", "musicVolume", "float", self.musicVolume)
  elseif self.itemIndex == 5 then
    if event == "menuLeft" then
      self.ambientVolume = self.ambientVolume - 0.1
    elseif event == "menuRight" then
      self.ambientVolume = self.ambientVolume + 0.1
    end

    self.ambientVolume = math.clamp(self.ambientVolume, 0.0, 1.0)
    Config.setKey("sound", "ambientVolume", "float", self.ambientVolume)
  end

  Kernel.soundEmitter:setRTPC("MasterVolume", self.masterVolume * 100.0, true)
  Kernel.soundEmitter:setRTPC("MusicVolume", self.musicVolume * 100.0, true)
  Kernel.soundEmitter:setRTPC("SfxVolume", self.sfxVolume * 100.0, true)
  Kernel.soundEmitter:setRTPC("AmbientVolume", self.ambientVolume * 100.0, true)

  Config.save()
end

function OptionsMenu:handleKeyboardTabEvent(event)
  if self.itemIndex == 1 or event ~= "menuSelect" then
    return
  end

  if self.itemIndex > #self.keyBindings+1 then
    Kernel.inputController:resetDefaults()
    self:refreshKeyBindings()
    self:updateBindings()
  else
    self.rebindingIndex = self.itemIndex - 1
    self.timeout = 0.05
  end
end