local SaveGameStats = require "util/savegamestats"
local SaveGameManager = require "util/savegamemanager"

local ItemDefs = require "data/itemdefs"

local TradingView = Behavior("TradingView")

function TradingView:initialize()
  self:on("destroy", function()
    if self.visible then
      UI.detachView("TradingView")
      Kernel.menuEvents:releaseLock("Trading")
    end
  end)

  self.visible = false

  self.items = {}
  self.selectedIndex = 1
  self.tabIndex = 1

  Kernel.inputController:on("menuBack", function(event)
    if not self.visible then
      return
    end

    self:hide()
    event.handled= true
  end)

  Kernel.inputController:on("menuUp", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.selectedIndex > 1 then
      self.selectedIndex = self.selectedIndex - 1
    end

    self:updateBindings()
    Kernel.soundEmitter:postEvent("MenuClick")
  end)

  Kernel.inputController:on("menuDown", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.tabIndex == 1 then
      if self.selectedIndex < #self.items then
        self.selectedIndex = self.selectedIndex + 1
      end
    else
      if self.selectedIndex < #self.traderItems then
        self.selectedIndex = self.selectedIndex + 1
      end
    end

    Kernel.soundEmitter:postEvent("MenuClick")
    self:updateBindings()
  end)

  Kernel.inputController:on("menuLeft", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.tabIndex == 2 then
      self.tabIndex = self.tabIndex - 1
      self.selectedIndex = 0
      self:refreshItems()
      Kernel.soundEmitter:postEvent("MenuClick")
    end
  end)

  Kernel.inputController:on("menuRight", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.tabIndex == 1 then
      self.tabIndex = self.tabIndex + 1
      self.selectedIndex = 0
      self:refreshItems()
      Kernel.soundEmitter:postEvent("MenuClick")
    end
  end)

  Kernel.inputController:on("menuOption", function(event)
    if not self.visible then
      return
    end

    event.handled = true
    self:updateBindings()
  end)

  Kernel.inputController:on("menuSelect", function(event)
    if not self.visible then
      return
    end

    event.handled = true

    if self.tabIndex == 1 then
      self:sellItem()
    elseif self.tabIndex == 2 then
      self:buyItem()
    end
  end)

  self.leftScroll = 0.0
  self.rightScroll = 0.0
end

function TradingView:awake()
  self.heroRef = BehaviorUtil.findFirstOfType("Hero")
end

function TradingView:update()
  local leftScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 or self.tabIndex == 2 then
    leftScroll = 0.0
  end

  local rightScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 or self.tabIndex == 1 then
    rightScroll = 0.0
  end

  if self.leftScroll ~= leftScroll or self.rightScroll ~= rightScroll then
    self:updateBindings()
  end
end

function TradingView:updateBindings()
  local selectedIndex = 0
  local traderIndex = 0

  if self.tabIndex == 1 then
    selectedIndex = self.selectedIndex
  else
    traderIndex = self.selectedIndex
  end

  local leftScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 or self.tabIndex == 2 then
    leftScroll = 0.0
  end

  local rightScroll = (self.selectedIndex - 3.0) * -64.0
  if self.selectedIndex < 4 or self.tabIndex == 1 then
    rightScroll = 0.0
  end

  self.leftScroll = self.leftScroll + (leftScroll - self.leftScroll) * Time.deltaTime * 8.0
  self.rightScroll = self.rightScroll + (rightScroll - self.rightScroll) * Time.deltaTime * 8.0

  local selectedItem = nil
  if self.tabIndex == 1 then
    selectedItem = self.items[self.selectedIndex]
  else
    selectedItem = self.traderItems[self.selectedIndex]
  end

  local bindings = {
    inputDevice = Kernel.inputController.device,
    buttons = Kernel.inputController:getKeyboardBindings(),
    items = self.items,
    traderItems = self.traderItems,
    selectedIndex = selectedIndex,
    traderIndex = traderIndex,
    selectedItem = selectedItem,
    tabs = self.tabs,
    tabIndex = self.tabIndex,
    coins = self.heroRef.inventory.coins,
    leftScroll = self.leftScroll,
    rightScroll = self.rightScroll
  }

  UI.updateBinding("TradingView", bindings)
end

function TradingView:show(traderInventory)
  self.traderInventory = traderInventory

  if not self.visible and Kernel.menuEvents:takeLock("Trading") then
    UI.attachView("TradingView", "ui/trading.rml")
    self.visible = true

    self.tabIndex = 1
    self.selectedIndex = 0

    self:refreshItems()
    Kernel.soundEmitter:postEvent("WindowShow")
  end
end

function TradingView:hide()
  if not self.visible then
    return
  end

  UI.detachView("TradingView")
  Kernel.menuEvents:releaseLock("Trading")
  Kernel.soundEmitter:postEvent("WindowHide")

  self.visible = false
  self.traderInventory = nil
end

function TradingView:refreshItems()
  if not self.heroRef then
    return
  end

  self.items = {}

  for itemId, qty in pairs(self.heroRef.inventory.items) do
    local def = ItemDefs[itemId]

    if def.type ~= "Rune" then
      self.items[#self.items + 1] = {
        def = def,
        id = itemId,
        qty = qty
      }
    end
  end

  table.sort(self.items, function (a, b)
    return a.def.cost < b.def.cost
  end)

  if self.tabIndex == 1 and #self.items == 0 then
    self.tabIndex = 2
  end

  self.traderItems = {}

  for itemId, qty in pairs(self.traderInventory.items) do
    local def = ItemDefs[itemId]

    self.traderItems[#self.traderItems + 1] = {
      def = def,
      id = itemId,
      qty = qty
    }
  end

  table.sort(self.traderItems, function (a, b)
    return a.def.cost < b.def.cost
  end)

  if self.tabIndex == 1 then
    if self.selectedIndex == 0 then
      self.selectedIndex = 1
    end

    if self.selectedIndex >= #self.items then
      self.selectedIndex = #self.items
    end
  else
    if self.selectedIndex == 0 then
      self.selectedIndex = 1
    end

    if self.selectedIndex >= #self.traderItems then
      self.selectedIndex = #self.traderItems
    end
  end

  self:updateBindings()
end

function TradingView:buyItem()
  if self.selectedIndex == 0 then
    return
  end

  local itemId = self.traderItems[self.selectedIndex].id

  local def = ItemDefs[itemId]
  if not def then
    return
  end

  local cost = def.cost or 0

  local inventory = self.heroRef.inventory

  if inventory.coins < cost then
    return
  end

  if inventory:transferCoins(self.traderInventory, cost) then
    self.traderInventory:transferItems(inventory, itemId, 1)
  end

  SaveGameStats.increase("goldSpent", cost)

  Kernel.soundEmitter:postEvent("ItemBought")
  self:refreshItems()
  
  SaveGameManager.save()
end

function TradingView:sellItem()
  if self.selectedIndex == 0 then
    return
  end

  local itemId = self.items[self.selectedIndex].id

  local def = ItemDefs[itemId]
  if not def then
    return
  end

  local cost = def.cost or 0
  local inventory = self.heroRef.inventory
  local traderInventory = self.traderInventory

  if traderInventory.coins < cost then
    return
  end

  if inventory:transferItems(traderInventory, itemId, 1) == 1 then
    traderInventory:transferCoins(inventory, cost)
  end
  
  SaveGameStats.increase("goldCollected", cost)

  Kernel.soundEmitter:postEvent("ItemSold")
  self:refreshItems()

  SaveGameManager.save()
end
