local waitObjects = {}
local threadToActorMap = {}

local Actor = Behavior("Actor")

function Actor:initialize()
  self.scripts = {}
  self.currentThreads = {}
  self.threadArgs = {}

  self.debug = false
  self.nextId = 1

  self:on("destroy", function()
    for scriptName, thread in pairs(self.currentThreads) do
      waitObjects[thread] = nil
      threadToActorMap[thread] = nil
    end
  end)
end

function Actor:fixedUpdate()
  for scriptName, thread in pairs(self.currentThreads) do
    if waitObjects[thread] ~= nil then
      if waitObjects[thread](Time.fixedDeltaTime) then
        waitObjects[thread] = nil
      end
    end

    if not waitObjects[thread] then
      if coroutine.status(thread) == "dead" then
        self.currentThreads[scriptName] = nil
        self.threadArgs[thread] = nil
        threadToActorMap[thread] = nil
      else
        local status, errorMessage = coroutine.resume(thread, unpack(self.threadArgs[thread]))
        if not status then
          print("[ACTOR] (" .. scriptName .. ") " .. errorMessage)
          self.currentThreads[scriptName] = nil
        end
      end
    end
  end
end

function Actor:hasScript(name)
  return self.scripts[name] ~= nil
end

function Actor:addScript(name, fn)
  self.scripts[name] = fn
end

function Actor:playScript(name, ...)
  if self.scripts[name] == nil then
    return
  end

  local thread = coroutine.create(self.scripts[name])

  self.currentThreads[name] = thread
  self.threadArgs[thread] = {...}

  threadToActorMap[thread] = self

  return thread
end

function Actor:runOnce(fn, ...)
  local name = "once_" .. tostring(self.nextId)
  self.nextId = self.nextId + 1
  local thread = coroutine.create(fn)
  self.currentThreads[name] = thread
  self.threadArgs[thread] = {...}
  threadToActorMap[thread] = self
  return thread
end

function Actor:stopAllScripts()
  for scriptName, thread in pairs(self.currentThreads) do
    waitObjects[thread] = nil
    threadToActorMap[thread] = nil
  end

  self.currentThreads = {}
  self.threadArgs = {}
  if self.debug then
    print("[ACTOR DEBUG] Stopping all scripts")
  end
end

function Actor:customEditor()
  for scriptName, thread in pairs(self.currentThreads) do
    EditorUI.text("Script: " .. scriptName)

    EditorUI.sameLine()

    local status = coroutine.status(thread)

    if status == "dead" then
      EditorUI.textColored("(Finished)", 0x00BB00FF)
    elseif waitObjects[thread] ~= nil then
      EditorUI.textColored("(Waiting on event)", 0x7E7E7EFF)
    else
      EditorUI.textColored("(Running)", 0x00DD00FF)
    end
  end

  self.debug = EditorUI.checkbox("Debug", self.debug)
end

-- wait objects

function WaitForNumberOfFrames(frameCount)
  waitObjects[coroutine.running()] = function()
    frameCount = frameCount - 1
    return frameCount <= 0
  end

  coroutine.yield()
end

function WaitForSeconds(seconds)
  local timer = seconds

  waitObjects[coroutine.running()] = function(deltaTime)
    timer = timer - deltaTime
    return timer <= 0.0
  end

  coroutine.yield()
end

function WaitForAnimation(animationName, trackId)
  trackId = trackId or 0

  local actor = threadToActorMap[coroutine.running()]

  local animationEnded = false
  local listener = nil

  waitObjects[coroutine.running()] = function()
    if not listener then
      listener = function (_trackId, _animationName)
        if _trackId == trackId and _animationName == animationName then
          animationEnded = true
        end
      end

      actor.object:addListener("animationComplete", listener) 
    end

    if animationEnded then
      actor.object:removeListener("animationComplete", listener)
      return true
    end

    return false
  end

  coroutine.yield()
end

function WaitUntil(predicate)
  waitObjects[coroutine.running()] = predicate
  coroutine.yield()
end

function WaitWhile(predicate)
  waitObjects[coroutine.running()] = function()
    return not predicate()
  end

  coroutine.yield()
end

function WaitUntilFadeOut(skeletonRenderer, fadeSpeed)
  local alpha = 1.0
  fadeSpeed = fadeSpeed or 2.0

  WaitWhile(function()
    alpha = alpha - Time.fixedDeltaTime * fadeSpeed
    skeletonRenderer.tintColor = Color(1.0, 1.0, 1.0, math.max(alpha, 0.0))
    return alpha > 0.0
  end)
end

function WaitOne(fn, ...)
  local actor = threadToActorMap[coroutine.running()]
  local thread = actor:runOnce(fn, ...)

  WaitWhile(function()
    return coroutine.status(thread) ~= "dead"
  end)
end
