local CameraFollow = Behavior("CameraFollow")

function CameraFollow:initialize(followTransform)
  self.followTransform = followTransform
  self.cameraPosition = Camera.getPosition()
  self.velocity = Vector.zero()

  self.speed = 128.0
  self.stiffness = 0.02
  self.boxScale = 0.2
  self.damping = 0.975

  self.enabled = true
  self.debug = false
  self.locked = false

  if _EDITOR then
    Input.onKeyDown(KEYS.KEY_F3, function()
      self.locked = not self.locked
    end)
  end
end

function CameraFollow:fixedUpdate()
  if not self.enabled or self.locked or not self.followTransform then
    return
  end

  local cameraSize = Camera.getSize()
  local target = self.followTransform.worldPosition
  local p0, p1 = self:calculateInternalBox()
  if target.x >= p0.x and target.x <= p1.x and target.y >= p0.y and target.y <= p1.y then
    self.velocity = self.velocity * self.damping
  else
    local targetPosition = target - cameraSize * 0.5
    self.velocity = Vector.lerp(self.velocity, (targetPosition - self.cameraPosition):normalized(), self.stiffness)
  end

  self.cameraPosition = self.cameraPosition + self.velocity * Time.fixedDeltaTime * self.speed
  Camera.setPosition(self.cameraPosition)
end

function CameraFollow:snapToPosition()
  local cameraSize = Camera.getSize()
  self.cameraPosition = self.followTransform.worldPosition - cameraSize * 0.5
  Camera.resetPosition(self.cameraPosition)
end

function CameraFollow:update()
  if not self.debug then
    return
  end

  local p0, p1 = self:calculateInternalBox()

  Debug.drawLine(Vector(p0.x, p0.y), Vector(p1.x, p0.y), 0xFF00FFFF)
  Debug.drawLine(Vector(p0.x, p0.y), Vector(p0.x, p1.y), 0xFF00FFFF)
  Debug.drawLine(Vector(p0.x, p1.y), Vector(p1.x, p1.y), 0xFF00FFFF)
  Debug.drawLine(Vector(p1.x, p0.y), Vector(p1.x, p1.y), 0xFF00FFFF)
end

function CameraFollow:calculateInternalBox()
  local cameraPos = Camera.getPosition()
  local size = Camera.getSize()
  local boxSize = size * self.boxScale

  local margin = (size - boxSize) * 0.5

  local p0 = cameraPos + margin
  local p1 = p0 + boxSize

  return p0, p1
end
