local Character = Behavior("Character")

Character.property("faction", "Neutral")
Character.property("hp", 100.0)

function Character:initialize(properties)
  self.properties = Character.defaultProperties(properties)
  self.faction = self.properties.faction or "Neutral"

  self:addComponent("SoundEmitter")
  self:addComponent("Health", self.properties.hp)
  self:addComponent("StatusEffects")
  
  local movable = self:addComponent("Movable", self.properties)

  self:addComponent("SkeletonRenderer")
  self.skeletonRenderer:enableEvents()
  self.skeletonRenderer.zOrder = 100

  self.lookDir = Vector(0.0, -1.0)
  self.currentSkeleton = nil
  self.canAttack = true
  self.isAlert = false

  self.currentAnimation = nil
  self.loopAnimation = nil

  self.target = nil

  self.visibilityFactor = 1.0
  self.canLock = true

  self.onResetTarget = nil

  self.object:addListener("animationComplete", function(trackId, animationName)
    if animationName == self.currentAnimation and self.loopAnimation ~= animationName then
      self.currentAnimation = nil
    end
  end)
end

function Character:update()
  if self.textCo then
    if not self.textCo:update() then
      self.textCo = nil
    end
  end
end

function Character:say(text, immediate)
  if self.textCo then
    Co.dispose(self.textCo)
  end

  local bubble = CreateObject("SpeechBubble", Vector(0.0, 128.0), self.object):addComponent("SpeechBubble")
  bubble:setText(text, immediate)

  self.textCo = Co.create(function ()
    while not bubble.done do
      Co.yield()
    end

    Co.sleep(3.0)

    bubble.object:destroy()
  end)

  self.textCo.disposeFn = function()
    bubble.object:destroy()
  end
end

function Character:lookAt(point)
  local dir = (point - self.transform.worldPosition):normalized()
  self.lookDir = dir
  self:setOrientedSkeleton()
end

function Character:setOrientedSkeleton()
  self.lookDir:normalize()
  self.soundEmitter:setOrientation(self.lookDir)

  local x = self.lookDir.x
  local y = self.lookDir.y

  if self.properties.isTwoSided then
    if x > 0 then
      self:setSkeleton("e")
    else
      self:setSkeleton("w")
    end
  elseif self.properties.isEightSided then
    local eps = 0.5
    if x > eps and math.abs(y) < eps then
      self:setSkeleton("e")
    elseif x < eps and math.abs(y) < eps then
      self:setSkeleton("w")
    elseif math.abs(x) < eps and y > eps then
      self:setSkeleton("n")
    elseif math.abs(x) < eps and y < eps then
      self:setSkeleton("s")
    elseif x > eps and y > eps then
      self:setSkeleton("nw")
    elseif x < eps and y > eps then
      self:setSkeleton("ne")
    elseif x > eps and y < eps then
      self:setSkeleton("sw")
    elseif x < eps and y < eps then
      self:setSkeleton("se")
    end
  else
    if math.abs(x) > math.abs(y) then
      if x > 0 then
        self:setSkeleton("e")
      else
        self:setSkeleton("w")
      end
    else
      if y > 0 then
        self:setSkeleton("n")
      else
        self:setSkeleton("s")
      end
    end
  end
end

function Character:setSkeleton(name)
  if self.currentSkeleton == name then
    return
  end

  if not self.properties.skeleton[name] then
    print("Invalid skeleton name \"" .. name .. "\" for object " .. tostring(self.object:getId()))
    return
  end

  self.currentAnimation = nil

  self.currentSkeleton = name
  self.skeletonRenderer.skeleton = self.properties.skeleton[name]

  for colliderName, object in pairs(self.properties.colliders) do
    self.skeletonRenderer:linkBoundingBoxCollider(colliderName, object)
  end

  if self.loopAnimation then
    self:setAnimation(self.loopAnimation, true)
  end

  if self.object:hasAnyListeners("skeletonChanged") then
    self.object:emit("skeletonChanged", self.currentSkeleton)
  end
end

function Character:setLinkedColliders(colliders)
  self.properties.colliders = colliders
end

function Character:setAnimation(name, loop, force)
  if self.currentAnimation == name and not force then
    return
  end

  loop = loop or false
  self.currentAnimation = name

  if loop then
    self.loopAnimation = name
  else
    self.loopAnimation = nil
  end

  self.skeletonRenderer:setAnimation(0, name, loop)
end

function Character:addAnimation(name, loop)
  loop = loop or false
  self.currentAnimation = name

  if loop then
    self.loopAnimation = name
  else
    self.loopAnimation = nil
  end

  self.skeletonRenderer:addAnimation(0, name, loop)
end

function Character:findEntityInRange(range, predicate)
  local myPosition = self.transform.worldPosition
  local bottomLeft = myPosition - Vector(range, range)
  local topRight = myPosition + Vector(range, range)

  local found = nil
  local closestDistance = 1000000.0

  Physics.aabbQuery(bottomLeft, topRight, function (object)
    if object == self.object then
      return
    end

    if not object.character then
      return true
    end

    if not predicate(object) then
      return true
    end

    local distance = (myPosition - object.transform.worldPosition):len()
    if distance < closestDistance then
      found = object
      closestDistance = distance
    end

    return true
  end, World.layer("MovementCollider"))

  return found
end

function Character:findVisibleEnemyInRange(range)
  local closest = nil
  local closestDistance = 10000.0

  self:forEachEntityInRange(range, function(object)
    if not object.character or object.character.faction == self.faction then
      return true
    end

    if not object.health or not object.health.isAlive then
      return true
    end

    local myPos = self.transform.worldPosition
    local enemyPos = object.transform.worldPosition

    local distance = Vector.distance(myPos, enemyPos)
      
    local actualRange = range * object.character.visibilityFactor
    if distance < actualRange and distance < closestDistance and Physics.lineOfSight(myPos, enemyPos) then
      closestDistance = distance
      closest = object
    end

    return true
  end, World.layer("MovementCollider"))

  return closest
end

function Character:findEnemyInRange(range)
  return self:findEntityInRange(range, function(object)
    if object == self.object then
      return false
    end

    if not object.character or object.character.faction == self.faction
      or object.character.faction == "Neutral" then
      return false
    end

    if not object.health or not object.health.isAlive then
      return false
    end

    local actualRange = range * object.character.visibilityFactor
    return Vector.distance(object.transform.worldPosition, self.transform.worldPosition) < actualRange 
  end)
end

function Character:findAllyInRange(range)
  return self:findEntityInRange(range, function(object)
    return object.character and object.character.faction == self.faction
      and object.health and object.health.isAlive
  end)
end

function Character:forEachEntityInRange(range, predicate, physicsLayer)
  physicsLayer = physicsLayer or World.layer("MovementCollider")

  local myPosition = self.transform.worldPosition
  local bottomLeft = myPosition - Vector(range, range)
  local topRight = myPosition + Vector(range, range)

  Physics.aabbQuery(bottomLeft, topRight, function (object)
    if object == self.object then
      return true
    end

    return predicate(object)
  end, physicsLayer)
end

function Character:warnAllies(range, target)
  target = target or self.target
  if not target then
    return
  end

  self:forEachEntityInRange(range, function (object)
    local character = object.character

    if character and character.faction == self.faction then 
      if not character.target then
        character.target = target
      end
    end

    return true
  end, World.layer("MovementCollider"))
end

function Character:useConsumable(itemId)
  self.object:emit("useConsumable", itemId)
end
