local Inventory = Behavior("Inventory")

function Inventory:initialize()
  self.items = {}
  self.coins = 0
end

function Inventory:addItems(itemId, quantity)
  quantity = quantity or 1
  quantity = math.min(math.floor(quantity), 99)

  if quantity < 0 or not itemId then
    return 0
  end

  local hasAmount = self.items[itemId]

  if not hasAmount then
    self.items[itemId] = quantity
    return 0
  elseif hasAmount + quantity >= 99 then
    self.items[itemId] = 99
    return hasAmount + quantity - 99
  else
    self.items[itemId] = hasAmount + quantity
    return 0
  end
end

function Inventory:removeItems(itemId, quantity)
  quantity = quantity or 1
  quantity = math.floor(quantity)

  if quantity < 0 or not itemId then
    return 0
  end

  local hasAmount = self.items[itemId]
  if not hasAmount then
    return 0
  end

  local removeAmount = math.min(hasAmount, quantity)
  self.items[itemId] = hasAmount - removeAmount
  if self.items[itemId] == 0 then
    self.items[itemId] = nil
  end

  return removeAmount
end

function Inventory:containsItems(itemId, quantity)
  quantity = quantity or 1
  quantity = math.floor(quantity)

  local hasAmount = self.items[itemId]
  if not hasAmount then
    return false
  end

  return hasAmount >= quantity
end

function Inventory:containsItem(itemId)
  return self.items[itemId] ~= nil
end

function Inventory:transferItems(otherInventory, itemId, quantity)
  quantity = quantity or 1
  quantity = math.floor(quantity)

  if quantity < 0 or not itemId then
    return 0
  end

  local hasAmount = self.items[itemId]
  if not hasAmount then
    return 0
  end

  local amountToTransfer = math.min(hasAmount, quantity)
  self.items[itemId] = self.items[itemId] - amountToTransfer

  local leftOverAmount = otherInventory:addItems(itemId, amountToTransfer)
  self:addItems(itemId, leftOverAmount)

  if self.items[itemId] == 0 then
    self.items[itemId] = nil
  end

  return amountToTransfer - leftOverAmount
end

function Inventory:transferCoins(otherInventory, amount)
  if amount <= 0 then
    return 0
  end

  amount = math.floor(amount)

  local amountToTransfer = math.min(self.coins, amount)
  self.coins = self.coins - amountToTransfer
  otherInventory.coins = otherInventory.coins + amountToTransfer
  return amountToTransfer
end

function Inventory:isEmpty()
  if self.coins > 0 then
    return false
  end

  for itemId, quantity in pairs(self.items) do
    if quantity > 0 then
      return false
    end
  end

  return true
end
