local Movable = Behavior("Movable")

Movable.property("colliderDisplacement", Vector(0.0, 0.0))
Movable.property("collisionRadius", 28.0)
Movable.property("colliderDensity", 128.0)
Movable.property("collisionAvoidance", true)
Movable.property("movementSpeed", 64.0)
Movable.property("flocking", true)

function Movable:initialize(properties)
  self.properties = Movable.defaultProperties(properties)

  self.velocity = Vector.zero()
  self.movementSpeed = self.properties.movementSpeed
  self.radius = self.properties.collisionRadius

  local movementCollider = self:addComponent("PhysicsCollider")
  movementCollider:addCircleShape(
    self.properties.colliderDisplacement, 
    self.radius, 
    { density = self.properties.colliderDensity }
  )

  movementCollider.categories = World.layer("MovementCollider")
  movementCollider.mask = flags(World.layer("World"), World.layer("MovementCollider"))

  self.blocked = false
  self.blockedCheckDistance = 8.0
  self.blockedLastPosition = Vector.zero()

  self.rayRotation = 0.45
  self.collisionAvoidance = self.properties.collisionAvoidance
  self.collisionAvoidanceStrength = 3.0

  self.flocking = self.properties.flocking
  self.flockingRange = 96.0

  self.debug = false

  self.movementColliderLayer = World.layer("MovementCollider")
end

function Movable:fixedUpdate()
  local displacement = self.velocity * self.movementSpeed * Time.fixedDeltaTime
  local speed = displacement:len()
  if speed == 0.0 then
    return
  end

  displacement = displacement / speed

  local myPosition = self.transform.worldPosition

  if self.collisionAvoidance then
    local sample = Pathfinding.sampleCollisionAvoidanceBilinear(myPosition)
    local magnitude = sample:len()

    if magnitude > 0.0 then
      sample = sample / magnitude
    end

    displacement = Vector.lerp(displacement, sample, math.clamp(magnitude * self.collisionAvoidanceStrength, 0.0, 1.0)):normalized()
    displacement, speed = self:doDynamicCollisionAvoidance(displacement, speed)

    if self.flocking then
      displacement, speed = self:doFlockingBehaviors(displacement, speed)
    end
  end

  if self.debug then
    Debug.drawLine(myPosition, myPosition + displacement * self.radius * 3.0, 0x0000FFFF)
  end

  local position = myPosition + displacement * speed

  self.blockedCheckDistance = self.blockedCheckDistance - speed
  if self.blockedCheckDistance < 0.0 then
    self.blockedCheckDistance = 24.0

    local dist = Vector.distance(self.blockedLastPosition, myPosition)
    if dist < 4.0 then
      self.blocked = true
    else
      self.blocked = false
    end

    self.blockedLastPosition = position
  end

  self.transform.localPosition = position
end

function Movable:predictPosition(time)
  local displacement = self.velocity * self.movementSpeed * time
  return self.transform.worldPosition + displacement
end

function Movable:doDynamicCollisionAvoidance(displacement, speed)
  local position = self.transform.worldPosition

  local rayLength = self.radius * 2.0
  local rayDirA = displacement:rotated(self.rayRotation):normalized() * rayLength
  local rayDirB = displacement:rotated(-self.rayRotation):normalized() * rayLength
  
  if self.debug then
    Debug.drawLine(position, position + rayDirA, 0xFF00FFFF)
    Debug.drawLine(position, position + rayDirB, 0xFF00FFFF)
  end
  
  local hitA = nil
  Physics.raycast(position, position + rayDirA, function(hitInfo)
    if hitInfo.object ~= self then
      if not hitA then
        hitA = hitInfo
      elseif Vector.distance(position, hitInfo.point) < Vector.distance(position, hitA.point) then
        hitA = hitInfo
      end
    end

    return 1.0
  end, self.movementColliderLayer)

  local hitB = nil
  Physics.raycast(position, position + rayDirB, function(hitInfo)
    if hitInfo.object ~= self then
      if not hitB then
        hitB = hitInfo
      elseif Vector.distance(position, hitInfo.point) < Vector.distance(position, hitB.point) then
        hitB = hitInfo
      end
    end

    return 1.0
  end, self.movementColliderLayer)

  if hitA ~= nil then
    if self.debug then
      Debug.drawCircle(hitA.point, 6.0, 0x00FF00FF)
    end

    local dir = (position - hitA.point)
    local dist = dir:len()
    dir = dir / dist
    displacement = Vector.lerp(dir, displacement, dist / rayLength)
  end

  displacement = displacement:normalized()

  if hitB ~= nil then
    if self.debug then
      Debug.drawCircle(hitB.point, 6.0, 0x00FF00FF)
    end
    
    local dir = (position - hitB.point)
    local dist = dir:len()
    dir = dir / dist
    displacement = Vector.lerp(dir, displacement, dist / rayLength)
  end

  displacement = displacement:normalized()

  return displacement, speed
end

function Movable:doFlockingBehaviors(displacement, speed)
  if not self.character then
    return
  end

  local faction = self.character.faction

  local myPosition = self.transform.worldPosition
  local bottomLeft = myPosition - Vector(self.flockingRange, self.flockingRange)
  local topRight = myPosition + Vector(self.flockingRange, self.flockingRange)

  local separation = Vector.zero()
  local alignment = Vector.zero()

  local count = 0.0

  Physics.aabbQuery(bottomLeft, topRight, function (object)
    if object == self.object or not object.character or object.character.faction ~= faction then
      return true
    end

    local otherPosition = object.transform.worldPosition
    local dir = myPosition - otherPosition
    local dist = dir:len()
    dir = dir / dist

    if self.debug then
      Debug.drawLine(myPosition, otherPosition, 0x00FF00FF)
    end

    local separationForce = dist / self.flockingRange
    separation = Vector.lerp(separation, dir, separationForce)
    alignment = alignment + object.movable.velocity

    count = count + 1.0

    return true
  end, self.movementColliderLayer)

  if count > 0.0 then
    alignment = alignment / count

    if self.debug then
      Debug.drawLine(myPosition, myPosition + separation * 32.0, 0xFF00FFFF)
    end

    displacement = Vector.lerp(displacement, alignment, 0.15)
    displacement = Vector.lerp(displacement, separation, 0.25)
    displacement:normalize()
  end

  return displacement, speed
end
