local SpeechBubble = Behavior("SpeechBubble")

function SpeechBubble:initialize()
  self.text = "Hello world"

  self.maxCharsPerLine = 22
  self.textScale = 1.25
  self.pixel = AssetUtil.find("no_atlas/pixel.png")

  self:setText(self.text)

  self.lineIndex = 1
  self.charIndex = 1
  self.textDelay = 0.02

  self.done = false
end

function SpeechBubble:update()
  if not self.text or not self.poly then
    return
  end

  if self.showTextCo then
    if not self.showTextCo:update() then
      self.showTextCo = nil
    end
  end

  local pos = self.transform.worldPosition
  self.poly.position = pos
  Graphics.drawMesh(self.pixel, self.poly)

  self.arrowPoly.position = pos + Vector(0.0, -40.0)
  Graphics.drawMesh(self.pixel, self.arrowPoly)

  local y = self.height + pos.y
  local x  = pos.x

  for i, line in ipairs(self.lines) do
    if i > self.lineIndex then
      break
    end

    local size = self.lineSizes[i]
    local margin = (self.largestWidth - size.x) * 0.5

    local text = line
    if i == self.lineIndex then
      text = line:sub(1, self.charIndex)
    end

    Graphics.drawText(text, {
      position = Vector(x + margin, y - size.y),
      color = Color.black,
      scale = self.textScale,
      zOrder = 905,
      isUI = true
    })

    y = y - size.y
  end
end

function SpeechBubble:setText(text, immediate)
  self.text = text
  self.lines = self:splitText(text)

  self:buildTextPoly(self.lines)
  self.poly.zOrder = 900
  self.poly.outline = true
  self.poly.isUI = true
  self.poly.tint = Color(1.0, 0.0, 0.0, 1.0)

  self:buildArrowPoly()
  self.arrowPoly.zOrder = 895
  self.arrowPoly.outline = true
  self.arrowPoly.isUI = true

  self.lineIndex = 1
  self.charIndex = 1

  if immediate then
    self.lineIndex = #self.lines
    self.charIndex = #(self.lines[#self.lines])
  end

  self.done = false

  if not immediate then
    self.showTextCo = Co.create(function()
      while self.lineIndex <= #self.lines do
        while self.charIndex <= #self.lines[self.lineIndex] do
          self.charIndex = self.charIndex + 1
          Co.sleep(self.textDelay)
        end

        self.lineIndex = self.lineIndex + 1
        self.charIndex = 1
      end

      self.done = true
    end)
  end
end

function SpeechBubble:splitText(text)
  local length = #text
  local numLines = math.floor(length / self.maxCharsPerLine)
  if numLines == 1 then
    return { text }
  end

  local lines = {}
  local mid = length / 2

  if numLines == 2 then
    local current = nil
    for word in text:gmatch("%S+") do
      if not current then
        current = word
      else
        current = current .. " " .. word
      end

      if #current >= mid then
        lines[#lines+1] = current
        current = nil
      end
    end

    if current ~= nil then
      lines[#lines+1] = current
    end

    return lines
  end

  if numLines == 3 then
    local i = 0
    local c = 0
    local current = nil
    for word in text:gmatch("%S+") do
      if current == nil then
        current = word
        c = c + #word
      else
        current = current .. " " .. word
        c = c + #word + 1
      end

      if i == 0 and c >= mid - self.maxCharsPerLine * 0.65 then
        lines[#lines+1] = current
        current = nil
        i = i + 1
      elseif i == 1 and c >= mid + self.maxCharsPerLine * 0.65 then
        lines[#lines+1] = current
        current = nil
        i = i + 1
      end
    end

    if current ~= nil then
      lines[#lines+1] = current
    end

    return lines
  end

  local current = nil
  for word in text:gmatch("%S+") do
    if current == nil then
      current = word
    else
      current = current .. " " .. word
    end

    if #current >= self.maxCharsPerLine then
      lines[#lines+1] = current
      current = nil
    end
  end

  if current ~= nil then
    lines[#lines+1] = current
  end

  return lines
end

function SpeechBubble:buildArrowPoly()
  local uv = Vector(0.5, 0.5)

  local poly = {
    vertexCount = 3,
    vertices = {
      [1] = {
        position = Vector(0.0, 0.0),
        uv = uv,
        color = Color.white
      },
      [2] = {
        position = Vector(24.0, 72.0),
        uv = uv,
        color = Color.white
      },
      [3] = {
        position = Vector(64.0, 32.0),
        uv = uv,
        color = Color.white
      }
    },
    triangleCount = 1,
    indices = { 0, 1, 2 }
  }

  self.arrowPoly = poly
end

function SpeechBubble:buildTextPoly(lns)
  local sizes = {}
  local largestWidth = 0.0
  local totalHeight = 0.0

  for i, line in ipairs(lns) do
    sizes[i] = Graphics.measureText(line, nil, self.textScale)

    if sizes[i].x > largestWidth then
      largestWidth = sizes[i].x
    end

    totalHeight = totalHeight + sizes[i].y
  end

  local points = {}

  local y = totalHeight
  for i, line in ipairs(sizes) do
    local margin = (largestWidth - line.x) * 0.5
    points[#points+1] = Vector(margin, y - line.y)
    points[#points+1] = Vector(margin + line.x, y - line.y)
    points[#points+1] = Vector(margin + line.x, y)
    points[#points+1] = Vector(margin, y)
    y = y - line.y
  end

  local convexHull = {}
  _L._convexHull({ vertexCount = #points, vertices = points }, convexHull)

  local center = Vector(convexHull.vertices[1])
  for i=2,convexHull.vertexCount do
    convexHull.vertices[i] = Vector(convexHull.vertices[i])
    center = center + convexHull.vertices[i]
  end

  center = center / convexHull.vertexCount

  local padding = Vector(16.0, 64.0)

  local whRatio = largestWidth / totalHeight
  local upscale = Vector(0.18, 0.24) * whRatio

  if #self.lines == 1 then
    padding = Vector(72.0, 64.0)
    upscale.x = upscale.x * 0.35
  elseif #self.lines == 2 then
    upscale = Vector(0.20, 0.24) * whRatio
  end

  if #self.lines == 1 then
  end

  for i=1,convexHull.vertexCount do
    local v = convexHull.vertices[i]
    local dir = v - center
    local length = dir:len()
    dir = dir / length

    convexHull.vertices[i] = center + dir * upscale * length + padding * dir
  end

  for i=1,4 do
    convexHull = self:subdividePoly(convexHull)
  end

  local poly = {
    vertexCount = convexHull.vertexCount + 1,
    triangleCount = convexHull.vertexCount,
    vertices = {},
    indices = {}
  }

  poly.vertices[1] = {
    position = center,
    uv = Vector.zero(),
    color = Color(1.0, 1.0, 1.0, 1.0)
  }

  local uv = Vector(0.5, 0.5)
  for i=1,convexHull.vertexCount+1 do
    poly.vertices[i+1] = {
      position = convexHull.vertices[i],
      uv = uv,
      color = Color(1.0, 1.0, 1.0, 1.0)
    }
  end

  for i=1,convexHull.vertexCount do
    poly.indices[#poly.indices+1] = 0
    poly.indices[#poly.indices+1] = i

    if i + 1 > convexHull.vertexCount then
      poly.indices[#poly.indices+1] = 1
    else
      poly.indices[#poly.indices+1] = i + 1
    end
  end

  self.poly = poly
  self.lineSizes = sizes
  self.height = totalHeight
  self.largestWidth = largestWidth
end

function SpeechBubble:subdividePoly(poly)
  local vertexCount = poly.vertexCount

  local subdivided = {
    vertexCount = vertexCount * 2,
    vertices = {}
  }

  for i=1,vertexCount do
    subdivided.vertices[#subdivided.vertices+1] = poly.vertices[i]

    local nxt = nil
    if i == vertexCount then
      nxt = poly.vertices[1]
    else
      nxt = poly.vertices[i+1]
    end

    subdivided.vertices[#subdivided.vertices+1] = (poly.vertices[i] + nxt) * 0.5
  end

  for i=1,vertexCount*2,2 do
    local right = subdivided.vertices[i+1]
    local left = nil

    if i == 1 then
      left = subdivided.vertices[#subdivided.vertices]
    else
      left = subdivided.vertices[i-1]
    end

    local mean = (left + right) * 0.5
    subdivided.vertices[i] = (subdivided.vertices[i] + mean) * 0.5
  end

  return subdivided
end
