local StatusEffectTypes = {
  Healing = require "util/statuseffects/healing",
  BoostStaminaRegen = require "util/statuseffects/booststaminaregen",
  SecondChance = require "util/statuseffects/secondchance",
  Berserk = require "util/statuseffects/berserk",
  Poison = require "util/statuseffects/poison",
  Knockback = require "util/statuseffects/knockback",
  Slow = require "util/statuseffects/slow",
  Invisibility = require "util/statuseffects/invisibility",
  Frozen = require "util/statuseffects/frozen"
}

local StatusEffects = Behavior("StatusEffects")

function StatusEffects:initialize()
  self.stacks = {}
end

function StatusEffects:clearAll()
  for name, statusEffect in pairs(self.stacks) do
    if statusEffect.dispose then
      statusEffect:dispose()
    else
      self.stacks[name] = nil
    end
  end
end

function StatusEffects:update(deltaTime)
  for name, statusEffect in pairs(self.stacks) do
    if statusEffect.run then
      if not statusEffect:run() then
        self.stacks[name] = nil
      end
    end
  end
end

function StatusEffects:applyByName(effectName, properties)
  local statusType = StatusEffectTypes[effectName]
  if not statusType then
    print("ERROR! StatusEffects:applyByName no status effect named \"" .. effectName .. "\"")
    return
  end

  self:apply(statusType(self.object, properties))
end

function StatusEffects:apply(statusEffect)
  local name = statusEffect.name

  local current = self.stacks[name]
  if not current then
    self.stacks[name] = statusEffect
  else
    if not current.ended and current.stackWith then
      current:stackWith(statusEffect)
    else
      if current.dispose then
        current:dispose()
      end

      self.stacks[name] = statusEffect
    end
  end

  return statusEffect
end

function StatusEffects:hasStatusEffect(name)
  local status = self.stacks[name]
  return status and not status.ended
end

function StatusEffects:serialize()
  local serialized = {}

  for name, statusEffect in pairs(self.stacks) do
    if not statusEffect.ended and statusEffect.serialize then
      serialized[name] = statusEffect:serialize()
    end
  end

  return serialized
end

function StatusEffects:deserialize(serialized)
  for name, data in pairs(serialized) do
    local class = StatusEffectTypes[name]
    if not class then
      print("ERROR! Failed to deserialize status effect type \"" .. name .. "\"")
    else
      self:apply(class(self.object, serialized[name]))
    end
  end
end
