local AI = require "util/aicommon"
local LootSpawner = require "util/lootspawner"
local Runes = require "util/runes"

local Barrel = Behavior("Barrel")

Barrel.editorIcon("barrel/barrel.json")
Barrel.property("indestructible", false)
Barrel.property("lootTable", "Barrel")
Barrel.property("breakChance", 0.3)

function Barrel:initialize(properties)
  self.properties = Barrel.defaultProperties(properties)

  local soundEmitter = self:addComponent("SoundEmitter")

  local renderer = self:addComponent("SkeletonRenderer")
  renderer.zOrder = 100
  renderer.skeleton = "barrel/barrel.json"
  renderer:enableEvents()

  if self.properties.indestructible then
    renderer:setAnimation(0, "idle_indestructible", true)
  else
    renderer:setAnimation(0, "idle", true)
  end

  local collider = self:addComponent("PhysicsCollider")
  collider:addCircleShape(Vector(0.0, 8.0), 26.0)
  collider.type = collider.Types.Static
  collider.categories = flags(World.layer("MovementCollider"), World.layer("Hitbox"))
  collider.mask = flags(World.layer("Weapon"), World.layer("MovementCollider"))

  self.breakChance = 0.0

  self:on("collisionStart", function(collisionInfo)
    if not collisionInfo.object.weapon or collider.mask == 0 then
      return
    end

    local object = collisionInfo.object

    local parent = nil
    if object.transform.parent then
      parent = object.transform.parent.object
    else
      parent = object
    end

    self.breakChance = self.breakChance + self.properties.breakChance
    local breakChance = self.breakChance

    if parent.hero and Runes.isEquipped("BarrelCrasher") then
      breakChance = 1.0
    end

    if self.properties.indestructible or breakChance <= Random.value() then
      local dir = (self.transform.worldPosition - parent.transform.worldPosition):normalized()

      if self.properties.indestructible then
        renderer:setAnimation(0, "hit_indestructible", false)
      else
        if math.abs(dir.x) > math.abs(dir.y) then
          if dir.x > 0.0 then
            renderer:setAnimation(0, "hit_left", false)
          else
            renderer:setAnimation(0, "hit_right", false)
          end
        else
          if dir.y < 0.0 then
            renderer:setAnimation(0, "hit_top", false)
          else
            renderer:setAnimation(0, "hit_bottom", false)
          end
        end
      end

      soundEmitter:postEvent("BarrelGetHit")
    else
      if self.properties.lootTable ~= "" then
        LootSpawner.spawnPouch(self.transform.worldPosition, self.properties.lootTable)
      end

      collider.mask = 0
      self:removeComponent("PhysicsCollider")
      soundEmitter:postEvent("BarrelBreak")
      
      self.mainCo = Co.create(function()
        renderer:setAnimation(0, "break", false)
        AI.yieldWaitForAnimation(self.object, "break")
        self.object:destroy()
      end)
    end
  end)
end

function Barrel:update()
  if not self.mainCo then
    return
  end

  if not self.mainCo:update() then
    self.mainCo = nil
    return
  end
end
