local SaveGameManager = require "util/savegamemanager"

local Portal = Behavior("Portal")

Portal.property("name", "A")
Portal.property("target", "B")
Portal.property("targetWorld", "")
Portal.property("spawnDirection", "down")
Portal.property("disableIfEnemiesInRange", false)
Portal.property("enemyRange", 600.0)
Portal.property("skin", "blank")
Portal.property("wind", false)

Portal.editorIcon("portal/portal.json")

function Portal:initialize(properties)
  self.properties = properties
  self.waitForExit = {}

  local skeleton = self:addComponent("SkeletonRenderer")
  skeleton.zOrder = 100
  skeleton.skeleton = "portal/portal.json"
  skeleton.timeScale = 0.5
  skeleton.skin = self.properties.skin

  local sound = self:addComponent("SoundEmitter")

  if self.properties.target ~= "" or self.properties.targetWorld ~= "" then
    sound:postEvent("PortalHum")

    if self.properties.wind then
      skeleton:setAnimation(0, "idle_on_wind", true)
    else
      skeleton:setAnimation(0, "idle_on", true)
    end

    self.isActive = true
  else
    if self.properties.wind then
      skeleton:setAnimation(0, "idle_off_wind", true)
    else
      skeleton:setAnimation(0, "idle_off", true)
    end
    self.isActive = false
  end

  local collider = self:addComponent("PhysicsCollider")
  collider:addRectangleShape(Vector(-48.0, 16.0), Vector(24.0, 48.0))
  collider:addRectangleShape(Vector(48.0, 16.0), Vector(24.0, 48.0))
  collider.mask = 255
  collider.categories = World.layer("World")
  collider.type = collider.Types.Static

  local tpCollider = CreateObject("TeleportCollider", Vector.zero(), self.object):addComponent("PhysicsCollider")
  tpCollider:addRectangleShape(Vector(0.0, 16.0), Vector(64.0, 16.0), { isSensor = true })
  tpCollider.categories = World.layer("MovementCollider")
  tpCollider.mask = World.layer("MovementCollider")
  tpCollider.type = collider.Types.Static

  tpCollider.object:on("collisionStart", function(collisionInfo)
    self:handleCollision(collisionInfo)
  end)

  tpCollider.object:on("collisionEnd", function(collisionInfo)
    local hero = collisionInfo.object.hero
    if hero then
      self.waitForExit[hero] = nil
    end

    local movable = collisionInfo.object.movable
    if movable then
      self.waitForExit[movable] = nil
    end
  end)

  if self.properties.disableIfEnemiesInRange then
    self.checkForEnemiesCo = Co.create(function ()
      while true do
        local enemies = self:checkForEnemiesInRange(self.properties.enemyRange)
        if enemies and self.isActive then
          if self.properties.wind then
            skeleton:setAnimation(0, "idle_off_wind", true)
          else
            skeleton:setAnimation(0, "idle_off", true)
          end

          sound:stopAllSounds()
          self.isActive = false
        elseif not enemies and not self.isActive then
          if self.properties.wind then
            skeleton:setAnimation(0, "idle_on_wind", true)
          else
            skeleton:setAnimation(0, "idle_on", true)
          end
          
          sound:postEvent("PortalHum")
          self.isActive = true
        end

        Co.sleep(1.0)
      end
    end)
  end
end

function Portal:setWind(wind)
  self.properties.wind = wind

  if self.properties.target ~= "" or self.properties.targetWorld ~= "" then
    if self.properties.wind then
      self.skeletonRenderer:setAnimation(0, "idle_on_wind", true)
    else
      self.skeletonRenderer:setAnimation(0, "idle_on", true)
    end
  else
    if self.properties.wind then
      self.skeletonRenderer:setAnimation(0, "idle_off_wind", true)
    else
      self.skeletonRenderer:setAnimation(0, "idle_off", true)
    end
  end

  self.skeletonRenderer:fastForward(Random.value() * 8.0)
end

function Portal:update(deltaTime)
  if self.checkForEnemiesCo then
    if not self.checkForEnemiesCo:update() then
      self.checkForEnemiesCo = nil
    end
  end

  if self.teleportHeroCo then
    if not self.teleportHeroCo:update() then
      self.teleportHeroCo = nil
    end
  end
end

function Portal:handleCollision(collisionInfo)
  if self.properties.disableIfEnemiesInRange then
    if self:checkForEnemiesInRange(self.properties.enemyRange) then
      return
    end
  end

  local hero = collisionInfo.object.hero

  if self.properties.targetWorld ~= "" then
    if hero then
      self:teleportHeroToWorld(collisionInfo.object.hero, self.properties.targetWorld)
    end

    return
  end

  local portal = self:findTargetPortal(self.properties.target)
  if not portal then
    return
  end

  local movable = collisionInfo.object.movable

  if hero then
    if not self.waitForExit[hero] then
      self:teleportHero(hero, portal)
    end
  elseif movable then
    if not self.waitForExit[movable] then
      self:teleportMovable(movable, portal)
    end
  end
end

function Portal:findTargetPortal(name)
  local portals = BehaviorUtil.findAllOfType("Portal")
  for i, portal in ipairs(portals) do
    if portal.properties.name == name then
      return portal
    end
  end

  return nil
end

function Portal:teleportMovable(movable, portal)
  portal.waitForExit[movable] = true

  self.soundEmitter:postEvent("PortalOnTeleport")
  movable.transform.localPosition = portal.transform.worldPosition
end

function Portal:teleportHero(hero, portal)
  portal.waitForExit[hero] = true
  Kernel.menuEvents:takeLock("Teleporting")

  self.soundEmitter:postEvent("PortalOnTeleport")

  self.teleportHeroCo = Co.create(function()
    local renderer = hero.skeletonRenderer
    local alpha = 1.0

    while alpha > 0.0 do
      alpha = math.max(0.0, alpha - Time.fixedDeltaTime * 3.0)
      renderer.tintColor = Color(1.0, 1.0, 1.0, alpha)
      Co.yield()
    end

    Kernel.screenFade:fadeOut(0.25)
    Co.sleep(0.35)

    local displacement = Vector(0.0, 0.0)
    if portal.properties.spawnDirection == "down" then
      displacement = Vector(0.0, -24.0)
      hero.character.lookDir = Vector(0.0, -1.0)
    elseif portal.properties.spawnDirection == "up" then
      displacement = Vector(0.0, 24.0)
      hero.character.lookDir = Vector(0.0, 1.0)
    else
      hero.character.lookDir = Vector(0.0, 0.0)
    end

    hero.moveDir = Vector.zero()
    hero.character:setOrientedSkeleton()
    hero.transform.localPosition = portal.transform.worldPosition + displacement

    hero.cameraFollower:snapToPosition()

    Kernel.screenFade:fadeIn(0.25)
    Co.sleep(0.35)

    while alpha < 1.0 do
      alpha = math.min(1.0, alpha + Time.fixedDeltaTime * 3.0)
      renderer.tintColor = Color(1.0, 1.0, 1.0, alpha)
      Co.yield()
    end

    Kernel.menuEvents:releaseLock("Teleporting")
  end)
end

function Portal:teleportHeroToWorld(hero, world)
  self.soundEmitter:postEvent("PortalOnTeleport")
  Kernel.menuEvents:takeLock("Teleporting")

  self.teleportHeroCo = Co.create(function()
    local renderer = hero.skeletonRenderer
    local alpha = 1.0

    while alpha > 0.0 do
      alpha = math.max(0.0, alpha - Time.fixedDeltaTime * 3.0)
      renderer.tintColor = Color(1.0, 1.0, 1.0, alpha)
      Co.yield()
    end

    Kernel.screenFade:fadeOut(0.25)
    Co.sleep(0.35)

    SaveGameManager.save()
    Kernel.menuEvents:releaseLock("Teleporting")
    World.setCurrent(world)
  end)
end

function Portal:checkForEnemiesInRange(range)
  local bl = self.transform.worldPosition - Vector(range, range)
  local tr = self.transform.worldPosition + Vector(range, range)
  local foundEnemy = false
  local faction = nil

  Physics.aabbQuery(bl, tr, function (object)
    if not object.character then
      return true
    end

    if not faction then
      faction = object.character.faction
      return true
    end

    if object.character.faction ~= faction then
      foundEnemy = true
      return false
    end

    return true
  end, World.layer("MovementCollider"))

  return foundEnemy
end
