local AI = require "util/aicommon"
local SawTrap = Behavior("SawTrap")

SawTrap.editorIcon("saw/saw.json")
SawTrap.property("direction", Vector(1.0, 0.0))
SawTrap.property("distance", 200.0)
SawTrap.property("speed", 1.0)
SawTrap.property("damage", 60.0)
SawTrap.property("selfDestruct", false)

function SawTrap:initialize(properties)
  self.properties = SawTrap.defaultProperties(properties)
  self.properties.direction:normalize()

  local soundEmitter = self:addComponent("SoundEmitter")

  local renderer = self:addComponent("SkeletonRenderer")
  renderer.zOrder = 100
  renderer.skeleton = "saw/saw.json"
  renderer:enableEvents()
  renderer:setAnimation(0, "idle", true)

  local particleEmitter = self:addComponent("ParticleEmitter")
  particleEmitter.image = "default/particle.png"

  self.particleDef = {
    lifetimeMin = 0.5,
    lifetimeMax = 0.65,
    sizeMin = 8.0,
    sizeMax = 12.0,
    growth = 0.0,
    velocityMin = Vector(0.0, 32.0),
    velocityMax = Vector(32.0, 32.0),
    acceleration = Vector(0.0, -96.0),
    colorStart = Color(0.96, 0.96, 0.96, 1.0),
    colorEnd = Color(0.74, 0.74, 0.74, 0.0),
    blendMode = "alphaBlend"
  }

  particleEmitter.particleDef = self.particleDef

  local weapon = self:addComponent("Weapon")
  weapon:startAttack(25.0)

  weapon.onApplyEffects = function (object)
    local statusEffects = object.statusEffects
    if not statusEffects then
      return
    end

    local dir = (object.transform.worldPosition - self.transform.worldPosition):normalized()

    statusEffects:applyByName("Knockback", { impulse = dir * 3600.0 })
  end

  local collider = self.physicsCollider
  collider:addRectangleShape(Vector(0.0, -8.0), Vector(72.0, 24.0), { isSensor = true })
  collider.categories = World.layer("Weapon"), World.layer("MovementCollider")
  collider.mask = World.layer("Hitbox")

  self.p0 = self.transform.worldPosition
  self.p1 = self.transform.worldPosition + self.properties.direction * self.properties.distance

  local turnOn = Co.create(function()
    renderer:setAnimation(0, "on")

    local dir = self.properties.direction
    renderer:addAnimation(0, "active", true)

    AI.yieldWaitForAnimation(self.object, "on")
    renderer.timeScale = 3.0
    weapon:activate()
    weapon:startAttack(self.properties.damage)
  end)

  local turnOff = Co.create(function()
    weapon:deactivate()
    renderer.timeScale = 1.0
    renderer:setAnimation(0, "off")
    renderer:addAnimation(0, "idle", true)
    AI.yieldWaitForAnimation(self.object, "off")
  end)

  self.mainCo = Co.create(function()
    local speed = self.properties.speed
    local t = 0.0

    while true do
      Co.yield(turnOn())

      self.particleDef.velocityMax = Vector(64.0, 64.0)
      self.particleEmitter.particleDef = self.particleDef

      while t < 1.0 do
        t = t + Time.fixedDeltaTime * speed
        self.transform.localPosition = Vector.lerp(self.p0, self.p1, t)
        particleEmitter:spawnParticles(Vector(42.0, -34.0), 1)
        Co.yield()
      end

      Co.yield(turnOff())
      Co.sleep(1.0)
      Co.yield(turnOn())

      self.particleDef.velocityMax = Vector(-64.0, 64.0)
      self.particleEmitter.particleDef = self.particleDef

      while t > 0.0 do
        t = t - Time.fixedDeltaTime * speed
        self.transform.localPosition = Vector.lerp(self.p0, self.p1, t)
        particleEmitter:spawnParticles(Vector(-42.0, -34.0), 1)
        Co.yield()
      end

      Co.yield(turnOff())
      Co.sleep(1.0)

      if self.properties.selfDestruct then
        self.object:destroy()
      end
    end
  end)
end

function SawTrap:update()
  if not self.mainCo then
    return
  end

  if not self.mainCo:update() then
    self.mainCo = nil
    return
  end
end
