local AI = require "util/aicommon"

local SaveGameManager = require "util/savegamemanager"
local HubWorld = Behavior("HubWorld")

function HubWorld:initialize()
  Graphics.setColorLUT(AssetUtil.find("no_atlas/default_lut.png"), false)
  Graphics.setColorLUTAlpha(0.0)

  Graphics.useFogOfWar(false)
  self.missionAreaIndex = 1
  self.pixelTexture = AssetUtil.find("no_atlas/pixel.png")
  self.letterboxAlpha = 0.0
  self.playTutorialCutscene = true
  self.tutorialCutscenePlayed = false

  SaveGameManager:on("saveGame", function(saveGame)
    saveGame:setKey("hub", { tutorialCutscene = self.tutorialCutscenePlayed })
  end)

  SaveGameManager:on("loadGame", function(saveGame)
    local missionAreas = saveGame:getKey("missionAreas")
    if missionAreas then
      self.missionAreaIndex = missionAreas.current
    end

    local hub = saveGame:getKey("hub")
    if hub then
      if hub.tutorialCutscene then
        self.playTutorialCutscene = false
        self.tutorialCutscenePlayed = true
      end
    end
  end)
end

function HubWorld:awake()
  local eve = BehaviorUtil.findFirstOfType("Eve")
  local hero = BehaviorUtil.findFirstOfType("Hero")
  local joe = BehaviorUtil.findFirstOfType("TraderJoe")

  self.co = Co.create(function()
    if self.playTutorialCutscene then
      self:playTutorialCutsceneFn()
    end

    eve.coMain = eve.hubWorldRoute
  end)

  if math.floor(self.missionAreaIndex / 3) % 2 == 1 then
    Kernel.soundEmitter:postEvent("PlayBlizzardSound")

    Graphics.setColorLUT(AssetUtil.find("no_atlas/night_lut.png"), false)
    Graphics.setColorLUTAlpha(0.0)

    local snow = BehaviorUtil.findFirstOfType("SnowParticleSystem")
    snow:setProperty("acceleration", Vector(128.0, -64.0))
    snow:setProperty("spawnRate", 0.025)
    snow:prewarm(24)

    local banners = BehaviorUtil.findAllOfType("Banner")
    for i, banner in ipairs(banners) do
      banner:setWind(true)
    end

    local portals = BehaviorUtil.findAllOfType("Portal")
    for i, portal in ipairs(portals) do
      portal:setWind(true)
    end

    local dummies = BehaviorUtil.findAllOfType("TrainingDummy")
    for i, dummy in ipairs(dummies) do
      dummy:setWind(true)
    end
  end
end

function HubWorld:update()
  if self.letterboxAlpha > 0.0 then
    self:drawLetterbox()
  end
end

function HubWorld:fixedUpdate()
  if not self.co then
    return
  end

  if not self.co:update() then
    self.co = nil
  end
end

function HubWorld:drawLetterbox()
  local cameraPosition = Camera.getPosition()
  local cameraSize = Camera.getSize()
  local letterboxSize = 200

  Graphics.drawImage(self.pixelTexture, {
    position = cameraPosition,
    size = Vector(2000.0, letterboxSize),
    tint = Color(0.0, 0.0, 0.0, self.letterboxAlpha),
    isUI = true,
    zOrder = 10000
  })

  Graphics.drawImage(self.pixelTexture, {
    position = cameraPosition + Vector(0.0, 1080.0 - letterboxSize),
    size = Vector(2000.0, letterboxSize),
    tint = Color(0.0, 0.0, 0.0, self.letterboxAlpha),
    isUI = true,
    zOrder = 10000
  })
end

function HubWorld:playTutorialCutsceneFn()
  local eve = BehaviorUtil.findFirstOfType("Eve")
  local hero = BehaviorUtil.findFirstOfType("Hero")
  local joe = BehaviorUtil.findFirstOfType("TraderJoe")
  local hud = BehaviorUtil.findFirstOfType("HUD")
  hud:hide()
  Kernel.menuEvents:takeLock("tutorial")
  self.letterboxAlpha = 1.0

  eve.coMain = nil

  local tutorialStart = WorldUtil.getLandmarkByName("EveTutorialStart").position
  Co.yield(eve:createGoToCo(tutorialStart))
  eve.character:lookAt(hero.transform.worldPosition)
  hero.character:lookAt(eve.transform.worldPosition)
  hero.cameraFollower.enabled = false

  local cameraTarget = (eve.transform.worldPosition + hero.transform.worldPosition) * 0.5 - Camera.getSize() * 0.5
  local cameraPosition = Camera.getPosition()
  local dir = (cameraTarget - cameraPosition)

  local t = 0.0
  while t < 1.0 do
    t = t + Time.fixedDeltaTime
    Camera.setPosition(cameraPosition + dir * t)
    Co.yield()
  end

  eve.coMain = nil
  eve.walkCo = nil
  eve.character:setAnimation("idle_dialogue", true)
  eve.character:lookAt(hero.transform.worldPosition)
  eve:say("I see you're finally awake.")
  Co.sleep(2.0)
  eve:say("So what's the brilliant plan for getting us out of here?")
  Co.sleep(3.0)
  eve.character:setAnimation("idle", true)

  hero:say("Eve, what's going on?")
  Co.sleep(3.0)
  hero:say("Where are we?")
  Co.sleep(3.0)

  eve.character:setAnimation("idle_dialogue", true)
  eve:say("You seriously don't remember what just happened?")
  Co.sleep(3.0)
  eve.character:setAnimation("idle", true)

  hero:say("I'll go with... no?")
  Co.sleep(2.0)

  eve.character:setAnimation("idle_dialogue", true)
  eve:say("You don't remember how just five minutes ago")
  Co.sleep(3.0)
  eve:say("we got ambushed by a pack of those... things.")
  Co.sleep(4.0)
  eve.character:setAnimation("idle", true)
  hero:say("And?")
  Co.sleep(3.0)
  eve.character:setAnimation("idle_dialogue", true)
  eve:say("And what?")
  Co.sleep(2.0)
  eve:say("We were banished to this island by some nasty spell.")
  Co.sleep(4.0)
  eve:say("They said it's some kind of trial.")
  Co.sleep(3.0)
  eve:say("Something about runes.")
  Co.sleep(2.5)
  eve:say("And then becoming king of the arena.")
  Co.sleep(4.0)

  eve:say("You know these crazy monks and their non-stop trials.")
  Co.sleep(4.0)
  eve:say("It's always \"you're the chosen one\" and \"trial this, trial that\".")
  Co.sleep(4.0)
  eve:say("Just figure out a way out of here before I freeze to death.")
  Co.sleep(4.0)
  eve.character:setAnimation("idle", true)
  hero.character:lookAt(joe.transform.worldPosition)
  Co.sleep(0.5)

  local cameraTarget = (hero.transform.worldPosition + joe.transform.worldPosition) * 0.5 - Camera.getSize() * 0.5
  local cameraPosition = Camera.getPosition()
  local dir = (cameraTarget - cameraPosition)

  local t = 0.0
  while t < 1.0 do
    t = t + Time.fixedDeltaTime * 0.75
    Camera.setPosition(cameraPosition + dir * t)
    Co.yield()
  end

  hero:say("Wait, is that Joe over there?")
  Co.sleep(3.0)
  joe:say("Hi, kid!")
  Co.sleep(2.0)
  hero:say("Hi, Joe.")
  Co.sleep(3.0)
  eve.character:setAnimation("idle_dialogue", true)
  eve:say("Oh and I found this while you were knocked out cold.")
  Co.sleep(0.5)

  local cameraTarget = (hero.transform.worldPosition + eve.transform.worldPosition) * 0.5 - Camera.getSize() * 0.5
  local cameraPosition = Camera.getPosition()
  local dir = (cameraTarget - cameraPosition)

  local t = 0.0
  while t < 1.0 do
    t = t + Time.fixedDeltaTime * 0.75
    Camera.setPosition(cameraPosition + dir * t)
    Co.yield()
  end

  hero.character:lookAt(eve.transform.worldPosition)
  Co.sleep(2.5)
  CreateObject("ItemPouch", eve.transform.worldPosition - Vector(0.0, 48.0)):addComponent("ItemPouch", { lootTable = "EnergySurgeRune" })
  Co.sleep(2.5)
  eve:say("You should explore the island, there are probably more around.")
  Co.sleep(3.5)
  eve.character:setAnimation("idle", true)
  hero:say("Ok, I'm on it. I guess.")
  Co.sleep(1.0)

  eve.walkCo = eve.walkAnimFn

  local cameraTarget = hero.transform.worldPosition - Camera.getSize() * 0.5
  local cameraPosition = Camera.getPosition()
  local dir = (cameraTarget - cameraPosition)

  local t = 0.0
  while t < 1.0 do
    t = t + Time.fixedDeltaTime * 0.75
    Camera.setPosition(cameraPosition + dir * t)
    Co.yield()
  end

  hero.cameraFollower.enabled = true

  while self.letterboxAlpha > 0.0 do
    self.letterboxAlpha = self.letterboxAlpha - Time.fixedDeltaTime * 2.0
    Co.yield()
  end

  Kernel.menuEvents:releaseLock("tutorial")
  self.tutorialCutscenePlayed = true
  hud:show()
  SaveGameManager.save()

  eve.startIndex = 1
end