local SaveGameStats = require "util/savegamestats"
local SaveGameManager = require "util/savegamemanager"

local TheArena = Behavior("TheArena")
TheArena.property("mobList", "arena1")
TheArena.property("spawnSaws", false)
TheArena.property("sawsStartWave", 5)
TheArena.property("arenaId", 1)

function TheArena:initialize(properties)
  self.properties = TheArena.defaultProperties(properties)

  Graphics.setColorLUT(AssetUtil.find("no_atlas/default_lut.png"), true)
  Graphics.setColorLUT(AssetUtil.find("no_atlas/night_lut.png"), false)
  Graphics.setColorLUTAlpha(0.0)

  self:on("destroy", function()
    UI.detachView("ArenaView")
  end)

  self.bannerText = false

  local world = World.getCurrent()
  self.worldPixelSize = world:getTileSize() * world:getSize()

  UI.attachView("ArenaView", "ui/arena.rml")
  self:updateUIBindings(nil, false)

  self.mobLists = {
    arena1 = {
      [1] = { "SnowDraugur" },
      [2] = { "SnowDraugur", "Draugur" },
      [3] = { "Draugur", "Preacher" },
      [4] = { "Draugur", "SnowDraugur", "Preacher" },
      [5] = { "LittleAssassin", "LittleAssassin" },
      [6] = { "SnowDraugur", "Preacher", "Preacher", "Draugur" },
      [7] = { "LittleAssassin", "LittleAssassin", "LittleAssassin" },
      [8] = { "Preacher", "Preacher", "Preacher" },
      [9] = { "SnowDraugur", "Draugur", "Preacher", "LittleAssassin" },
      [10] = { "MiniBoss" }
    },
    arena2 = {
      [1] = { "Knight", "Draugur", "SnowDraugur" },
      [2] = { "Knight", "Draugur", "Draugur", "Preacher" },
      [3] = { "Knight", "LittleAssassin", "SnowDraugur", "Draugur" },
      [4] = { "Preacher", "Knight", "Draugur", "Preacher" },
      [5] = { "Knight", "Knight", "SnowDraugur" },
      [6] = { "Preacher", "Preacher", "Preacher", "SnowDraugur" },
      [7] = { "Knight", "Knight", "Preacher", "SnowDraugur" },
      [8] = { "Knight", "Knight", "Knight" },
      [9] = { "Knight", "Draugur", "Preacher", "LittleAssassin", "SnowDraugur" },
      [10] = { "MiniBoss", "MiniBoss" }
    },
	
    arena3 = {
      [1] = { "Draugur", "Draugur", "SnowDraugur" },
      [2] = { "LittleAssassin", "LittleAssassin", "SnowDraugur" },
      [3] = { "Preacher", "Preacher", "SnowDraugur", "SnowDraugur", "Draugur" },
      [4] = { "Knight", "Knight", "SnowDraugur", "SnowDraugur", "Draugur" },
      [5] = { "LittleAssassin", "LittleAssassin", "LittleAssassin", "Preacher", "Preacher" },
      [6] = { "Knight", "Knight", "Knight", "Draugur", "Draugur", "Draugur" },
      [7] = { "LittleAssassin", "LittleAssassin", "LittleAssassin", "LittleAssassin", "LittleAssassin" },
      [8] = { "Knight", "Knight", "Knight", "Knight", "SnowDraugur", "SnowDraugur" },
      [9] = { "Knight", "Knight", "LittleAssassin", "LittleAssassin", "Draugur", "Draugur", "SnowDraugur", "SnowDraugur",  "LittleAssassin", "LittleAssassin"},
      [10] = { "Knight", "Knight", "Knight", "LittleAssassin", "LittleAssassin", "LittleAssassin", "Draugur", "Draugur", "Draugur", "SnowDraugur", "SnowDraugur", "SnowDraugur", "LittleAssassin", "LittleAssassin", "LittleAssassin" }
    }
  }

  self.mobsCost = {
    Draugur = 50,
    LittleAssassin = 60,
    Preacher = 70,
    Knight = 85,
    SnowDraugur = 95,
    MiniBoss = 650
  }

  self.packs = {
    { "LittleAssassin", "LittleAssassin" },
    { "Draugur", "Draugur" },
    { "Draugur", "SnowDraugur" },
    { "SnowDraugur", "SnowDraugur" },
    { "Draugur", "Preacher" },
    { "Draugur", "Preacher", "Preacher" },
    { "Knight", "Knight" },
    { "Knight", "LittleAssassin", "LittleAssassin" },
    { "SnowDraugur", "LittleAssassin", "LittleAssassin" },
    { "Knight", "Draugur", "Draugur", "LittleAssassin" },
    { "Knight", "Draugur", "Draugur" },
    { "Knight", "Preacher", "Preacher" },
    { "SnowDraugur", "SnowDraugur", "Preacher", "Preacher" },
    { "Knight", "Draugur", "Preacher" },
    { "LittleAssassin", "LittleAssassin", "LittleAssassin", "LittleAssassin" },
    { "MiniBoss", "Draugur", "Draugur", "Preacher", "Preacher" }
  }

  self.cheapeastPack = 10000
  for i=1,#self.packs do
    local cost = self:calculatePackCost(self.packs[i])
    if cost < self.cheapeastPack then
      self.cheapeastPack = cost
    end
  end

  self.debug = false

  self.spawnWave = Co.create(function()
    self:updateUIBindings(nil, true)

    Co.sleep(4.0)

    self:updateUIBindings(nil, false)

    if (self.properties.arenaId == 1 or self.properties.arenaId == 2) and self.currentWave == 10 then
      Kernel.musicController:playBossMusic()

      local t = 0.0
      while t < 1.0 do
        Graphics.setColorLUTAlpha(t)
        t = t + Time.fixedDeltaTime
        Co.yield()
      end

      Graphics.setColorLUTAlpha(1.0)
    end

    local enemies = {}
    self.aliveEnemies = {}
    self.totalEnemyCount = 0

    if self.properties.mobList == "infinite" then
      local points = self:calculatePointsForWave(self.currentWave)
      enemies = self:createEnemyList(points)
    else
      local mobList = self.mobLists[self.properties.mobList]
      enemies = mobList[self.currentWave]
    end

    for i, enemyName in ipairs(enemies) do
      self:spawnEnemy(enemyName)
    end

    local aliveEnemies = self:countAliveEnemies()
    while aliveEnemies > 0 do
      aliveEnemies = self:countAliveEnemies()
      local enemies = {}
      for i=1,self.totalEnemyCount do
        enemies[i] = i <= aliveEnemies
      end

      Co.yield()

      self:updateUIBindings(enemies, false)
    end

    Co.sleep(2.5)

    self.currentWave = self.currentWave + 1
    local maxWaveReached = SaveGameStats.get("maxArenaWaveReached")
    if self.currentWave > maxWaveReached then
      SaveGameStats.set("maxArenaWaveReached", self.currentWave)
    end

    if self.currentWave > 10 then
      local nextTimer = 5.0
      while nextTimer > 0.0 do
        self.bannerText = "Arena complete. Returning to hub in " .. tostring(math.floor(nextTimer + 1.0)) .. "."
        self:updateUIBindings()
        nextTimer = nextTimer - Time.fixedDeltaTime
        Co.yield()
      end

      self.bannerText = false
      self:updateUIBindings()

      local saveGame = SaveGameManager.getCurrent()
      local arenaInfo = saveGame:getKey("arenaInfo")
      arenaInfo = arenaInfo or { current = 0 }
      arenaInfo.current = self.properties.arenaId
      saveGame:setKey("arenaInfo", arenaInfo)

      SaveGameManager.save()
      World.setCurrent("worlds/demo/hub.wrld")
      return
    end
  end)

  self.mainCo = Co.create(function()
    Co.sleep(1.5)

    while true do
      Co.yield(self.spawnWave())
    end
  end)

  self.resetTargetsCo = Co.create(function()
    while true do
      Co.yieldThrottle(1.0, function()
        for i=1,self.totalEnemyCount do
          local enemy = self.aliveEnemies[i]
          if enemy and not enemy.character.target then
            enemy.character.target = self.heroRef
          end
        end
      end)
    end
  end)

  self.heroAliveCo = Co.create(function()
    while true do
      if not self.heroRef.health.isAlive and
        not self.heroRef.statusEffects:hasStatusEffect("SecondChance") then
        break
      end

      Co.sleep(1.0)
    end

    Co.sleep(2.0)

    self.mainCo = nil
    self.resetTargetsCo = nil

    local nextTimer = 4.0
    while nextTimer > 0.0 do
      self.bannerText = "You died. Returning to hub in " .. tostring(math.floor(nextTimer + 1.0)) .. "."
      self:updateUIBindings()
      nextTimer = nextTimer - Time.fixedDeltaTime
      Co.yield()
    end

    self.bannerText = false
    self:updateUIBindings()

    SaveGameManager.save()
    if not _EDITOR then
      World.setCurrent("worlds/demo/hub.wrld")
    end
  end)

  if self.properties.spawnSaws then
    self.spawnSawsCo = Co.create(function()
      while true do
        while self.currentWave < self.properties.sawsStartWave do
          Co.sleep(5.0)
        end

        local location = nil

        while not location do
          location = self:findFreeLocation(Vector(300.0, 100.0))
          Co.yield()
        end

        CreateObject("SawTrap", location + Vector(-100.0, 32.0)):addComponent("SawTrap", {
          selfDestruct = true
        })

        Co.sleep(0.25)
      end
    end)
  end

  self.aliveEnemies = {}
  self.totalEnemyCount = 0

  self.currentWave = 1

  Graphics.useFogOfWar(true)
end

function TheArena:updateUIBindings(enemies, showRibbon)
  enemies = enemies or {}

  UI.updateBinding("ArenaView", {
    enemies = enemies,
    enemyCount = #enemies,
    showWaveRibbon = showRibbon,
    currentWave = self.currentWave,
    banner = self.bannerText,
    rnd = Random.value()
  })
end

function TheArena:awake()
  self.heroRef = BehaviorUtil.findFirstOfType("Hero")

  self.spawnLocations = {}

  local landmarks = WorldUtil.getLandmarks()
  for i, landmark in ipairs(landmarks) do
    if landmark.name == "ArenaSpawn" then
      self.spawnLocations[#self.spawnLocations+1] = landmark.position
    end
  end

  local triggers = WorldUtil.getTriggers()
  for i, trigger in ipairs(triggers) do
    if trigger.type == "BarrelsSpawnArea" then
      self:spawnBarrels(trigger.bottomLeft, trigger.topRight)
    end
  end
end


function TheArena:fixedUpdate()
  if self.heroAliveCo then
    if not self.heroAliveCo:update() then
      self.heroAliveCo:reset()
    end
  end

  if self.resetTargetsCo then
    if not self.resetTargetsCo:update() then
      self.resetTargetsCo:reset()
    end
  end

  if self.mainCo then
    if not self.mainCo:update() then
      self.mainCo = nil
    end
  end

  if self.spawnSawsCo then
    if not self.spawnSawsCo:update() then
      self.spawnSawsCo:reset()
    end
  end
end

function TheArena:createEnemyList(totalPoints)
  local points = totalPoints
  
  local enemies = {}

  while points > 0 do
    local pack, packCost = self:getRandomPackWithCost(points)
    if not pack then
      break
    end

    points = points - packCost
    for i, enemyName in ipairs(pack) do
      enemies[#enemies+1] = enemyName
    end
  end

  return enemies
end

function TheArena:calculatePackCost(pack)
  local cost = 0.0

  for i, enemyName in ipairs(pack) do
    cost = cost + self.mobsCost[enemyName]
  end

  return cost
end

function TheArena:getRandomPackWithCost(points)
  if points < self.cheapeastPack then
    return nil
  end

  local tries = 128
  while tries > 0 do
    local pack = table.pickOne(self.packs)
    local cost = self:calculatePackCost(pack)

    if cost < points then
      return pack, cost
    end

    tries = tries - 1
  end
end

function TheArena:spawnEnemy(enemyName)
  local position = table.pickOne(self.spawnLocations)
  position = position + Random.unitVector() * 32.0

  local lootTable = "ArenaMob"
  if enemyName == "MiniBoss" then
    lootTable = "ArenaBoss"
  end

  local enemy = CreateObject(enemyName, position):addComponent(enemyName, {
    faction = "ArenaMob",
    lootTable = lootTable,
    detectionRadius = 2000
  })

  enemy.character.target = self.heroRef

  local index = #self.aliveEnemies+1
  self.aliveEnemies[index] = enemy

  enemy.object:on("destroy", function ()
    self.aliveEnemies[index] = nil
  end)

  self.totalEnemyCount = self.totalEnemyCount + 1

  return enemy
end

function TheArena:countAliveEnemies()
  local aliveCount = 0

  for i=1,self.totalEnemyCount do
    if self.aliveEnemies[i] then
      aliveCount = aliveCount + 1
    end
  end

  return aliveCount
end

function TheArena:calculatePointsForWave(wave)
  return 125 + ((wave - 1) * (35.0 + math.floor(Random.value() * 25.0))) + math.floor(Random.value() * 25.0)
end

function TheArena:spawnBarrels(bottomLeft, topRight)
  local halfSize = Vector(96.0, 96.0)

  local tries = 256
  local count = 6
  while count > 0 do
    local position = bottomLeft + (topRight - bottomLeft) * Random.vector()

    local collision = false

    Physics.aabbQuery(position - halfSize, position + halfSize, function (object)
      collision = true
      return false
    end, flags(World.layer("MovementCollider"), World.layer("World")))

    if not collision then
      CreateObject("Barrel", position):addComponent("Barrel", {
        lootTable = "ArenaBarrel"
      })

      count = count - 1
    end

    tries = tries - 1
    if tries < 0 then
      break
    end
  end
end

function TheArena:findFreeLocation(size)
  local trigger = WorldUtil.getTriggerByType("BarrelsSpawnArea")

  local position = trigger.bottomLeft + Random.vector() * (trigger.topRight - trigger.bottomLeft)

  local halfSize = size * 0.5
  local p0 = position - halfSize
  local p1 = position + halfSize

  local collision = false
  Physics.aabbQuery(position - halfSize, position + halfSize, function (object)
    collision = true
    return false
  end, World.layer("World"))

  if not collision then
    return position
  end

  return nil
end
