local Transform = require "lib/components/transform"
local SpriteRenderer = require "lib/components/spriterenderer"
local PhysicsCollider = require "lib/components/physicscollider"
local ParticleEmitter = require "lib/components/particleemitter"
local PhysicsJoint = require "lib/components/physicsjoint"
local LightSource = require "lib/components/lightsource"
local SkeletonRenderer = require "lib/components/skeletonrenderer"

local ComponentEditor = EditorBehavior("ComponentEditor")

function ComponentEditor:initialize()
  self.visible = false
  self.object = nil
end

function ComponentEditor:update(deltaTime)
  if self.visible then
    self:drawWindow()
  end
end

function ComponentEditor:drawWindow()
  self.visible = EditorUI.beginWindowMenubar("Component editor", 400, 500)
  
  if not self.visible then
    EditorUI.endWindow()
    return
  end
  
  if self.object == nil then
    EditorUI.textWrapped("Click on a scene object")   
    EditorUI.endWindow()
    return
  end
  
  if not _L._objectExists(self.object._objectId) then
    EditorUI.endWindow()
    return
  end
  
  local objectBehaviors = _behaviorInstances[self.object._objectId]
  if not objectBehaviors then
    objectBehaviors = {}
  end
  
  if EditorUI.beginMenuBar() then
    if EditorUI.beginMenu("Add behavior") then
      if self.object.spriteRenderer == nil then
        if EditorUI.menuItem("SpriteRenderer") then
          self.object:addComponent("SpriteRenderer")
        end
      end
      
      if self.object.physicsCollider == nil then
        if EditorUI.menuItem("PhysicsCollider") then
          self.object:addComponent("PhysicsCollider")
        end
      end
      
      if self.object.physicsJoint == nil then
        if EditorUI.menuItem("PhysicsJoint") then
          self.object:addComponent("PhysicsJoint")
        end
      end
      
      if self.object.particleEmitter == nil then
        if EditorUI.menuItem("ParticleEmitter") then
          self.object:addComponent("ParticleEmitter")
        end
      end
      
      if self.object.lightSource == nil then
        if EditorUI.menuItem("LightSource") then
          self.object:addComponent("LightSource")
        end
      end

      if self.object.skeletonRenderer == nil then
        if EditorUI.menuItem("SkeletonRenderer") then
          self.object:addComponent("SkeletonRenderer")
        end
      end

      if self.object.soundEmitter == nil then
        if EditorUI.menuItem("SoundEmitter") then
          self.object:addComponent("SoundEmitter")
        end
      end

      for behaviorName, behaviorType in pairs(_behaviorTypes) do
        if objectBehaviors[behaviorName] == nil and behaviorName ~= "Main" then
          if EditorUI.menuItem(behaviorName) then
            self.object:addComponent(behaviorName)
          end
        end
      end
      
      EditorUI.endMenu()
    end
    
    if EditorUI.beginMenu("Remove behavior") then
      if self.object.spriteRenderer ~= nil then
        if EditorUI.menuItem("SpriteRenderer") then
          self.object:removeComponent("SpriteRenderer")
        end
      end
      
      if self.object.physicsCollider ~= nil then
        if EditorUI.menuItem("PhysicsCollider") then
          self.object:removeComponent("PhysicsCollider")
        end
      end
      
      if self.object.physicsJoint ~= nil then
        if EditorUI.menuItem("PhysicsJoint") then
          self.object:removeComponent("PhysicsJoint")
        end
      end
      
      if self.object.particleEmitter ~= nil then
        if EditorUI.menuItem("ParticleEmitter") then
          self.object:removeComponent("ParticleEmitter")
        end
      end
      
      if self.object.lightSource ~= nil then
        if EditorUI.menuItem("LightSource") then
          self.object:removeComponent("LightSource")
        end
      end

      if self.object.skeletonRenderer ~= nil then
        if EditorUI.menuItem("SkeletonRenderer") then
          self.object:removeComponent("SkeletonRenderer")
        end
      end

      if self.object.soundEmitter ~= nil then
        if EditorUI.menuItem("SoundEmitter") then
          self.object:removeComponent("SoundEmitter")
        end
      end
    
      for behaviorName, behaviorType in pairs(objectBehaviors) do
        if EditorUI.menuItem(behaviorName) then
          self.object:removeComponent(behaviorName)
        end
      end
    
      EditorUI.endMenu()
    end
    
    if EditorUI.beginMenu("Object") then
      if EditorUI.menuItem("Destroy") then
        self.object:destroy()
        self.object = nil
      end
    
      EditorUI.endMenu()
    end
    
    EditorUI.endMenuBar()
  end

  if self.object == nil then
    EditorUI.endWindow()
    return
  end
  
  local objectName = self.object:getName()
  local newName = EditorUI.inputText("Name", objectName)
  if newName ~= objectName then
    self.object:setName(newName)
  end
  
  self:drawTransformEditor(self.object.transform)
  
  if self.object.spriteRenderer ~= nil then
    self:drawSpriteRendererEditor(self.object.spriteRenderer)
  end
  
  if self.object.physicsCollider ~= nil then
    self:drawPhysicsColliderEditor(self.object.physicsCollider)
  end

  if self.object.particleEmitter ~= nil then
    self:drawParticleEmitterEditor(self.object.particleEmitter)
  end
  
  if self.object.lightSource ~= nil then 
    self:drawLightSourceEditor(self.object.lightSource)
  end

  if self.object.skeletonRenderer ~= nil then
    self:drawSkeletonRendererEditor(self.object.skeletonRenderer)
  end
  
  if self.object.soundEmitter ~= nil then
    self:drawSoundEmitterEditor(self.object.soundEmitter)
  end

  local behaviors = _behaviorInstances[self.object._objectId]
  
  if behaviors ~= nil then
    for behaviorName, behavior in pairs(behaviors) do
      self:drawBehaviorEditor(behavior)
    end
  end
  
  EditorUI.endWindow()
end

function ComponentEditor:drawTransformEditor(transform)
  if not EditorUI.collapsingHeader("Transform") then
    return
  end

  local parent = transform:getParent()
  if parent == nil then
    EditorUI.text("Parent: None")
  else
    EditorUI.text("Parent: ")
    EditorUI.sameLine()
    if EditorUI.button(parent.object:getName() .. " (#" .. parent.object:getId() .. ")") then
      self.object = parent.object
      return
    end
  end
  
  local position = EditorUI.inputFloat2("Position", transform:getLocalPosition())
  transform:setLocalPosition(position)
  
  if parent ~= nil then
    EditorUI.inputFloat2("Position (World)", transform:getWorldPosition())
  end

  if EditorUI.isItemHovered() then
    Debug.drawLine(Input.getMouseWorldPosition(), transform:getWorldPosition(), 0XFFFFFFFF)
  end
  
  local scale = EditorUI.inputFloat2("Scale", transform:getLocalScale())
  transform:setLocalScale(scale)

  if parent ~= nil then
    EditorUI.inputFloat2("Scale (World)", transform:getWorldScale())
  end
  
  if EditorUI.isItemHovered() then
    Debug.drawLine(Input.getMouseWorldPosition(), transform:getWorldPosition(), 0XFFFFFFFF)

    local bounds = self.object.transform:getWorldBounds()
    Debug.drawRect(bounds.bottomLeft, bounds.topRight - bounds.bottomLeft, 0xFFFF00FF)
  end
end

function ComponentEditor:drawSpriteRendererEditor(spriteRenderer)
  if not EditorUI.collapsingHeader("Sprite Renderer") then
    return
  end
end

function ComponentEditor:drawPhysicsColliderEditor(physicsCollider)
  if not EditorUI.collapsingHeader("Physics Collider") then
    return
  end
  
  local colliderType = physicsCollider:getType()
  local isStatic = EditorUI.radioButton("Static", colliderType == physicsCollider.Types.Static)
  EditorUI.sameLine()
  local isDynamic = EditorUI.radioButton("Dynamic", colliderType == physicsCollider.Types.Dynamic)
  EditorUI.sameLine()
  local isKinematic = EditorUI.radioButton("Kinematic", colliderType == physicsCollider.Types.Kinematic)

  local newColliderType = colliderType
  if isStatic then
    newColliderType = physicsCollider.Types.Static
  elseif isDynamic then
    newColliderType = physicsCollider.Types.Dynamic
  elseif isKinematic then
    newColliderType = physicsCollider.Types.Kinematic
  end

  if newColliderType ~= colliderType then
    physicsCollider:setType(newColliderType)
  end

  local mask = physicsCollider:getMask()
  local categories = physicsCollider:getCategories()

  local maskLayers = {}
  local categoriesLayers = {}
  local layerCount = World.getLayerCount()

  for i=0,layerCount-1 do
    local layerName = World.layerIndexToName(i)

    if Bitwise.band(mask, Bitwise.lshift(1, i)) > 0 then
      table.push(maskLayers, layerName)
    end

    if Bitwise.band(categories, Bitwise.lshift(1, i)) > 0 then
      table.push(categoriesLayers, layerName)
    end
  end

  EditorUI.text("Categories: " .. table.concat(categoriesLayers, ", "))
  EditorUI.text("Mask: " .. table.concat(maskLayers, ", "))

  local awake = physicsCollider:isAwake()
  local newAwake = EditorUI.checkbox("Awake", awake)
  if awake ~= newAwake then
    physicsCollider:setAwake(newAwake)
  end
end

function ComponentEditor:drawParticleEmitterEditor(particleEmitter)
  if not EditorUI.collapsingHeader("Particle Emitter") then
    return
  end

  local image = particleEmitter:getImage()
  EditorUI.inputText("Image", image)

  local particleCount = particleEmitter:getAliveParticleCount()
  EditorUI.text("Alive particles: " .. tostring(particleCount))
end

function ComponentEditor:drawLightSourceEditor(lightSource)
  if not EditorUI.collapsingHeader("Light Source") then
    return
  end
  
  local size = lightSource:getSize()
  size = EditorUI.sliderInt("Size", size, 16, 1024)
  lightSource:setSize(size)
  
  local shadowStrength = lightSource:getShadowStrength()
  shadowStrength = EditorUI.sliderFloat("Shadow strength", shadowStrength, 0.0, 1.0)
  lightSource:setShadowStrength(shadowStrength)
  
  local color = lightSource:getColor()
  EditorUI.color("Color", color)
  lightSource:setColor(color)

  local intensity = lightSource:getAmbientIntensity()
  intensity = EditorUI.sliderFloat("Ambient Intensity", intensity, 0.0, 2.0)
  lightSource:setAmbientIntensity(intensity)
end

function ComponentEditor:drawSkeletonRendererEditor(skeletonRenderer)
  if not EditorUI.collapsingHeader("Skeleton Renderer") then
    return
  end
  
  local skeletonId = skeletonRenderer:getSkeleton()
  local skeleton = AssetUtil.getById(skeletonId)

  if skeleton ~= nil then
    EditorUI.text("Skeleton: " .. skeleton:getName())
  else
    EditorUI.text("Skeleton: None")
  end

  local color = skeletonRenderer:getTintColor()
  EditorUI.color("Tint Color", color)
  skeletonRenderer:setTintColor(color)
end

function ComponentEditor:drawSoundEmitterEditor(soundEmitter)
  if not EditorUI.collapsingHeader("Sound Emitter") then
    return
  end

end

function ComponentEditor:drawBehaviorEditor(behavior)
  if not EditorUI.collapsingHeader(behavior._name) then
    return
  end
  
  if behavior.customEditor ~= nil then
    behavior:customEditor()
    return
  end
  
  for prop, _ in spairs(behavior._props) do
    local value = behavior[prop]

    if type(value) == "number" then
      local val = value
      local newVal = EditorUI.inputFloat(prop, val)
      if val ~= newVal then
        value = newVal
      end
    elseif type(value) == "cdata" then
      value = EditorUI.inputFloat2(prop, value)
    elseif type(value) == "table" then
      if type(value.r) == "number" and type(value.g) == "number" and
        type(value.b) == "number" and type(value.a) == "number" then
        value = EditorUI.color(prop, value)
      end
    elseif type(value) == "string" then
      local newVal = EditorUI.inputText(prop, value)
      if newVal ~= value then
        value = newVal
      end
    elseif type(value) == "boolean" then
      local newVal = EditorUI.checkbox(prop, value)
      if newVal ~= value then
        value = newVal
      end
    else
      EditorUI.text(prop .. " = nil")
    end

    behavior[prop] = value
  end
end
