require "editor/console"
require "editor/componenteditor"
require "editor/globalconfig"
require "editor/stats"
require "editor/scenegraph"
require "editor/editorsettings"
require "editor/soundtest"
require "editor/profiler"

local Editor = EditorBehavior("Editor")

function Editor:initialize()
  -- common

  self.console = _instantiateEditorBehavior("Console")
  self.componentEditor = _instantiateEditorBehavior("ComponentEditor")
  self.globalConfig = _instantiateEditorBehavior("GlobalConfig")
  self.editorSettings = _instantiateEditorBehavior("EditorSettings")
  self.soundTest = _instantiateEditorBehavior("SoundTest")
  self.profiler = _instantiateEditorBehavior("Profiler")

  -- game editor

  self.stats = _instantiateEditorBehavior("Stats")
  self.sceneGraph = _instantiateEditorBehavior("SceneGraph")
  self.sceneGraph.componentEditor = self.componentEditor

  self.simulationState = "Running"
  self.drawGrid = false

  Input.onKeyDown(KEYS.KEY_F5, function()
    Kernel:destroy()
    _L._debugExitSimulation()
  end)

  Input.onKeyDown(KEYS.KEY_F4, function()
    if self.simulationState == "Running" then
      _L._debugSetSimulationState(false)
      self.simulationState = "Paused"
    else
      _L._debugSetSimulationState(true)
      self.simulationState = "Running"
    end
  end)

  Input.onKeyDown(KEYS.KEY_F10, function()
    self.simulationState = "Paused"
    _L._debugStepSimulation()
  end)

  Input.onKeyDown(KEYS.KEY_GRAVE_ACCENT, function()
    self.console.visible = not self.console.visible
  end)
end

function Editor:update()
  self:drawTopMenu()

  local objUnderCursor = Input.getObjectUnderCursor()
  if objUnderCursor ~= nil then
    local position = objUnderCursor.transform:getWorldPosition()
    local scale = objUnderCursor.transform:getWorldScale()
    local bounds = objUnderCursor.transform:getWorldBounds()
    Debug.drawRect(bounds.bottomLeft, bounds.topRight - bounds.bottomLeft, 0xFF00FFFF)

    if Input.getMouseButtonDown(0) then
      self.componentEditor.object = objUnderCursor
      self.componentEditor.visible = true
    end
  end

  if self.drawGrid then
    local lStart = Vector(0.0, 0.0)
    local lEnd = Vector(0.0, 0.0)

    local world = World.getCurrent()
    local tileSize = world:getTileSize()
    local worldSize = tileSize * world:getSize()

    for x=0.0,worldSize.x,tileSize.x do
      lStart.x = x
      lStart.y = 0.0

      lEnd.x = x
      lEnd.y = worldSize.y

      Debug.drawLine(lStart, lEnd, 0xFF00FFFF)
    end

    for y=0.0,worldSize.y,tileSize.y do
      lStart.x = 0.0
      lStart.y = y

      lEnd.x = worldSize.x
      lEnd.y = y

      Debug.drawLine(lStart, lEnd, 0xFF00FFFF)
    end
  end
end

function Editor:drawTopMenu()
  EditorUI.setNextWindowPos(Vector(0.0, 0.0))
  EditorUI.setNextWindowSize(Vector(1280.0, 32.0))
  EditorUI.beginWindowBorderlessMenu("TopMenu", true, 1280, 8)

  if EditorUI.beginMenuBar() then
    if EditorUI.beginMenu("File") then
      if EditorUI.menuItem("Quit") then
        os.exit()
      end

      EditorUI.endMenu()
    end

    if EditorUI.beginMenu("Window") then
      if EditorUI.menuItem("Console") then
        self.console.visible = true
      end
      if EditorUI.menuItem("Component editor") then
        self.componentEditor.visible = true
      end
      if EditorUI.menuItem("Global configuration") then
        self.globalConfig.visible = true
      end
      if EditorUI.menuItem("Statistics") then
        self.stats.visible = true
      end
      if EditorUI.menuItem("Scene graph") then
        self.sceneGraph.visible = true
      end
      if EditorUI.menuItem("Settings") then
        self.editorSettings.visible = true
      end
      if EditorUI.menuItem("Sound test") then
        self.soundTest.visible = true
      end
      if EditorUI.menuItem("Profiler") then
        self.profiler.visible = true
      end

      EditorUI.endMenu()
    end

    if EditorUI.beginMenu("Tools") then
      if EditorUI.menuItem("Reload shaders") then
        Graphics.reloadShaders()
      end

      if not self.drawGrid then
        if EditorUI.menuItem("Show grid") then
          self.drawGrid = true
        end
      else
        if EditorUI.menuItem("Hide grid") then
          self.drawGrid = false
        end
      end

      EditorUI.endMenu()
    end

    if EditorUI.beginMenu("Simulation: " .. self.simulationState) then
      if EditorUI.menuItem("Pause simulation") then
        _L._debugSetSimulationState(false)
        self.simulationState = "Paused"
      end

      if EditorUI.menuItem("Run simulation") then
        _L._debugSetSimulationState(true)
        self.simulationState = "Running"
      end

      if EditorUI.menuItem("Stop simulation") then
        Kernel:destroy()
        _L._debugExitSimulation()
      end

      EditorUI.endMenu()
    end
  end

  EditorUI.endMenuBar()

  EditorUI.endWindow()
end

_instantiateEditorBehavior("Editor")
