local WorldObjectBase = require "lib/worldobjectbase"

local awakeCalled = false
_behaviorTypes = {}
_behaviorTypeNames = {}
_behaviorInstances = {}

Metatype = ReadOnlyClass({
  Route = "route",
  Behavior = "behavior"
}, "Metatype")

function Behavior(behaviorName)
  local info = debug.getinfo(2, 'S')

  if _behaviorTypes[behaviorName] ~= nil then
    print("WARNING! Behavior \"" .. behaviorName .. "\" is already defined! " .. info.source:sub(2))
    return
  end

  local behavior = {
    addComponent = function (self, componentType, ...)
      return self.object:addComponent(componentType, ...)
    end,
    removeComponent = function (self, componentType)
      return self.object:removeComponent(componentType)
    end,
    emit = function (self, ...)
      return self.object:emit(...)
    end,
    on = function (self, eventType, callback)
      return self.object:on(eventType, callback)
    end,
    once = function (self, eventType, callback)
      return self.object:once(eventType, callback)
    end
  }

  behavior._name = behaviorName
  behavior._path = info.source:sub(2)
  behavior._properties = {}
  behavior._editorIcon = nil

  behavior.defaultProperties = function (properties)
    properties = properties or {}

    for i, prop in ipairs(behavior._properties) do
      if properties[prop.name] == nil then
        properties[prop.name] = prop.value
      end
    end

    return properties
  end

  behavior.property = function (propertyName, value, metatype)
    local props = behavior._properties
    metatype = metatype or "default"

    if type(value) == "number" then
      props[#props + 1] = {
        name = propertyName,
        value = value,
        type = "number",
        metatype = metatype
      }
    elseif type(value) == "string" then
      props[#props + 1] = {
        name = propertyName,
        value = value,
        type = "string",
        metatype = metatype
      }
    elseif type(value) == "cdata" then
      props[#props + 1] = {
        name = propertyName,
        value = value,
        type = "vector",
        metatype = metatype
      }
    elseif type(value) == "table" then
      props[#props + 1] = {
        name = propertyName,
        value = value,
        type = "color",
        metatype = metatype
      }
    elseif type(value) == "boolean" then
      props[#props + 1] = {
        name = propertyName,
        value = value,
        type = "boolean",
        metatype = metatype
      }
    end
  end

  behavior.editorIcon = function (assetName)
    behavior._editorIcon = assetName
  end

  _behaviorTypes[behaviorName] = behavior
  _behaviorTypeNames[#_behaviorTypeNames+1] = behaviorName
  return behavior
end

function _instantiateBehavior(objectId, behaviorName, ...)
  if _behaviorTypes[behaviorName] == nil then
    print("Error! Cannot instantiate behavior " .. behaviorName .. " because it does not exist!")
    return nil
  end

  local constructorArgs = {...}

  local object = WorldObject(objectId)  

  local instance = {}
  instance.object = object
  instance._props = {}
  
  local keyCache = {}  

  setmetatable(instance, {
    __index = _behaviorTypes[behaviorName],
    __newindex = function(t, k, v)
      if not keyCache[k] then
        keyCache[k] = k:firstToUpper()
      end

      local key = keyCache[k]

      if _behaviorTypes[key] ~= nil then
        print("ERROR! " .. behaviorName .. " cannot assign to field \"" .. k
          .. "\" as it shadows behavior \"" .. key .. "\" (stacktrace follows)")
          
        print(debug.traceback())
        return
      end

      if WorldObjectBase.Components[key] ~= nil then
        print("ERROR! " .. behaviorName .. " cannot assign to field \"" .. k
          .. "\" as it shadows component \"" .. key .. "\" (stacktrace follows)")
          
        print(debug.traceback())
        return
      end

      instance._props[k] = true
      rawset(t, k, v)
    end 
  })
  
  if _behaviorInstances[objectId] == nil then
    _behaviorInstances[objectId] = {}
  end

  for componentName, componentClass in pairs(WorldObjectBase.Components) do
    local memberName = componentName:firstToLower()
    local componentInstance = rawget(object, memberName)
    if componentInstance ~= nil then
      rawset(instance, memberName, componentInstance)
    end
  end

  for behaviorName, behavior in pairs(_behaviorInstances[object._objectId]) do
    if behaviorName ~= componentType then
      rawset(instance, behaviorName:firstToLower(), behavior)
    end
  end
  
  _behaviorInstances[objectId][behaviorName] = instance
  
  object:on("attachComponent", function (component)
    rawset(instance, component._name:firstToLower(), component)
  end)

  object:on("detachComponent", function (component)
    rawset(instance, component._name:firstToLower(), nil)
  end)

  if instance.initialize ~= nil then
    instance:initialize(unpack(constructorArgs))
  end

  if awakeCalled and instance.awake ~= nil then
    instance:awake()
  end

  return instance
end

function _destroyBehavior(objectId, behaviorName)
  if _behaviorInstances[objectId] ~= nil then
    _behaviorInstances[objectId][behaviorName] = nil
  end
end

function _destroyBehaviorsForObject(objectId)
  if _behaviorInstances[objectId] ~= nil then
    for name, behavior in pairs(_behaviorInstances[objectId]) do
      _destroyBehavior(objectId, name)
    end
  end
  
  _behaviorInstances[objectId] = nil
end

function _callFunctionForAllBehaviors(fnName, ...)
  if fnName == "awake" then
    awakeCalled = true
  end

  local args = {...}

  local all = _behaviorInstances

  for objectId, behaviors in pairs(all) do
    for behaviorName, behavior in pairs(behaviors) do
      if behavior[fnName] ~= nil then
        behavior[fnName](behavior, unpack(args))
      end
    end
  end
end

function _fixedUpdateBehaviors(fixedDeltaTime)
  Time.setFixedDeltaTime(fixedDeltaTime)

  if World.isLoaded() then
    _physicsDispatchCollisionInfo()
    _skeletonRendererDispatchEvents()
    _pathfindingHandleIncomingRequests()
  end

  _callFunctionForAllBehaviors("fixedUpdate", fixedDeltaTime)
end

function _updateBehaviors(deltaTime, timeScale)
  Time.setDeltaTime(deltaTime, timeScale)

  _inputDrainMessageQueue()
  _updateTimers(deltaTime * timeScale)
  _callFunctionForAllBehaviors("update", deltaTime * timeScale)
end
