local CO_DEBUG = false

local Co = {} 

function Co.create(fn, ...)
  if CO_DEBUG then
    local originalFn = fn

    fn = function(...)
      local args = {...}

      xpcall(
        function()
          return originalFn(unpack(args))
        end,
        function()
          print(debug.traceback())
        end
      )
    end
  end

  local co = {
    fn = fn,
    co = coroutine.create(fn),
    disposeFn = nil,
    sub = nil,
    args = {...}
  }

  setmetatable(co, {
    __call = function (self, ...)
      return Co.create(co.fn, ...)
    end,
    __index = Co
  })

  return co
end

function Co:reset()
  self.co = coroutine.create(self.fn)
  self.sub = nil
end

function Co:update()
  if not self.co then
    return false
  end

  if self.sub then
    if not self.sub:update() then
      self.sub = nil
    end
  end

  if not self.sub then
    if coroutine.status(self.co) == "dead" then
      self:dispose()
      self.co = nil
      return false
    end

    local status, cont = coroutine.resume(self.co, unpack(self.args))
    if not status then
      print("[CO] " .. cont) -- cont is actually the error msg here
      self:dispose()
      self.co = nil
      return false
    end

    if cont then
      self.sub = cont
    end
  end

  return true
end

function Co:dispose()
  if self.disposeFn then
    self.disposeFn()
  end
end

function Co.yield(ct)
  if type(ct) == "function" then
    ct = Co.create(ct)
  end

  coroutine.yield(ct)
end

function Co.sleep(min, max)
  local seconds = min

  if max ~= nil then
    seconds = seconds + Random.value() * (max - min)
  end

  if seconds == 0.0 then
    Co.yield()
    return
  end

  while seconds > 0.0 do
    seconds = seconds - Time.fixedDeltaTime
    Co.yield()
  end

  return seconds
end

function Co.waitFor(min, max)
  return Co.create(function ()
    Co.sleep(min, max)
  end)
end

function Co.runWhile(predicate, fn)
  local co = fn
  if type(co) == "function" then
    co = Co.create(co)
  end

  return Co.create(function()
    co:reset()

    while predicate() do
      if not co:update() then
        co:reset()
      end

      Co.yield()
    end
  end)
end

function Co.yieldRunWhile(predicate, fn)
  return Co.yield(Co.runWhile(predicate, fn))
end

function Co.concurrent(...)
  local args = {...}
  
  local cos = {}
  for i, fn in ipairs(args) do
    if type(fn) == "function" then
      cos[#cos+1] = Co.create(fn)
    else
      cos[#cos+1] = fn
    end
  end

  return Co.create(function()
    for i, co in ipairs(cos) do
      co:reset()
    end

    local running = true
    while running do
      for i, co in ipairs(cos) do
        if not running then
          co:dispose()
        elseif not co:update() then
          running = false
        end
      end

      Co.yield()
    end
  end)
end

function Co.yieldConcurrent(...)
  return Co.yield(Co.concurrent(...))
end

function Co.series(...)
  local args = {...}

  local cos = {}
  for i, fn in ipairs(args) do
    if type(fn) == "function" then
      cos[#cos+1] = Co.create(fn)
    else
      cos[#cos+1] = fn
    end
  end

  return Co.create(function()
    for i, co in ipairs(cos) do
      co:reset()
    end

    local current = 1

    while current <= #cos do
      if not cos[current]:update() then
        current = current + 1
      end

      Co.yield()
    end
  end)
end

function Co.yieldSeries(...)
  return Co.yield(Co.series(...))
end

function Co.maybe(fn, chance)
  local co = fn
  if type(co) == "function" then
    co = Co.create(co)
  end

  return Co.create(function()
    co:reset()

    if Random.value() < chance then
      while co:update() do
        Co.yield()
      end
    end
  end)
end

function Co.pickOne(...)
  local args = {...}

  local chances = table.remove(args, #args)

  return Co.create(function()
    local pick = args[#args-1]

    local current = 0.0
    for i, chance in ipairs(chances) do
      current = current + chance
      if Random.value() < current then
        pick = args[i]
        break
      end
    end

    pick:reset()
    while pick:update() do
      Co.yield()
    end
  end)
end

function Co.yieldPickOne(...)
  Co.yield(Co.pickOne(...))
end

function Co.throttle(timeStep, fn)
  local co = fn
  if type(co) == "function" then
    co = Co.create(co)
  end

  return Co.create(function()
    local acc = 0.0

    while true do
      acc = acc + Time.fixedDeltaTime
      if acc >= timeStep then
        if not co:update() then
          return
        end

        acc = acc - timeStep
      end

      Co.yield()
    end
  end)
end

function Co.yieldThrottle(timeStep, fn)
  Co.yield(Co.throttle(timeStep, fn))
end

return Co
