EventEmitter = Class()

function EventEmitter:construct()
  self.listeners = {}
end

function EventEmitter:addListener(event, fn)
  if not self.listeners[event] then
    self.listeners[event] = {}
  end

  local listeners = self.listeners[event]
  
  table.insert(listeners, #listeners+1, fn)
  self:emit("newListener", { event = event, fn = fn })
end

function EventEmitter:on(event, fn)
  self:addListener(event, fn)
end

function EventEmitter:once(event, fn)
  local emitter = self
  self:addListener(event, function(eventArgs)
    fn(eventArgs)
    emitter:removeListener(event, fn)
  end)
end

function EventEmitter:removeListener(event, fn)
  if not self.listeners then
    self.listeners = {}
  end

  if not self.listeners[event] then
    return
  end
  
  for index, _fn in ipairs(self.listeners[event]) do
    if fn == _fn then
      table.remove(self.listeners[event], index)
      self:emit("removeListener", { event = event, fn = fn })
      break
    end
  end
end

function EventEmitter:removeAllListeners()
  self.listeners = {}
end

function EventEmitter:listenerCount()
  if not self.listeners then
    self.listeners = {}
  end

  local count = 0
  
  for index, event in ipairs(self.listeners) do
    count = count + #self.listeners[event]
  end
  
  return count
end

function EventEmitter:hasAnyListeners(event)
  if not self.listeners then
    self.listeners = {}
  end
  
  if not event then
    return self:listenerCount() > 0
  end
  
  if not self.listeners[event] then
    return false
  end
  
  return #self.listeners[event] > 0
end

function EventEmitter:emit(event, ...)
  if not self.listeners then
    self.listeners = {}
  end

  if not self.listeners[event] then
    return
  end

  local args = {...}
  
  for index, fn in ipairs(self.listeners[event]) do
    xpcall(function()
      fn(unpack(args))
    end, function(err)
      print("Error: " .. err .. " (stacktrace follows)")
      for i, line in ipairs(lines(debug.traceback())) do
        print(i .. ". " .. line)
      end
    end)
  end
end
