local ffi = require "ffi"

local cos = math.cos
local sin = math.sin
local sqrt = math.sqrt
local atan2 = math.atan2

ffi.cdef[[
  typedef struct Vector { double x, y; } Vector2_t;
]]

Vector = {}
local index = {}
local meta = { __index = index }

function meta.__eq(a, b)
  if not a or not b then
    return false
  end

  return a.x == b.x and a.y == b.y
end

function meta.__lt(a, b)
  if not a or not b then
    return false
  end

  return (a.x*a.x + a.y*a.y) < (b.x*b.x + b.y*b.y)
end

function meta.__le(a,b)
  if not a or not b then
    return false
  end
  
  return (a.x*a.x + a.y*a.y) <= (b.x*b.x + b.y*b.y)
end

-- Common math operators
function meta.__unm(a)
  return Vector(-a.x, -a.y)
end

function meta.__add(a, b)
  return Vector(a.x + b.x, a.y + b.y)
end

function meta.__sub(a, b)
  return Vector(a.x - b.x, a.y - b.y)
end

function meta.__mul(a, b)
  if type(a) == "number" then
    return Vector(a * b.x, a * b.y)
  elseif type(b) == "number" then
    return Vector(a.x * b, a.y * b)
  end

  return Vector(a.x * b.x, a.y * b.y)
end

function meta.__div(a, b)
  if type(a) == "number" then
    return Vector(a / b.x, a / b.y)
  elseif type(b) == "number" then
    return Vector(a.x / b, a.y / b)
  end

  return Vector(a.x / b.x, a.y / b.y)
end

function meta.__len(a)
  return sqrt(a.x * a.x + a.y * a.y)
end

function meta.__tostring(a)
  return "(" .. a.x .. "," .. a.y .. ")"
end

function index.clone(a)
  return Vector(a.x, a.y)
end

function index.unpack(a)
  return a.x, a.y
end

function index.len(a)
  return sqrt(a.x * a.x + a.y * a.y)
end

function index.len2(a)
  return a.x * a.x + a.y * a.y
end

function index.cross(a, b)
  return a.x * b.y - a.y * b.x
end

function index.distance(a, b)
  local x = a.x - b.x
  local y = a.y - b.y
  return sqrt(x*x + y*y)
end

function index.distance2(a, b)
  local x = a.x - b.x
  local y = a.y - b.y
  return x*x + y*y
end

function index.angle(a, b)
  return cos(a:dot(b))
end

function index.dot(a, b)
  return a.x * b.x + a.y * b.y
end

function index.normalized(a)
  local len = #a
  if len > 0 then
    return Vector(a.x / len, a.y / len)
  end
  return Vector(a.x, a.y)
end

function index.normalize(a)
  local len = #a
  if len > 0 then
    a.x, a.y = a.x / len, a.y / len
  end
  return a
end

function index.abs(a)
  return Vector(math.abs(a.x), math.abs(a.y))
end

function index.perpendicular(a)
  return Vector(-a.y, a.x)
end

function index.rotated(a, phi)
  local c, s = cos(phi), sin(phi)
  return Vector(c * a.x - s * a.y, s * a.x + c * a.y)
end

function index.rotate(a, phi)
  local c, s = cos(phi), sin(phi)
  a.x, a.y = c * a.x - s * a.y, s * a.x + c * a.y
  return a
end

function index.project(a, b)
  local s = (a.x*b.x + a.y*b.y) / (b.x*b.x + b.y*b.y)
  return Vector(s * a.x, s * a.y)
end

function index.projected(a, b)
  local s = (a.x*b.x + a.y*b.y) / (b.x*b.x + b.y*b.y)
  a.x, a.y = s * a.x, s * a.y
  return a
end

function index.mirror(a, b)
  local s = 2 * (a.x*b.x + a.y*b.y) / (b.x*b.x + b.y*b.y)
  return Vector(s * b.x - a.x, s * b.y - a.y)
end

function index.mirrored(a, b)
  local s = 2 * (a.x*b.x + a.y*b.y) / (b.x*b.x + b.y*b.y)
  a.x, a.y = s * b.x - a.x, s * b.y - a.y
  return a
end

function index.lerp(a, b, t)
  return Vector(math.lerp(a.x, b.x, t), math.lerp(a.y, b.y, t))
end

function index.zero()
  return Vector(0.0, 0.0)
end

function index.serialize()
  return { x = self.x, y = self.y }
end

Vector = ffi.metatype("Vector2_t", meta)
