local _L = _L

-- WorldObjectBase --

local WorldObjectBase = Class(EventEmitter)

WorldObjectBase.Components = {
  Transform = require "lib/components/transform",
  SpriteRenderer = require "lib/components/spriterenderer",
  PhysicsCollider = require "lib/components/physicscollider",
  ParticleEmitter = require "lib/components/particleemitter",
  PhysicsJoint = require "lib/components/physicsjoint",
  LightSource = require "lib/components/lightsource",
  SkeletonRenderer = require "lib/components/skeletonrenderer",
  SoundEmitter = require "lib/components/soundemitter"
}

function WorldObjectBase:construct(objectId)
  EventEmitter.construct(self)
  self._objectId = objectId
end

function WorldObjectBase:isValid()
  return self._objectId ~= 0
end

function WorldObjectBase:getId()
  return self._objectId
end

function WorldObjectBase:getName()
  return _L._objectGetName(self._objectId)
end

function WorldObjectBase:setName(name)
  return _L._objectSetName(self._objectId, name)
end

function WorldObjectBase:addComponent(componentType, ...)
  local memberName = componentType:firstToLower()

  if rawget(self, memberName) ~= nil then
    return rawget(self, memberName)
  end
  
  local component = nil

  if self.Components[componentType] ~= nil then
    local ptr = _L._objectAddComponent(self._objectId, componentType)
    component = self.Components[componentType](self, ptr)
  else
    component = _instantiateBehavior(self._objectId, componentType, ...)
  end

  if component == nil then
    return nil
  end

  self:emit("attachComponent", component)
  rawset(self, memberName, component)
  return self[memberName]
end

function WorldObjectBase:removeComponent(componentType)
  local memberName = componentType:firstToLower()
  local component = rawget(self, memberName)

  if component == nil then
    return
  end
  
  self:emit("detachComponent", component)

  if self.Components[componentType] ~= nil then
    _L._objectRemoveComponent(self._objectId, componentType)
  else
    _destroyBehavior(self._objectId, componentType)
  end

  rawset(self, memberName, nil)
end

function WorldObjectBase:destroy()
  self:emit("destroy")
  _L._destroyObject(self._objectId)
end

function WorldObjectBase.__tostring(object)
  return "WorldObject (" .. object._objectId .. ") - " .. object:getName()
end

return WorldObjectBase
