WorldUtil = {}

local scriptors = {}
local scriptorObjects = {}

local triggers = {}
local triggersByType = {}

local landmarks = {}
local landmarksByName = {}

local routes = {}
local routesByName = {}

function WorldUtil.spawnAndDespawnScriptors()
  local cameraDist = Camera.getSize():len() * 0.5
  local cameraCenter = Camera.getPosition() + Camera.getSize() * 0.5

  for i, scriptor in ipairs(scriptors) do
    if not scriptor.spawnImmediately then
      local dist = (cameraCenter - scriptor.position):len()

      if scriptorObjects[i] == nil then
        if dist < cameraDist * 1.25 then
          scriptorObjects[i] = WorldUtil.spawnScriptor(scriptor)
        end
      else
        if dist > cameraDist * 1.5 then
          scriptorObjects[i].object:destroy()
          scriptorObjects[i] = nil
        end
      end
    end
  end
end

function WorldUtil.initScriptors()
  local world = World.getCurrent()
  local count = world:getScriptorCount()

  scriptors = {}
  scriptorObjects = {}

  for i=0, count-1 do
    local scriptor = world:getScriptor(i)

    if scriptor.behavior:len() == 0 then
      print("WARNING! Empty scriptor at index " .. tostring(i))
    else
        local deserializeProps = loadstring("return " .. scriptor.properties)
        if not deserializeProps then
          print("ERROR! Failed to parse properties object for scriptor \"" .. scriptor.behavior .. "\" at index " .. tostring(i))
        elseif not scriptor.disabled then
          scriptor.deserializedProps = deserializeProps()

          local props = _behaviorTypes[scriptor.behavior]._properties
          for i, prop in ipairs(props) do
            if not scriptor.deserializedProps[prop.name] then
              scriptor.deserializedProps[prop.name] = prop.value
            end
          end

          local scriptorIndex = #scriptors+1
          scriptors[scriptorIndex] = scriptor

          if scriptor.spawnImmediately then
            if scriptor.spawnChance == 1.0 or scriptor.spawnChance <= Random.value() then
              scriptorObjects[scriptorIndex] = WorldUtil.spawnScriptor(scriptor)
            end
          end
        end
    end
  end
end

function WorldUtil.spawnScriptor(scriptor)
  local object = CreateObject(scriptor.behavior, scriptor.position)
  return object:addComponent(scriptor.behavior, scriptor.deserializedProps)
end

function WorldUtil.initTriggers()
  local world = World.getCurrent()
  local count = world:getTriggerCount()

  triggers = {}

  for i=0, count-1 do
    local trigger = world:getTrigger(i)

    if trigger.type:len() == 0 then
      print("WARNING! Empty trigger at index " .. tostring(i))
    else
      local deserializeProps = loadstring("return " .. trigger.properties)
      if not deserializeProps then
        print("ERROR! Failed to parse properties object for trigger \"" .. trigger.type .. "\" at index " .. tostring(i))
      else
        trigger.deserializedProps = deserializeProps()
        triggers[#triggers+1] = WorldUtil.spawnTrigger(trigger)
      end
    end
  end
end

function WorldUtil.spawnTrigger(trigger)
  local center = (Vector(trigger.bottomLeft) + Vector(trigger.topRight)) * 0.5
  local object = CreateObject("Trigger \"" .. trigger.type .. "\"", center)
  local instance = object:addComponent("Trigger")
  instance:initTrigger(trigger)

  triggersByType[trigger.type] = instance

  return instance
end

function WorldUtil.getTriggerByType(triggerType)
  return triggersByType[triggerType]
end

function WorldUtil.getTriggers()
  return triggers
end

function WorldUtil.initLandmarks()
  local world = World.getCurrent()
  local count = world:getLandmarkCount()

  landmarks = {}

  for i=0, count-1 do
    local landmark = world:getLandmark(i)
    landmark.position = Vector(landmark.position)

    if landmark.name:len() == 0 then
      print("WARNING! Empty landmark at index " .. tostring(i))
    else
      local deserializeProps = loadstring("return " .. landmark.properties)
      if not deserializeProps then
        print("ERROR! Failed to parse properties object for landmark \"" .. landmark.name .. "\" at index " .. tostring(i))
      else
        landmark.deserializedProps = deserializeProps()
        landmarks[#landmarks+1] = landmark
        landmarksByName[landmark.name] = landmark
      end
    end
  end
end

function WorldUtil.getLandmarks()
  return landmarks
end

function WorldUtil.getLandmarkByName(name)
  return landmarksByName[name]
end

function WorldUtil.initRoutes()
  local world = World.getCurrent()
  local count = world:getRouteCount()

  routes = {}

  for i=0, count-1 do
    local route = world:getRoute(i)
    for i, node in ipairs(route.nodes) do
      node = Vector(node.x, node.y)
    end

    routesByName[route.name] = route
  end
end

function WorldUtil.getRouteByName(name)
  return routesByName[name]
end

function WorldUtil.destroyAll()
  for i, scriptor in ipairs(scriptorObjects) do
    scriptor.object:destroy()
  end

  for i, trigger in ipairs(triggers) do
    trigger.object:destroy()
  end

  scriptors = {}
  scriptorObject = {}
  triggers = {}
  landmarks = {}
end

WorldUtil = ReadOnlyClass(WorldUtil, "WorldUtil")
