local AI = {}

function AI.waitForCollision(object)
  local collisionInfo = nil

  object:on("collisionStart", function(_collisionInfo)
    collisionInfo = _collisionInfo
  end)

  while not collisionInfo do
    Co.yield()
  end

  return collisionInfo
end

function AI.waitForAnimation(object, animationName)
  return AI.waitForAnimation2(object, 0, animationName)
end

function AI.waitForAnimation2(object, trackId, animationName)
  return Co.create(function()
    local animationEnded = false

    local listener = function (_trackId, _animationName)
      if _trackId == trackId and _animationName == animationName then
        animationEnded = true
      end
    end

    object:addListener("animationComplete", listener) 

    while not animationEnded do
      Co.yield()
    end

    object:removeListener("animationComplete", listener)
  end)
end

function AI.yieldWaitForAnimation(object, animationName)
  Co.yield(AI.waitForAnimation2(object, 0, animationName))
end

function AI.yieldWaitForAnimation2(object, trackId, animationName)
  Co.yield(AI.waitForAnimation2(object, trackId, animationName))
end

function AI.playFootstepSounds(movable, soundName, footstepSpeed)
  return Co.create(function()
    while true do
      movable.soundEmitter:postEvent(soundName)

      local speed = movable.velocity:len()
      if speed > 0.0 then
        Co.sleep(footstepSpeed / speed)
      else
        Co.yield()
      end
    end
  end)
end

function AI.fadeOut(renderer, fadeSpeed)
  local color = renderer.tintColor

  return Co.create(function()
    local alpha = 1.0
    
    while alpha > 0.0 do
      alpha = alpha - Time.fixedDeltaTime * fadeSpeed
      color.a = math.max(alpha, 0.0)
      renderer.tintColor = color
      Co.yield()
    end
  end)
end

function AI.moveTowards(movable, targetTransform)
  if not targetTransform then
    return
  end

  return Co.create(function ()
    while true do
      local otherPos
      if targetTransform.x and targetTransform.y then
        otherPos = targetTransform
      else
        otherPos = targetTransform.worldPosition
      end

      local myPos = movable.transform.worldPosition
      local dist = Vector.distance(myPos, otherPos)
      if dist <= 1.5 then
        return
      end

      movable.velocity = (otherPos - myPos) / dist
      
      if movable.character then
        movable.character:lookAt(otherPos)
      end
      
      if movable.blocked then
        return
      end

      Co.yield()
    end
  end)
end

function AI.alignForAttack(movable, enemyTransform, displacement)
  displacement = displacement or Vector.zero()

  return Co.create(function()
    local character = movable.character

    while true do
      if character then
        character:lookAt(enemyTransform.worldPosition)
      end

      local enemyPos = enemyTransform.worldPosition + displacement
      local myPos = movable.transform.worldPosition

      local diff = myPos - enemyPos
      diff = diff:abs()

      if diff.x < 4.0 or diff.y < 4.0 then
        return
      end

      local target = myPos:clone()
      if diff.x < diff.y then
        target.x = enemyPos.x
      else
        target.y = enemyPos.y
      end

      movable.velocity = (target - myPos):normalized()

      if movable.blocked then
        return
      end

      Co.yield()
    end
  end)
end

function AI.findPath(from, to, retPath)
  return Co.create(function()
    local waiting = true

    Pathfinding.findPath(from, to, function(path)
      waiting = false

      if path then
        retPath.nodes = path.nodes
        retPath.nodeCount = path.nodeCount
        retPath.found = true
      else
        retPath.found = false
      end
    end)

    while waiting do
      Co.yield()
    end
  end)
end

function AI.yieldFindPath(from, to, retPath)
  Co.yield(AI.findPath(from, to, retPath))
end

function AI.lineOfSight(x0, x1, agentRadius, layerMask)
  local dir = (x1 - x0):normalized()
  local perp = dir:perpendicular()
  local displacement = perp * agentRadius * 0.5

  --Debug.drawLine(x0 + displacement, x1 + displacement, 0xFF00FFFF)
  --Debug.drawLine(x0 - displacement, x1 - displacement, 0xFF00FFFF)

  local lineOfSight = Physics.lineOfSight(x0 + displacement, x1 + displacement, layerMask)
  return lineOfSight and Physics.lineOfSight(x0 - displacement, x1 - displacement, layerMask)
end

function AI.followPath(movable, path)
  return Co.create(function()
    local current = 1

    while current <= #path.nodes do
      local node = Pathfinding.pullNodeAwayFromWalls(path.nodes[current], movable.radius * 1.05)
      local prevNode = nil

      if current > 1 then
        prevNode = path.nodes[current-1]
      else
        prevNode = movable.transform.worldPosition
      end

      local skipNode = current < #path.nodes and AI.lineOfSight(movable.transform.worldPosition, path.nodes[current+1], movable.radius * 1.05) 
      if not skipNode then
        local dir = (node - prevNode)
        local dist = dir:len()

        if dist > 0.0 then
          dir = dir / dist
        end

        while true do
          local myPos = movable.transform.worldPosition
          local t = Vector.dot(dir, myPos - prevNode)

          local targetPos = prevNode + dir * (t + movable.radius * 4.0)
          --Debug.drawCircle(targetPos, 4.0, 0xFF00FFFF, true)
          --Debug.drawCircle(node, 4.0, 0x00FF00FF, true)

          local distance = Vector.distance(myPos, node)
          if distance <= movable.radius * 1.05 or (t / dist) > 1.0 then
            break
          end

          movable.velocity = Vector.lerp(movable.velocity, (targetPos - myPos):normalized(), Time.fixedDeltaTime * 4.0)
          movable.velocity:normalize()
          if movable.character then
            movable.character.lookDir = movable.velocity
            movable.character:setOrientedSkeleton()
          end

          Co.yield()
        end
      end

      current = current + 1
    end
  end)
end

function AI.yieldFollowPath(movable, path)
  Co.yield(AI.followPath(movable, path))
end

function AI.goTo(movable, position)
  return Co.create(function()
    local transform = movable.transform

    local lineOfSight = function()
      return AI.lineOfSight(transform.worldPosition, position, movable.radius * 1.05)
    end

    return Co.runWhile(function ()
      return Vector.distance(transform.worldPosition, position) > movable.radius * 1.05
    end, function ()
      Co.yieldRunWhile(function ()
        return lineOfSight()
      end, AI.moveTowards(movable, position))

      if lineOfSight() then
        return
      end

      local path = {}
      AI.yieldFindPath(transform.worldPosition, position, path)
      if not path.found then
        return
      end

      AI.yieldFollowPath(movable, path)
    end)
  end)
end

function AI.chase(movable, targetTransform, minDistance)
  return Co.create(function()
    minDistance = minDistance or 80.0
    local transform = movable.transform

    local radius = movable.radius

    local lineOfSight = function()
      return AI.lineOfSight(transform.worldPosition, targetTransform.worldPosition, radius * 1.05)
    end

    return Co.runWhile(function ()
      return Vector.distance(transform.worldPosition, targetTransform.worldPosition) > minDistance or not lineOfSight()
    end, function ()
      Co.yieldRunWhile(function ()
        return lineOfSight()
      end, AI.moveTowards(movable, targetTransform))

      if lineOfSight() then
        return
      end

      local path = {}
      local pathTarget = targetTransform.worldPosition
      AI.yieldFindPath(transform.worldPosition, pathTarget, path)
      if not path.found then
        return
      end

      Co.yieldRunWhile(function()
        return Vector.distance(pathTarget, targetTransform.worldPosition) < radius * 1.05 and not lineOfSight()
      end, AI.followPath(movable, path))
    end)
  end)
end

function AI.followPatrolRoute(movable, patrolRoute)
  return Co.create(function()
    local closestNode = 1
    local closestDistance = 1000000.0

    local myPosition = movable.transform.worldPosition
    for i, node in ipairs(patrolRoute.nodes) do
      local dist = Vector.distance(myPosition, node)

      if dist < closestDistance then
        closestDistance = dist
        closestNode = i
      end
    end

    local nextNode = closestNode
    while nextNode <= #patrolRoute.nodes do
      if patrolRoute.nodes[nextNode] then
        Co.yield(AI.goTo(movable, patrolRoute.nodes[nextNode]))
      end

      nextNode = nextNode + 1
    end

    if not patrolRoute.loop then
      while nextNode > 0 do
        if patrolRoute.nodes[nextNode] then
          Co.yield(AI.goTo(movable, patrolRoute.nodes[nextNode]))
        end
        
        nextNode = nextNode - 1
      end
    else
      Co.yield(AI.goTo(movable, patrolRoute.nodes[1]))
    end
  end)
end

return AI
