local InputContext = Class()

function InputContext:construct()
  self.actions = {}
  self.states = {}
  self.ranges = {}

  self.rangeValues = {}
  self.pressed = {}
end

function InputContext:serialize()
  local serialized = {}

  for name, bindings in pairs(self.actions) do
    for i, action in ipairs(bindings) do
      serialized[#serialized+1] = {
        type = "action",
        name = name,
        button = action.button,
        device = action.device
      }
    end
  end

  for name, bindings in pairs(self.states) do
    for i, state in ipairs(bindings) do
      serialized[#serialized+1] = {
        type = "state",
        name = name,
        button = state.button,
        device = state.device
      }
    end
  end

  for name, bindings in pairs(self.ranges) do
    for i, range in ipairs(bindings) do
      if not range.value then
        range.value = Vector.zero()
      end

      local s = {
        type = "range",
        name = name,
        device = range.device
      }

      if range.button then
        s.button = range.button
        s.value = {
          x = range.value.x,
          y = range.value.y
        }
      elseif range.axis then
        s.axis = range.axis
      end

      serialized[#serialized+1] = s
    end
  end

  return serialized
end

function InputContext:deserialize(serialized)
  for i, binding in ipairs(serialized) do
    if binding.type == "action" then
      local bindings = self.actions[binding.name]
      for q, binding2 in ipairs(bindings) do
        if binding.device == binding2.device then
          binding2.button = binding.button
        end
      end
    elseif binding.type == "state" then
      local bindings = self.states[binding.name]
      for q, binding2 in ipairs(bindings) do
        if binding.device == binding2.device then
          binding2.button = binding.button
        end
      end
    elseif binding.type == "range" then
      local bindings = self.ranges[binding.name]

      for q, binding2 in ipairs(bindings) do
        local value = nil
        if binding.button then
          value = Vector(binding.value.x, binding.value.y)
        end

        if binding.device == binding2.device then
          if binding.axis then
            binding2.axis = binding.axis
          elseif binding.button and value == binding2.value then
            binding2.button = binding.button
          end
        end
      end
    end
  end
end

function InputContext:registerAction(name)
  self.actions[name] = {}
end

function InputContext:registerState(name)
  self.states[name] = {}
end

function InputContext:registerRange(name)
  self.ranges[name] = {}
  self.rangeValues[name] = Vector(0.0, 0.0)
end

function InputContext:addActionBinding(name, device, button)
  local bindings = self.actions[name]
  bindings[#bindings+1] = {
    device = device,
    button = button
  }
end

function InputContext:addStateBinding(name, device, button)
  local bindings = self.states[name]
  bindings[#bindings+1] = {
    device = device,
    button = button
  }
end

function InputContext:addRangeBinding(name, device, axis)
  local bindings = self.ranges[name]
  bindings[#bindings+1] = {
    device = device,
    axis = axis
  }
end

function InputContext:addDiscreteRangeBinding(name, device, button, value)
  local bindings = self.ranges[name]
  bindings[#bindings+1] = {
    device = device,
    button = button,
    value = value
  }
end

function InputContext:evaluate(event, callback)
  local device = event.source

  if event.type == "ButtonPressed" then
    if not self:evaluateAction(event, callback) then
      return true
    end

    if not self:evaluateState(event, callback) then
      return true
    end

    if not self:evaluateRange(event, callback) then
      return true
    end
  elseif event.type == "ButtonReleased" then
    if not self:evaluateState(event, callback) then
      return true
    end

    if not self:evaluateRange(event, callback) then
      return true
    end
  elseif event.type == "AnalogValueChanged" then
    if not self:evaluateRange(event, callback) then
      return true
    end
  end

  return false
end

function InputContext:evaluateAction(event, callback)
  local device = event.source

  for name, bindings in pairs(self.actions) do
    for i, binding in ipairs(bindings) do
      if binding.device == device and binding.button == event.button then
        if not callback("action", name) then
          return false
        end
      end
    end
  end

  return true
end

function InputContext:evaluateState(event, callback)
  if event.type == "ButtonPressed" then
    self.pressed[event.button] = true
  elseif event.type == "ButtonReleased" then
    if not self.pressed[event.button] then
      return false
    end

    self.pressed[event.button] = nil
  end

  local device = event.source

  for name, bindings in pairs(self.states) do
    for i, binding in ipairs(bindings) do
      if binding.device == device and binding.button == event.button then
        local value = true
        if event.type == "ButtonReleased" then
          value = false
        end

        if not callback("value", name, value) then
          return false
        end
      end
    end
  end

  return true
end

function InputContext:evaluateRange(event, callback)
  local device = event.source

  for name, bindings in pairs(self.ranges) do
    for i, binding in ipairs(bindings) do
      if binding.device == device and binding.axis and binding.axis == event.axis then
        local value = event.value

        if value:len() > 1.0 then
          value:normalize()
        end

        if not callback("range", name, value) then
          return false
        end
      elseif binding.device == device and binding.button and binding.button == event.button then
        local scale = 1.0
        if event.type == "ButtonReleased" then
          scale = -1.0
        end

        local value = self.rangeValues[name]
        value = value + binding.value * scale
        self.rangeValues[name] = value:clone()

        if value:len() > 1.0 then
          value:normalize()
        end

        if not callback("range", name, value) then
          return false
        end
      end
    end
  end

  return true
end

return InputContext
